﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <vector>
#include <nn/nn_Result.h>
#include <nn/nn_Macro.h>
#include <nn/os/os_SdkMutex.h>
#include <nn/os/os_Tick.h>
#include <nn/xcd/xcd_Input.h>
#include "xcd_Peripheral.h"
#include "xcd_ICommandListener.h"
#include "detail/xcd_DScaleProcessor.h"

namespace nn { namespace xcd {

// !< アナログスティックの計算用の定数値をまとめた構造体
struct AnalogStickStaticValues
{
    bool isScalingRequired;                     //!< X軸の Scaling 処理が必要かどうか
    int16_t originPlay;                         //!< ゼロ点の遊びの値
    int16_t circuitValidRatio;                  //!< 外側不感体の割合 (%)
    AnalogStickDeviceParameter deviceParameter; //!< デバイス毎のモデル情報
};

// !<  有効なアナログスティックを指定する構造体
struct ValidAnalogStick
{
    bool isAnalogStickMainValid;   //!< AnalogStickMain が有効か否か
    bool isAnalogStickSubValid;    //!< AnalogStickSub  が有効か否か
};

//!< ボタン, アナログスティック, 6軸センサーを扱うクラス
class PadInput final : public Peripheral, public ICommandListener
{

private:
    //!< ボタン、アナログスティックの状態
    PadState m_State;

    //!< センサーなどの状態を表す構造体
    struct PadConfig
    {
        AnalogStickValidRange analogStickLValidRange;       //!< 左アナログスティックの可動範囲
        AnalogStickValidRange analogStickRValidRange;       //!< 右アナログスティックの可動範囲
        bool sensorSleep;                                   //!< センサーのスリープ状態
        SensorCalibrationValue sensorCalibrationValue;      //!< 6軸センサーのキャリブレーション値
        AccelerometerFsr accelerometerFsr;                  //!< 加速度センサーのFSR値
        GyroscopeFsr gyroscopeFsr;                          //!< ジャイロセンサーのFSR値
        bool isLeftStickCalibrated;                         //!< アナログスティックがキャリブレーション済みかどうか
        bool isRightStickCalibrated;                        //!< アナログスティックがキャリブレーション済みかどうか
        bool isStickCalReadDone;                            //!< アナログスティックの CAL 値読み出しが完了しているかどうか
        bool isSensorCalReadDone;                           //!< センサーの CAL 値読み出しが完了しているかどうか
        bool isColorReadDone;                               //!< 色情報読み出しが完了しているかどうか
    };

    PadConfig m_Config;             //!< 現在の各種設定値
    PadConfig m_TemporalConfig;     //!< 設定変更中の暫定値 TODO: 変更完了前に複数回変更されると上書きされてしまう

    AnalogStickStaticValues m_StickLeftStaticValues; //!< アナログスティックの計算用の Static な情報
    AnalogStickStaticValues m_StickRightStaticValues; //!< アナログスティックの計算用の Static な情報

    ValidAnalogStick m_ValidAnalogStick; //!< 有効なアナログスティックの情報

    SensorState m_SensorHorizontalOffset; //!< 6軸センサー の Horizontal Offset

    int m_SerialFlashFormatVersion; //!< シリアルフラッシュのフォーマットバージョン

    //!< 6軸センサーの状態
    SixAxisSensorState m_SensorStates[SixAxisSensorSampleCountMax];

    ::nn::os::SdkMutex m_SensorMutex;    //!< センサーを排他するための Mutex
    int m_SensorBufferIndex;             //!< バッファ内の現在のインデックス
    int m_SensorBufferCount;             //!< バッファの中身のデータ数
    int64_t m_LastSampleNumberForBuffer; //!< センサーデータに付与するサンプル番号

    //!< ボタンのトリガ状態を表す構造体
    struct ButtonTrigger
    {
        bool triggered;
        nn::os::Tick time;
    };

    //!< ボタンのトリガー状態に関する情報 L/R/ZL/ZR/SL/SR の6つ分
    ButtonTrigger m_ButtonTriggers[6];

    //!< 左アナログスティックの更新処理状況
    CalibrationUpdateStatus m_LeftAnalogStickValidRangeUpdateStatus;

    //!< 右アナログスティックの更新処理状況
    CalibrationUpdateStatus m_RightAnalogStickValidRangeUpdateStatus;

    //!< 左アナログスティックの更新処理の結果
    CalibrationUpdateStatus m_SensorCalibrationUpdateStatus;

    //!< DScale Processor
    detail::DScaleProcessor m_DScaleProcessor;

public:
    PadInput() NN_NOEXCEPT;
    virtual ~PadInput() NN_NOEXCEPT NN_OVERRIDE;

    //!< デバイスが接続されたときに呼ばれる関数
    virtual void Activate(DeviceType type, FirmwareVersionImpl firmwareVersion) NN_NOEXCEPT NN_OVERRIDE;
    //!< 接続直後の初期化処理が完了したかどうかを確認する関数。初期化が完了したら必ずtrueを返してください
    virtual bool IsActivated() NN_NOEXCEPT NN_OVERRIDE;
    //!< デバイスが切断された時に呼ばれる関数
    virtual void Deactivate() NN_NOEXCEPT NN_OVERRIDE;

    using Peripheral::ParseInputReport;
    void ParseInputReport(const uint8_t* pBuffer, size_t size, int sampleSinceLast, SensorSleepValueType sensor, int validSensorCount) NN_NOEXCEPT;
    virtual size_t GetOutputReport(uint8_t* pOutValue, size_t size) NN_NOEXCEPT NN_OVERRIDE;

    virtual void NotifyUpdateSixAxisSensorUserCal(Result result) NN_NOEXCEPT NN_OVERRIDE;
    virtual void NotifyUpdateLeftAnalogStickUserCal(Result result) NN_NOEXCEPT NN_OVERRIDE;
    virtual void NotifyUpdateRightAnalogStickUserCal(Result result) NN_NOEXCEPT NN_OVERRIDE;
    virtual void NotifyAck(Result result, uint8_t id) NN_NOEXCEPT NN_OVERRIDE;
    virtual void NotifyButtonTriggerElapsedTime(const ButtonTriggerElapsedTime& value) NN_NOEXCEPT NN_OVERRIDE;
    virtual void NotifyCal1(const SensorCalibrationValue& value) NN_NOEXCEPT NN_OVERRIDE;
    virtual void NotifyCal2(const AnalogStickValidRange& leftValue,
                            const AnalogStickValidRange& rightValue,
                            ::nn::util::Color4u8Type& mainColor,
                            ::nn::util::Color4u8Type& subColor) NN_NOEXCEPT NN_OVERRIDE;
    virtual void NotifyModel1(const SensorState& sensorHorizontalOffset,
                              const AnalogStickDeviceParameter& leftStickParam,
                              const uint16_t leftOriginPlay,
                              const uint16_t leftCircuitValidRatio,
                              const bool     leftIsXScalingRequired) NN_NOEXCEPT NN_OVERRIDE;
    virtual void NotifyModel2(const AnalogStickDeviceParameter& rightStickParam,
                              const uint16_t rightOriginPlay,
                              const uint16_t rightCircuitValidRatio,
                              const bool     rightIsXScalingRequired) NN_NOEXCEPT NN_OVERRIDE;
    virtual void NotifyUserCal1(const bool isLeftStickAvailable,
                                const AnalogStickValidRange& leftValue,
                                const bool isRightStickAvailable,
                                const AnalogStickValidRange& rightValue,
                                const bool isSensorAvailable) NN_NOEXCEPT NN_OVERRIDE;
    virtual void NotifyUserCal2(const SensorCalibrationValue& value) NN_NOEXCEPT NN_OVERRIDE;

    //!< 最新のPadの入力を1つ取得する
    Result GetPadState(PadState* pOutValue) NN_NOEXCEPT;

    //!< 最新のセンサーの入力を取得する
    Result GetSensorStates(int* pOutCount, SixAxisSensorState* pOutValue, int length) NN_NOEXCEPT;

    //!< ボタンが押されたタイミングを取得する
    Result GetButtonTriggerElapsedTime(nn::os::Tick* pOutValue, int buttonIndex) NN_NOEXCEPT;

    //!< センサーのスリープ制御
    void SleepSensor(bool sleep) NN_NOEXCEPT;
    void IsSensorSleep(bool* pOutValue) NN_NOEXCEPT;
    //!< センサーの設定変更
    void SetSensorConfig(AccelerometerFsr accelerometerFsr, GyroscopeFsr gyroscopeFsr) NN_NOEXCEPT;
    void GetSensorConfig(AccelerometerFsr* pOutAccelerometerFsr, GyroscopeFsr* pOutGyroscopeFsr) NN_NOEXCEPT;

    //!< センサーのキャリブレーション値を取得する
    void GetSensorCalibrationValue(SensorCalibrationValue* mOutValue) NN_NOEXCEPT;
    //!< センサーのキャリブレーション値を更新する
    Result UpdateSensorCalibrationValue(const SensorCalibrationValue& value) NN_NOEXCEPT;
    //!< センサーのキャリブレーション値をリセットする
    Result ResetSensorCalibrationValue() NN_NOEXCEPT;
    //!< センサーのキャリブレーション値の更新状況を取得する
    CalibrationUpdateStatus GetSensorCalibrationValueUpdateStatus() NN_NOEXCEPT;
    //!< センサーのモデル値を取得する
    SensorState GetSensorHorizontalOffset() NN_NOEXCEPT;

    //!< 左アナログスティックの有効範囲を取得する
    void GetLeftAnalogStickValidRange(AnalogStickValidRange* pOutValue) NN_NOEXCEPT;
    //!< 左アナログスティックの有効範囲を更新する
    Result UpdateLeftAnalogStickValidRange(const AnalogStickValidRange& value) NN_NOEXCEPT;
    //!< 左アナログスティックの有効範囲をリセットする
    Result ResetLeftAnalogStickValidRange() NN_NOEXCEPT;
    //!< 左アナログスティックの有効範囲の更新状況を取得する
    CalibrationUpdateStatus GetLeftAnalogStickValidRangeUpdateStatus() NN_NOEXCEPT;
    //!< 左アナログスティックのモデル値を取得する
    AnalogStickDeviceParameter GetLeftAnalogStickDeviceParameter() NN_NOEXCEPT;

    //!< 右アナログスティックの有効範囲を取得する
    void GetRightAnalogStickValidRange(AnalogStickValidRange* pOutValue) NN_NOEXCEPT;
    //!< 右アナログスティックの有効範囲を更新する
    Result UpdateRightAnalogStickValidRange(const AnalogStickValidRange& value) NN_NOEXCEPT;
    //!< 右アナログスティックの有効範囲をリセットする
    Result ResetRightAnalogStickValidRange() NN_NOEXCEPT;
    //!< 右アナログスティックの有効範囲の更新状況を取得する
    CalibrationUpdateStatus GetRightAnalogStickValidRangeUpdateStatus() NN_NOEXCEPT;
    //!< 右アナログスティックのモデル値を取得する
    AnalogStickDeviceParameter GetRightAnalogStickDeviceParameter() NN_NOEXCEPT;

    //!< SerialFlash のフォーマットバージョンをセットする
    void SetSerialFlashFormatVersion(int version) NN_NOEXCEPT;

private:
    //!< アナログスティックの CAL 値を検証する
    bool ValidateAnalogStickValidRange(const AnalogStickValidRange& value, const AnalogStickStaticValues& staticValues) NN_NOEXCEPT;

    void PrintDeviceInformation() NN_NOEXCEPT;
};

}} // namespace nn::xcd
