﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <algorithm>
#include <mutex>
#include <nn/nn_Macro.h>
#include <nn/nn_Result.h>
#include <nn/nn_StaticAssert.h>
#include <nn/os/os_MessageQueue.h>
#include <nn/os/os_Mutex.h>
#include <nn/os/os_NativeHandle.h>
#include <nn/os/os_SystemEvent.h>
#include <nn/xcd/xcd_Device.h>
#include <nn/xcd/xcd_Tera.h>
#include <nn/xcd/xcd_NfcTypes.h>
#include "xcd_Peripheral.h"
#include "detail/xcd_NfcCommandHandler.h"
#include "detail/xcd_TeraCommon.h"
#include "detail/xcd_TeraNfc.h"
#include "detail/xcd_TimeCounter.h"
#include "detail/xcd_TypedQueue.h"

namespace nn { namespace xcd {

/**
 * @brief   NFC を扱うクラス
 */
class NfcProcessor final : public Peripheral
{
    NN_DISALLOW_COPY(NfcProcessor);
    NN_DISALLOW_MOVE(NfcProcessor);

public:
    NfcProcessor() NN_NOEXCEPT :
        m_Mutex(true),
        m_CommonEvent(),
        m_DetectEvent(),
        m_CurrentNfcStatus(),
        m_LastResultCode(NfcResultCode_Ok),
        m_NfcCommandHandler(),
        m_LastEventType(detail::InternalNfcEventType::None),
        m_LastEventTypeForDetect(detail::InternalNfcEventType::None),
        m_StatusForGetInfo(),
        m_TagData(),
        m_IsEventCreated(false),
        m_HasValidTagData(false)
    {}

    virtual void Activate(DeviceType type, FirmwareVersionImpl firmwareVersion) NN_NOEXCEPT NN_OVERRIDE;
    virtual void Deactivate() NN_NOEXCEPT NN_OVERRIDE;

    virtual void ParseInputReport(const uint8_t* pBuffer, size_t size, uint8_t sampleNumber) NN_NOEXCEPT NN_OVERRIDE;
    virtual size_t GetOutputReport(uint8_t* pOutValue, size_t size) NN_NOEXCEPT NN_OVERRIDE;

    /**
     * @brief   NFC の状態通知イベントを登録
     *
     * @param[out]  pOutCommonHandle    NFC のタグ検出関連以外を通知するイベントのハンドル
     * @param[out]  pOutDetectHandle    NFC のタグ検出関連を通知するイベントのハンドル
     *
     * @pre
     *  - @a pOutCommonHandle != nullptr
     *  - @a pOutDetectHandle != nullptr
     */
    void SetEvent(nn::os::NativeHandle* pOutCommonHandle, nn::os::NativeHandle* pOutDetectHandle) NN_NOEXCEPT;

    /**
     * @brief   NFC イベントに対応する情報の取得
     *
     * @param[in]   pOutInfo            NFC のタグ検出関連以外を通知するイベント
     *
     * @pre
     *  - pOutInfo != nullptr
     */
    nn::Result GetNfcInfo(NfcInfo* pOutInfo) NN_NOEXCEPT;

    /**
     * @brief   タグの検出開始
     *
     * @param[in]   nfcDiscoveryParam   NFC_START_DISCOVERY のパラメータ
     */
    nn::Result StartDiscovery(const NfcDiscoveryParameter& nfcDiscoveryParam) NN_NOEXCEPT;

    /**
     * @brief   タグの検出停止
     */
    nn::Result StopDiscovery() NN_NOEXCEPT;

    /**
     * @brief   タグの読み取り開始
     *
     * @param[in]   ntagReadParam       NFC_READ_START のパラメータ
     */
    nn::Result StartNtagRead(const NtagReadParameter& ntagReadParam) NN_NOEXCEPT;

    /**
     * @brief   タグへの書き込み開始
     *
     * @param[in]   ntagWriteParam      NFC_WRITE_START のパラメータ
     */
    nn::Result StartNtagWrite(const NtagWriteParameter& ntagWriteParam) NN_NOEXCEPT;

    /**
     * @brief   パススルーコマンドの送信
     *
     * @param[in]   passThruParam       NFC_PASS_THRU_REQ のパラメータ
     */
    nn::Result SendRawData(const NfcPassThruParameter& passThruParam) NN_NOEXCEPT;

    /**
     * @brief   MIFARE 鍵の登録
     *
     * @param[in]   keyWriteParameter   鍵の書き込みパラメータ
     */
    nn::Result RegisterMifareKey(const MifareKeyWriteParameter& keyWriteParameter) NN_NOEXCEPT;

    /**
     * @brief   MIFARE 鍵の消去
     *
     * @param[in]   keyClearParameter   鍵の消去パラメータ
     */
    nn::Result ClearMifareKey(const MifareKeyClearParameter& keyClearParameter) NN_NOEXCEPT;

    /**
     * @brief   MIFARE タグの読み取り開始
     *
     * @param[in]   readParameter   読み取りパラメータ
     */
    nn::Result StartMifareRead(const MifareReadParameter& readParameter) NN_NOEXCEPT;

    /**
     * @brief   MIFARE タグへの書き込み開始
     *
     * @param[in]   writeParameter  書き込みパラメータ
     */
    nn::Result StartMifareWrite(const MifareWriteParameter& writeParameter) NN_NOEXCEPT;

private:
    mutable nn::os::Mutex   m_Mutex;            //!< 状態の変更・取得を排他するための Mutex

    // 上位レイヤーへの通知用イベント
    nn::os::SystemEventType          m_CommonEvent;             //!< NFC に関する状態変化を通知する共通イベント
    nn::os::SystemEventType          m_DetectEvent;             //!< タグの検知/喪失を通知するイベント

    // NFC 機能の制御に使用する内部状態
    detail::NfcPackedStateInfPayload m_CurrentNfcStatus;        //!< 現在の NFC の状態
    NfcResultCode                    m_LastResultCode;          //!< 前回通信時のリザルトコード
    detail::NfcCommandHandler        m_NfcCommandHandler;       //!< コマンドの発行制御を行うオブジェクト

    // GetNfcInfo 関連
    detail::InternalNfcEventType     m_LastEventType;           //!< MCU からの応答の種類 (GetNfcInfo で取得する情報の種類を特定するために使用)
    detail::InternalNfcEventType     m_LastEventTypeForDetect;  //!< MCU からの応答の種類 (タグ検出・喪失時に GetNfcInfo で取得する情報の種類を特定するために使用)
    detail::StatusForGetInfo         m_StatusForGetInfo;        //!< GetNfcInfo で取得するための情報
    detail::ResponseTagData          m_TagData;                 //!< 検出したタグに対するコマンドの応答データ

    // 各種フラグ
    bool                             m_IsEventCreated;          //!< システムイベントを作成済みか
    bool                             m_HasValidTagData;         //!< 検出したタグの情報が有効か

private:
    /**
     * @brief   内部状態のクリア
     */
    void ClearInternalState() NN_NOEXCEPT;

    /**
     * @brief   NFC イベントのシグナル化
     *
     * @param[in]   pEvent  シグナル化するイベント
     *
     * @pre
     *  - pEvent != nullptr
     */
    void SignalEvent(nn::os::SystemEventType* pEvent) NN_NOEXCEPT;

    /**
     * @brief   入力データに応じた解析処理の切り替え
     *
     * @param[in]   header      パケットのヘッダ
     * @param[in]   pPayload    パケットのペイロード
     */
    void DispatchInputParser(const detail::NfcPackedPacketHeader& header, const uint8_t* pPayload) NN_NOEXCEPT;

    /**
     * @brief   NFC_STATE_INF コマンドの解析
     *
     * @param[in]   header      パケットのヘッダ
     * @param[in]   pPayload    パケットのペイロード
     */
    void ParseStateInf(const detail::NfcPackedPacketHeader& header, const uint8_t* pPayload) NN_NOEXCEPT;

    /**
     * @brief   ステート遷移判定
     *
     * @param[in]   expectState     遷移先として期待するステート
     * @param[in]   previousState   以前のステート
     *
     * @return  遷移成否
     * @retval  true    ステート遷移した
     * @retval  false   ステート遷移していない
     */
    bool IsStateChanged(uint8_t expectState, uint8_t previousState) const NN_NOEXCEPT;

    /**
     * @brief   タグ検出判定
     *
     * @param[in]   previousState   以前のステート (タグ検出判定に使用)
     *
     * @details
     *  ステートが Detected or Deactivated に遷移することをもって、タグの検出/喪失を判定する。
     */
    void CheckTagDetection(uint8_t previousState) NN_NOEXCEPT;

    /**
     * @brief   タグ喪失時のイベント解除処理
     *
     * @details
     *  タグ喪失に伴って意味を成さなくなるイベントを解除する。
     */
    void CancelEventOnTagLost() NN_NOEXCEPT;

    /**
     * @brief   イベント発生判定
     *
     * @param[in]   common  MCU からの応答データ
     *
     * @details
     *  MCU からの応答データをもとにイベントの判定を行う。イベント発生時には、登録されたイベントをシグナル化する。
     */
    void CheckEvent(const detail::NfcPackedCommonResponseHeader& common) NN_NOEXCEPT;

    /**
     * @brief   受信したパケットをタグデータとして処理
     *
     * @param[out]  pOutIsCompleted     最終パケットを受信したか
     * @param[in]   header              パケットのヘッダ
     * @param[in]   pPayload            パケットのペイロード
     *
     * @pre
     *  - @a pOutIsCompleted != nullptr
     *  - @a pPayload != nullptr
     *
     * @retval true     データを受理した
     * @retval false    データを拒否した (パケットが異常、または既に最終パケット受信済み)
     */
    bool ReceiveResponseAsTagData(
        bool* pOutIsCompleted,
        const detail::NfcPackedPacketHeader& header,
        const uint8_t* pPayload) NN_NOEXCEPT;

    /**
     * @brief   NFC_DATA_RSP コマンドの受信処理
     *
     * @param[in]   header      パケットのヘッダ
     * @param[in]   pPayload    パケットのペイロード
     *
     * @pre
     *  - @a pPayload != nullptr
     */
    void ReceiveReadResponse(const detail::NfcPackedPacketHeader& header, const uint8_t* pPayload) NN_NOEXCEPT;

    /**
     * @brief   NFC_PASS_THRU_RSP コマンドの受信処理
     *
     * @param[in]   header      パケットのヘッダ
     * @param[in]   pPayload    パケットのペイロード
     *
     * @pre
     *  - @a pPayload != nullptr
     */
    void ReceivePassThruResponse(const detail::NfcPackedPacketHeader& header, const uint8_t* pPayload) NN_NOEXCEPT;

    /**
     * @brief   NFC_MIFARE_RSP コマンドの受信処理
     *
     * @param[in]   header      パケットのヘッダ
     * @param[in]   pPayload    パケットのペイロード
     *
     * @pre
     *  - @a pPayload != nullptr
     */
    void ReceiveMifareResponse(const detail::NfcPackedPacketHeader& header, const uint8_t* pPayload) NN_NOEXCEPT;

    /**
     * @brief   エラーの発生確認を行う
     *
     * @param[in]   pPayload        パケットのペイロード
     * @param[in]   payloadLength   パケットのペイロード長
     *
     * @retval true     エラーあり
     * @retval false    エラーなし
     */
    bool CheckErrorEventOccurred(const uint8_t* pPayload, size_t payloadLength) NN_NOEXCEPT;

    /**
     * @brief   タグ検出イベントに対する NfcInfo を取得
     */
    nn::Result GetNfcInfoForDetectEvent(size_t* pOutSize, NfcInfo* pOutInfo) NN_NOEXCEPT;

    /**
     * @brief   タグ検出以外のイベントに対する NfcInfo を取得
     */
    nn::Result GetNfcInfoForGeneralEvent(size_t* pOutSize, NfcInfo* pOutInfo) NN_NOEXCEPT;
};

}} // namespace nn::xcd
