﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <algorithm>
#include <mutex>
#include <nn/nn_Macro.h>
#include <nn/nn_Result.h>
#include <nn/nn_SdkAssert.h>
#include <nn/os.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/xcd/xcd_Result.h>
#include "xcd_NfcProcessor.h"
#include "detail/xcd_TeraCommon.h"
#include "detail/xcd_TeraNfc.h"
#include "detail/xcd_NfcParser.h"

//#define VERBOSE

/*
 * 引数に指定された文字列を、NfcProcessor のログとして出力します。
 */
#ifdef NN_BUILD_CONFIG_COMPILER_VC
#define NN_XCD_NFC_LOG(...)             NN_DETAIL_XCD_INFO("[NFC] " ##__VA_ARGS__)
#else
#define NN_XCD_NFC_LOG(format, ...)     NN_DETAIL_XCD_INFO("[NFC] " format, ##__VA_ARGS__)
#endif

/*
 * VERBOSE モード時のみ、引数に指定された文字列を、NfcProcessor のログとして出力します。
 */
#ifdef VERBOSE
#define NN_XCD_NFC_LOG_VERBOSE(...)     NN_XCD_NFC_LOG(__VA_ARGS__)
#else
#define NN_XCD_NFC_LOG_VERBOSE(...)     static_cast<void>(0)
#endif

namespace nn { namespace xcd {

void NfcProcessor::Activate(DeviceType type, FirmwareVersionImpl firmwareVersion) NN_NOEXCEPT
{
    std::lock_guard<decltype(m_Mutex)> lock(m_Mutex);

    if (m_Activated)
    {
        return;
    }

    m_Activated = true;
    m_Type      = type;
    m_FirmwareVersion = firmwareVersion;

    ClearInternalState();
}

void NfcProcessor::Deactivate() NN_NOEXCEPT
{
    std::lock_guard<decltype(m_Mutex)> lock(m_Mutex);

    if (!m_Activated)
    {
        return;
    }

    ClearInternalState();

    if (m_IsEventCreated)
    {
        nn::os::DestroySystemEvent(&m_CommonEvent);
        nn::os::DestroySystemEvent(&m_DetectEvent);
        m_IsEventCreated = false;
    }

    m_Type      = DeviceType_Unknown;
    m_Activated = false;
}

void NfcProcessor::ClearInternalState() NN_NOEXCEPT
{
    if (m_IsEventCreated)
    {
        nn::os::ClearSystemEvent(&m_CommonEvent);
        nn::os::ClearSystemEvent(&m_DetectEvent);
    }

    m_CurrentNfcStatus.Initialize();
    m_NfcCommandHandler.Clear();
    m_LastResultCode         = NfcResultCode_Ok;
    m_LastEventType          = detail::InternalNfcEventType::None;
    m_LastEventTypeForDetect = detail::InternalNfcEventType::None;
    std::memset(&m_StatusForGetInfo, 0, sizeof(m_StatusForGetInfo));
    m_TagData.Clear();
    m_HasValidTagData = false;
}

void NfcProcessor::SignalEvent(nn::os::SystemEventType* pEvent) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pEvent);

    std::lock_guard<decltype(m_Mutex)> lock(m_Mutex);

    m_StatusForGetInfo.stateInfo = m_CurrentNfcStatus;
    if (m_IsEventCreated)
    {
        nn::os::SignalSystemEvent(pEvent);
    }
}

void NfcProcessor::ParseInputReport(const uint8_t* pBuffer, size_t size, uint8_t sampleNumber) NN_NOEXCEPT
{
    NN_UNUSED(sampleNumber);
    NN_SDK_REQUIRES_GREATER_EQUAL(size, detail::NfcCommandHandler::GetTransferSize().receive);

    std::lock_guard<decltype(m_Mutex)> lock(m_Mutex);

    // 先頭バイトの MisoProtocol は使用しないので無視する
    auto* pNfcData    = &pBuffer[1];
    auto  nfcDataSize = size - 1;

    // 無効なパケットは捨てる
    if (!detail::NfcParser::IsValidReceivedPacket(pNfcData, nfcDataSize))
    {
        NN_XCD_NFC_LOG_VERBOSE("Invalid packet\n");
        return;
    }

    // エラー時はエラー応答を優先する
    if (CheckErrorEventOccurred(pNfcData, nfcDataSize))
    {
        return;
    }

    const auto& PacketHeader = *reinterpret_cast<const detail::NfcPackedPacketHeader*>(pNfcData);
    const auto& CommonHeader = *reinterpret_cast<const detail::NfcPackedCommonResponseHeader*>(
        pNfcData + sizeof(PacketHeader));

    // 送受信成功判定
    if (!m_NfcCommandHandler.CheckPacketTransferSuccess(PacketHeader, CommonHeader))
    {
        return;
    }

    DispatchInputParser(PacketHeader, pNfcData + sizeof(PacketHeader));
}

void NfcProcessor::DispatchInputParser(const detail::NfcPackedPacketHeader& header, const uint8_t* pPayload) NN_NOEXCEPT
{
    switch (header.command)
    {
    case detail::NfcCommand_StateInf:
        {
            ParseStateInf(header, pPayload);
        }
        break;
    case detail::NfcCommand_ReadDataResponse:
        {
            ReceiveReadResponse(header, pPayload);
        }
        break;
    case detail::NfcCommand_PassThruResponse:
        {
            ReceivePassThruResponse(header, pPayload);
        }
        break;
    case detail::NfcCommand_MifareResponse:
        {
            ReceiveMifareResponse(header, pPayload);
        }
        break;
    default:
        return;
    }
}

bool NfcProcessor::CheckErrorEventOccurred(const uint8_t* pPayload, size_t payloadLength) NN_NOEXCEPT
{
    NN_UNUSED(payloadLength);

    const auto& PacketHeader = *reinterpret_cast<const detail::NfcPackedPacketHeader*>(pPayload);
    auto resultCode = static_cast<NfcResultCode>(PacketHeader.resultCode);

    // リザルトコードが OK 以外に変化した場合はエラー発生と見なす

    if (resultCode == m_LastResultCode)
    {
        return false;
    }

    m_LastResultCode = resultCode;

    if (resultCode == NfcResultCode_Ok)
    {
        return false;
    }

#ifdef VERBOSE
    const auto& CommonResponse = *reinterpret_cast<const detail::NfcPackedCommonResponseHeader*>(
        pPayload + sizeof(detail::NfcPackedPacketHeader));
    NN_XCD_NFC_LOG("\n  State: %d (%s), Result code: %d (%s), Shim API: %d, Shim error: %d\n",
        CommonResponse.state,
        detail::GetNfcStateString(static_cast<detail::NfcState>(CommonResponse.state)),
        resultCode,
        detail::GetNfcResultString(static_cast<NfcResultCode>(resultCode)),
        CommonResponse.shimApi,
        CommonResponse.shimError);
#endif

    // エラー時は積まれていた未発行コマンドをキャンセルする
    m_NfcCommandHandler.CancelAllCommand();

    m_LastEventType               = detail::InternalNfcEventType::Error;
    m_StatusForGetInfo.resultCode = resultCode;
    SignalEvent(&m_CommonEvent);

    return true;
}

void NfcProcessor::ParseStateInf(const detail::NfcPackedPacketHeader& header, const uint8_t* pPayload) NN_NOEXCEPT
{
    NN_UNUSED(header);

    std::lock_guard<decltype(m_Mutex)> lock(m_Mutex);

    const auto& stateInf = *reinterpret_cast<const detail::NfcPackedStateInfPayload*>(pPayload);

    // 状態変化の検知
    bool isStateChanged =
        stateInf.common.state != m_CurrentNfcStatus.common.state ||
        stateInf.detectedTagNumber != m_CurrentNfcStatus.detectedTagNumber;

    if (isStateChanged)
    {
        NN_XCD_NFC_LOG_VERBOSE("\n  state: %d (%s), tags: %d\n",
            stateInf.common.state,
            detail::GetNfcStateString(static_cast<detail::NfcState>(stateInf.common.state)),
            stateInf.detectedTagNumber);

        auto previousStatus = m_CurrentNfcStatus.common;
        m_CurrentNfcStatus = stateInf;
        CheckTagDetection(previousStatus.state);
        CheckEvent(previousStatus);
    }
    else
    {
        m_CurrentNfcStatus = stateInf;
    }

    // 受信待ちが完了したらイベントを起こす
    if (m_NfcCommandHandler.CheckReceiveFinished(
            &m_LastEventType,
            static_cast<detail::NfcState>(m_CurrentNfcStatus.common.state)))
    {
        SignalEvent(&m_CommonEvent);
    }
}

bool NfcProcessor::IsStateChanged(uint8_t expectState, uint8_t previousState) const NN_NOEXCEPT
{
    std::lock_guard<decltype(m_Mutex)> lock(m_Mutex);

    return previousState != expectState &&
        m_CurrentNfcStatus.common.state == expectState;
}

void NfcProcessor::CheckTagDetection(uint8_t previousState) NN_NOEXCEPT
{
    std::lock_guard<decltype(m_Mutex)> lock(m_Mutex);

    if (IsStateChanged(detail::NfcState_Detected, previousState))
    {
        m_LastEventTypeForDetect = detail::InternalNfcEventType::TagDetect;
        SignalEvent(&m_DetectEvent);
    }
    else if (IsStateChanged(detail::NfcState_Deactivated, previousState))
    {
        m_LastEventTypeForDetect = detail::InternalNfcEventType::TagLost;
        CancelEventOnTagLost();
        SignalEvent(&m_DetectEvent);
    }
}

void NfcProcessor::CancelEventOnTagLost() NN_NOEXCEPT
{
    if (m_LastEventType == detail::InternalNfcEventType::Error)
    {
        switch (m_StatusForGetInfo.resultCode)
        {
        case NfcResultCode_FunctionError:   // HW 異常
        case NfcResultCode_ResetRequired:   // NFC スタック異常
            // HW, SW スタックの異常によるエラーはキャンセルしない
            break;
        default:
            {
                // 他のエラーはキャンセル
                m_LastEventType = detail::InternalNfcEventType::None;
            }
            break;
        }
    }
    else
    {
        m_LastEventType = detail::InternalNfcEventType::None;
    }

    if (m_IsEventCreated)
    {
        nn::os::ClearSystemEvent(&m_CommonEvent);
    }
}

void NfcProcessor::CheckEvent(const detail::NfcPackedCommonResponseHeader& common) NN_NOEXCEPT
{
    std::lock_guard<decltype(m_Mutex)> lock(m_Mutex);

    if (IsStateChanged(detail::NfcState_WriteFinish, common.state))
    {
        m_LastEventType = detail::InternalNfcEventType::DataWritten;
        SignalEvent(&m_CommonEvent);
    }
    else if (IsStateChanged(detail::NfcState_MifareKeyWriteFinish, common.state))
    {
        m_LastEventType = detail::InternalNfcEventType::MifareKeyWritten;
        SignalEvent(&m_CommonEvent);
    }
}

bool NfcProcessor::ReceiveResponseAsTagData(
    bool* pOutIsComplete,
    const detail::NfcPackedPacketHeader& header,
    const uint8_t* pPayload) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutIsComplete);
    NN_SDK_REQUIRES_NOT_NULL(pPayload);

    *pOutIsComplete = false;

    auto payloadLength = header.GetPayloadLength();

    // ペイロード長 0 は異常パケットのため無視
    if (payloadLength == 0)
    {
        return false;
    }

    // 既に読み込みが完了している場合は受理しない
    if (m_HasValidTagData)
    {
        return false;
    }

#if 0
    NN_SDK_LOG("[xcd] Payload size: %d\n", payloadLength);
    NN_SDK_LOG("[xcd] Packet data:\n");
    detail::DumpBinaryFormatted(reinterpret_cast<const char*>(pPayload), payloadLength);
    NN_SDK_LOG("\n");
#endif

    // バッファ長を超えるデータは受け入れない
    auto acceptableLength = std::min<size_t>(
        payloadLength,
        sizeof(m_TagData.rawData) - m_TagData.size);
    if (acceptableLength < payloadLength)
    {
        NN_XCD_NFC_LOG_VERBOSE(
            "Received %zu bytes, but acceptable only %zu bytes\n",
            payloadLength,
            acceptableLength);
    }

    if (acceptableLength > 0)
    {
        std::memcpy(
            &m_TagData.rawData[m_TagData.size],
            pPayload,
            acceptableLength);
        m_TagData.size += acceptableLength;
    }

    // 終端パケットまたはバッファ上限に到達していたら受信終了
    if (header.IsEndOfCommand() || m_TagData.IsFull())
    {
        m_HasValidTagData = true;
        *pOutIsComplete   = true;
    }

    return true;
}

void NfcProcessor::ReceiveReadResponse(const detail::NfcPackedPacketHeader& header, const uint8_t* pPayload) NN_NOEXCEPT
{
    std::lock_guard<decltype(m_Mutex)> lock(m_Mutex);

    bool isCompleted;
    if (!ReceiveResponseAsTagData(&isCompleted, header, pPayload))
    {
        return;
    }

    if (isCompleted)
    {
        // 読み込み完了を待機する
        m_NfcCommandHandler.StartWaitingReceive(detail::NfcState_ReadFinish);
    }
}

void NfcProcessor::ReceivePassThruResponse(const detail::NfcPackedPacketHeader& header, const uint8_t* pPayload) NN_NOEXCEPT
{
    std::lock_guard<decltype(m_Mutex)> lock(m_Mutex);

    bool isCompleted;
    if (!ReceiveResponseAsTagData(&isCompleted, header, pPayload))
    {
        return;
    }

    if (isCompleted)
    {
        // PassThru コマンドの完了を待機する
        m_NfcCommandHandler.StartWaitingReceive(detail::NfcState_PassThruFinish);
    }
}

void NfcProcessor::ReceiveMifareResponse(const detail::NfcPackedPacketHeader& header, const uint8_t* pPayload) NN_NOEXCEPT
{
    std::lock_guard<decltype(m_Mutex)> lock(m_Mutex);

    bool isCompleted;
    if (!ReceiveResponseAsTagData(&isCompleted, header, pPayload))
    {
        return;
    }

    if (isCompleted)
    {
        // MIFARE コマンドの完了を待機する
        m_NfcCommandHandler.StartWaitingReceive(detail::NfcState_MifareFinish);
    }
}

size_t NfcProcessor::GetOutputReport(uint8_t* pOutValue, size_t size) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutValue);

    const auto& TransferSize = detail::NfcCommandHandler::GetTransferSize();

    NN_SDK_REQUIRES_GREATER_EQUAL(size, TransferSize.send);
    NN_UNUSED(size);

    std::lock_guard<decltype(m_Mutex)> lockStatus(m_Mutex);

    bool needsClearTagData;
    m_NfcCommandHandler.PrepareNextCommand(
        &needsClearTagData,
        static_cast<detail::NfcState>(m_CurrentNfcStatus.common.state));
    if (needsClearTagData)
    {
        // 以前受信したタグのデータは無効にする
        m_HasValidTagData = false;
        m_TagData.Clear();
    }

    // 送信可能なサイズ分ずつパケット化
    size_t outSize;
    m_NfcCommandHandler.CreateNextCommandPacket(&outSize, pOutValue, TransferSize.send);

    if (outSize < TransferSize.send)
    {
        std::memset(&pOutValue[outSize], 0, TransferSize.send - outSize);
    }

    return TransferSize.send;
}

void NfcProcessor::SetEvent(nn::os::NativeHandle* pOutCommonHandle, nn::os::NativeHandle* pOutDetectHandle) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutCommonHandle);
    NN_SDK_REQUIRES_NOT_NULL(pOutDetectHandle);

    std::lock_guard<decltype(m_Mutex)> lock(m_Mutex);

    // イベントが未初期化なら作成
    if (!m_IsEventCreated)
    {
        NN_ABORT_UNLESS_RESULT_SUCCESS(
            nn::os::CreateSystemEvent(&m_CommonEvent, nn::os::EventClearMode_ManualClear, true)
        );
        NN_ABORT_UNLESS_RESULT_SUCCESS(
            nn::os::CreateSystemEvent(&m_DetectEvent, nn::os::EventClearMode_ManualClear, true)
        );
        m_IsEventCreated = true;
    }

    // 呼び出し元には ReadableHandle (Wait と Clear のみ可能) を渡す
    *pOutCommonHandle = nn::os::GetReadableHandleOfSystemEvent(&m_CommonEvent);
    *pOutDetectHandle = nn::os::GetReadableHandleOfSystemEvent(&m_DetectEvent);
}

nn::Result NfcProcessor::GetNfcInfo(NfcInfo* pOutInfo) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutInfo);

    std::lock_guard<decltype(m_Mutex)> lock(m_Mutex);

    nn::Result result;
    size_t infoSize = 0;
    if (m_LastEventTypeForDetect != detail::InternalNfcEventType::None)
    {
        result = GetNfcInfoForDetectEvent(&infoSize, pOutInfo);
        m_LastEventTypeForDetect = detail::InternalNfcEventType::None;
    }
    else
    {
        result = GetNfcInfoForGeneralEvent(&infoSize, pOutInfo);
        m_LastEventType = detail::InternalNfcEventType::None;
    }

    // 残りの予約領域は all 0
    if (infoSize < NfcReceiveDataSizeMax)
    {
        std::memset(
            &pOutInfo->_dummy[infoSize],
            0,
            NfcReceiveDataSizeMax - infoSize);
    }

    return result;
}

nn::Result NfcProcessor::GetNfcInfoForDetectEvent(size_t* pOutSize, NfcInfo* pOutInfo) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutSize);
    NN_SDK_REQUIRES_NOT_NULL(pOutInfo);

    auto parser = detail::NfcParser::GetParserForDetectEvent(m_LastEventTypeForDetect);
    NN_ABORT_UNLESS_NOT_NULL(parser);

    return parser(pOutSize, pOutInfo, m_StatusForGetInfo);
}

nn::Result NfcProcessor::GetNfcInfoForGeneralEvent(size_t* pOutSize, NfcInfo* pOutInfo) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutSize);
    NN_SDK_REQUIRES_NOT_NULL(pOutInfo);

    auto parser = detail::NfcParser::GetParserForGeneralEvent(m_LastEventType);
    NN_ABORT_UNLESS_NOT_NULL(parser);

    return parser(pOutSize, pOutInfo, m_StatusForGetInfo, m_TagData);
}

nn::Result NfcProcessor::StartDiscovery(const NfcDiscoveryParameter& discoveryParameter) NN_NOEXCEPT
{
    return m_NfcCommandHandler.StartDiscovery(discoveryParameter);
}

nn::Result NfcProcessor::StopDiscovery() NN_NOEXCEPT
{
    return m_NfcCommandHandler.StopDiscovery();
}

nn::Result NfcProcessor::StartNtagRead(const NtagReadParameter& readParameter) NN_NOEXCEPT
{
    return m_NfcCommandHandler.StartNtagRead(readParameter);
}

nn::Result NfcProcessor::StartNtagWrite(const NtagWriteParameter& writeParameter) NN_NOEXCEPT
{
    return m_NfcCommandHandler.StartNtagWrite(writeParameter);
}

nn::Result NfcProcessor::SendRawData(const NfcPassThruParameter& passThruParameter) NN_NOEXCEPT
{
    return m_NfcCommandHandler.SendRawData(passThruParameter);
}

nn::Result NfcProcessor::RegisterMifareKey(const MifareKeyWriteParameter& keyWriteParameter) NN_NOEXCEPT
{
    return m_NfcCommandHandler.RegisterMifareKey(keyWriteParameter);
}

nn::Result NfcProcessor::ClearMifareKey(const MifareKeyClearParameter& keyClearParameter) NN_NOEXCEPT
{
    return m_NfcCommandHandler.ClearMifareKey(keyClearParameter);
}

nn::Result NfcProcessor::StartMifareRead(const MifareReadParameter& readParameter) NN_NOEXCEPT
{
    return m_NfcCommandHandler.StartMifareRead(readParameter);
}

nn::Result NfcProcessor::StartMifareWrite(const MifareWriteParameter& writeParameter) NN_NOEXCEPT
{
    return m_NfcCommandHandler.StartMifareWrite(writeParameter);
}

}} // namespace nn::xcd
