﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Macro.h>
#include <nn/nn_SdkAssert.h>
#include <nn/os/os_Event.h>
#include <nn/os/os_MultipleWait.h>
#include <nn/os/os_TimerEvent.h>
#include <nn/os/os_SystemEvent.h>

#include "xcd_ActivationCount.h"

namespace nn { namespace xcd {

//!< 多重待ちイベントを扱うクラスの基底クラスです。
class IMultiWaitBase
{
    NN_DISALLOW_COPY(IMultiWaitBase);
    NN_DISALLOW_MOVE(IMultiWaitBase);

protected:
    //!< 多重待ちオブジェクトホルダー
    ::nn::os::MultiWaitHolderType m_MultiWaitHolder;

    //!< アクティベートされた回数
    ActivationCount m_ActivationCount;

public:
    IMultiWaitBase() NN_NOEXCEPT { /* 何もしない */ };
    virtual ~IMultiWaitBase() NN_NOEXCEPT { /* 何もしない */ };

    //!< 多重待ちオブジェクトにイベントオブジェクトをリンクします。
    virtual void Link(::nn::os::MultiWaitType* pMultiWait) NN_NOEXCEPT = 0;

    //!< イベントオブジェクトを全ての多重待ちオブジェクトからアンリンクします。
    virtual void Unlink() NN_NOEXCEPT = 0;

    //!< 待ちの識別子を返します。
    const ::nn::os::MultiWaitHolderType* GetWaitId() const NN_NOEXCEPT
    {
        return &m_MultiWaitHolder;
    }

    //!< 多重待ちホルダーに受け渡し用のオブジェクトを登録
    void SetUserData(uintptr_t userData)
    {
        // 通知先のオブジェクトを登録
        nn::os::SetMultiWaitHolderUserData(&m_MultiWaitHolder, userData);
    }

    //!< イベントをクリアします。
    virtual void Clear() NN_NOEXCEPT = 0;

    //!< イベントが初期化されているか確認します。
    bool IsInitialized() const NN_NOEXCEPT
    {
        return !m_ActivationCount.IsZero();
    }
};

//!< 多重待ちイベントを扱うクラスの基底クラスです。
class MultiWaitEvent final : public IMultiWaitBase
{
private:
    ::nn::os::EventType m_Event;

public:
    virtual ~MultiWaitEvent() NN_NOEXCEPT NN_OVERRIDE
    {
    }

    //!< 多重待ちイベントをシグナル状態にセットします。
    void Signal() NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(!m_ActivationCount.IsZero());
        ::nn::os::SignalEvent(&m_Event);
    }

    //!< イベントオブジェクトを取得します
    ::nn::os::EventType* GetBase() NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(!m_ActivationCount.IsZero());
        return &m_Event;
    }

    //!< 多重待ちイベントオブジェクトを有効にします。
    void Initialize() NN_NOEXCEPT
    {
        NN_ABORT_UNLESS(!m_ActivationCount.IsMax());

        if (m_ActivationCount.IsZero())
        {
            // イベントオブジェクトを初期化
            ::nn::os::InitializeEvent(&m_Event,
                                      false,
                                      ::nn::os::EventClearMode_ManualClear);

            // 多重待ちオブジェクトホルダーを初期化
            ::nn::os::InitializeMultiWaitHolder(&m_MultiWaitHolder, &m_Event);
        }
        ++m_ActivationCount;
    }

    //!< 多重待ちイベントオブジェクトを解放します。
    void Finalize() NN_NOEXCEPT
    {
        NN_ABORT_UNLESS(!m_ActivationCount.IsZero());

        --m_ActivationCount;

        if (m_ActivationCount.IsZero())
        {
            // 多重待ちオブジェクトホルダーを破棄
            ::nn::os::FinalizeMultiWaitHolder(&m_MultiWaitHolder);

            // イベントオブジェクトを破棄
            ::nn::os::FinalizeEvent(&m_Event);
        }
    }

    //!< 多重待ちオブジェクトにイベントオブジェクトをリンクします。
    virtual void Link(::nn::os::MultiWaitType* pMultiWait) NN_NOEXCEPT NN_OVERRIDE
    {
        NN_SDK_REQUIRES_NOT_NULL(pMultiWait);
        NN_SDK_REQUIRES(!m_ActivationCount.IsZero());
        ::nn::os::LinkMultiWaitHolder(pMultiWait, &m_MultiWaitHolder);
    }

    //!< イベントオブジェクトを全ての多重待ちオブジェクトからアンリンクします。
    virtual void Unlink() NN_NOEXCEPT NN_OVERRIDE
    {
        NN_SDK_REQUIRES(!m_ActivationCount.IsZero());
        ::nn::os::UnlinkMultiWaitHolder(&m_MultiWaitHolder);
    }

    //!< 多重待ちイベントを非シグナル状態にクリアします。
    virtual void Clear() NN_NOEXCEPT NN_OVERRIDE
    {
        NN_SDK_REQUIRES(!m_ActivationCount.IsZero());
        ::nn::os::ClearEvent(&m_Event);
    }
};

//!< 多重待ちタイマーイベントを扱うクラスの基底クラスです。
class MultiWaitTimerEvent final
    : public IMultiWaitBase
{
private:
    ::nn::os::TimerEventType m_Event;

public:
    virtual ~MultiWaitTimerEvent() NN_NOEXCEPT NN_OVERRIDE
    {
    }

    //!< 多重待ちタイマーオブジェクトを有効にします。
    void Initialize() NN_NOEXCEPT
    {
        NN_ABORT_UNLESS(!m_ActivationCount.IsMax());

        if (m_ActivationCount.IsZero())
        {
            // タイマーイベントオブジェクトを初期化
            ::nn::os::InitializeTimerEvent(&m_Event,
                                           ::nn::os::EventClearMode_ManualClear);

            // 多重待ちオブジェクトホルダーを初期化
            ::nn::os::InitializeMultiWaitHolder(&m_MultiWaitHolder, &m_Event);
        }
        ++m_ActivationCount;
    }

    //!< 多重待ちタイマーオブジェクトを解放します。
    void Finalize() NN_NOEXCEPT
    {
        NN_ABORT_UNLESS(!m_ActivationCount.IsZero());

        --m_ActivationCount;;

        if (m_ActivationCount.IsZero())
        {
            // 多重待ちオブジェクトホルダーを破棄
            ::nn::os::FinalizeMultiWaitHolder(&m_MultiWaitHolder);

            // タイマーイベントオブジェクトを破棄
            ::nn::os::FinalizeTimerEvent(&m_Event);
        }
    }

    //!< 多重待ちオブジェクトにイベントオブジェクトをリンクします。
    virtual void Link(::nn::os::MultiWaitType* pMultiWait) NN_NOEXCEPT NN_OVERRIDE
    {
        NN_SDK_REQUIRES(!m_ActivationCount.IsZero());
        NN_SDK_REQUIRES_NOT_NULL(pMultiWait);
        ::nn::os::LinkMultiWaitHolder(pMultiWait, &m_MultiWaitHolder);
    }

    //!< イベントオブジェクトを全ての多重待ちオブジェクトからアンリンクします。
    virtual void Unlink() NN_NOEXCEPT NN_OVERRIDE
    {
        NN_SDK_REQUIRES(!m_ActivationCount.IsZero());
        ::nn::os::UnlinkMultiWaitHolder(&m_MultiWaitHolder);
    }

    //!< イベントオブジェクトを取得します
    ::nn::os::TimerEventType* GetBase() NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(!m_ActivationCount.IsZero());
        return &m_Event;
    }

    //!< 多重待ちタイマーイベントを非シグナル状態にクリアします。
    virtual void Clear() NN_NOEXCEPT NN_OVERRIDE
    {
        NN_SDK_REQUIRES(!m_ActivationCount.IsZero());
        ::nn::os::ClearTimerEvent(&m_Event);
    }
};

//!< 多重待ちシステムイベントを扱うクラスの基底クラスです。
class MultiWaitSystemEvent final
    : public IMultiWaitBase
{
private:
    ::nn::os::SystemEventType m_Event;

public:
    virtual ~MultiWaitSystemEvent() NN_NOEXCEPT NN_OVERRIDE
    {
        // 何もしない
    }

    //!< イベントオブジェクトを取得します
    ::nn::os::SystemEventType* GetBase() NN_NOEXCEPT
    {
        return &m_Event;
    }

    //!< 多重待ちオブジェクトにイベントオブジェクトをリンクします。
    virtual void Link(::nn::os::MultiWaitType* pMultiWait) NN_NOEXCEPT NN_OVERRIDE
    {
        NN_SDK_REQUIRES_NOT_NULL(pMultiWait);
        // 多重待ちオブジェクトホルダーを初期化
        ::nn::os::InitializeMultiWaitHolder(&m_MultiWaitHolder, &m_Event);

        ::nn::os::LinkMultiWaitHolder(pMultiWait, &m_MultiWaitHolder);
    }

    //!< イベントオブジェクトを全ての多重待ちオブジェクトからアンリンクします。
    virtual void Unlink() NN_NOEXCEPT NN_OVERRIDE
    {
        ::nn::os::UnlinkMultiWaitHolder(&m_MultiWaitHolder);

        // 多重待ちオブジェクトホルダーを破棄
        ::nn::os::FinalizeMultiWaitHolder(&m_MultiWaitHolder);
    }

    //!< 多重待ちシステムイベントを非シグナル状態にクリアします。
    virtual void Clear() NN_NOEXCEPT NN_OVERRIDE
    {
        ::nn::os::ClearSystemEvent(&m_Event);
    }

    //!< イベントオブジェクトを破棄します。
    void Destroy() NN_NOEXCEPT
    {
        ::nn::os::DestroySystemEvent(&m_Event);
    }
};

}} // namespace nn::xcd
