﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <mutex>
#include <nn/nn_Macro.h>
#include <nn/nn_Result.h>
#include <nn/nn_SdkAssert.h>
#include <nn/nn_SdkLog.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/xcd/xcd_Result.h>
#include "xcd_IrsensorTeraPluginProcessor.h"

namespace nn { namespace xcd {

IrsensorTeraPluginProcessor::~IrsensorTeraPluginProcessor() NN_NOEXCEPT
{
    // 何もしない
}

Result IrsensorTeraPluginProcessor::ParseTeraPluginData(const uint8_t* pBuffer, size_t size, uint8_t sampleNumber) NN_NOEXCEPT
{
    std::lock_guard<nn::os::Mutex> lock(m_Mutex);

    NN_UNUSED(size);
    NN_UNUSED(sampleNumber);
    if (!IsProcessorReady())
    {
        return nn::xcd::ResultIrProcessorNotReady();
    }

    // ImageHeader の情報を取得する
    ParseIrsensorHeader(pBuffer, size, sampleNumber);

    // Timestampの更新
    m_pTeraPluginProcessorState->samplingNumber = sampleNumber;
    // diffTimestampの更新
    m_pTeraPluginProcessorState->diffTimeStampCount = m_DiffTimeStamp;

    // 単純にコピーする
    std::memcpy(m_pTeraPluginProcessorState->rawData, pBuffer + InputReportHeaderSize, TeraPluginProcessorInputReportSize);

    NN_RESULT_SUCCESS;
}

Result IrsensorTeraPluginProcessor::SetupProcessor(IrCommonData* pIrCommonWorkBuffer, IrTeraPluginProcessorState* pTeraPluginProcessorWorkBuffer) NN_NOEXCEPT
{
    std::lock_guard<nn::os::Mutex> lock(m_Mutex);

    NN_UNUSED(pIrCommonWorkBuffer);
    // ワークバッファを保持する
    m_pTeraPluginProcessorState = pTeraPluginProcessorWorkBuffer;
    m_DiffTimeStamp = 0;
    m_TimeStamp = 0;
    m_IsProcessorReady = true;
    NN_RESULT_SUCCESS;
}

Result IrsensorTeraPluginProcessor::StopProcessor() NN_NOEXCEPT
{
    std::lock_guard<nn::os::Mutex> lock(m_Mutex);

    if (!IsProcessorReady())
    {
        return nn::xcd::ResultIrProcessorNotReady();
    }

    // ワークバッファを解放する
    m_pTeraPluginProcessorState = nullptr;
    m_IsProcessorReady = false;
    NN_RESULT_SUCCESS;
}

Result IrsensorTeraPluginProcessor::GetTeraPluginState(IrCommonData* pOutIrCommonData, IrTeraPluginProcessorState* pOutTeraPluginProcessorState) NN_NOEXCEPT
{
    std::lock_guard<nn::os::Mutex> lock(m_Mutex);

    if (!IsProcessorReady())
    {
        return nn::xcd::ResultIrProcessorNotReady();
    }

    *pOutIrCommonData = m_IrCommonData[0];
    pOutTeraPluginProcessorState->samplingNumber = m_pTeraPluginProcessorState->samplingNumber;
    pOutTeraPluginProcessorState->diffTimeStampCount = m_pTeraPluginProcessorState->diffTimeStampCount;
    std::memcpy(pOutTeraPluginProcessorState->rawData, m_pTeraPluginProcessorState->rawData, TeraPluginProcessorInputReportSize);

    // コピーが終わったらMutexを解く
    NN_RESULT_SUCCESS;
}

}} // namespace nn::xcd
