﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Result.h>
#include <nn/nn_Macro.h>
#include <nn/os/os_SystemEventTypes.h>
#include "xcd_IrsensorProcessorBase.h"

namespace nn { namespace xcd {

//!< Moment Processor の実装クラス
class IrsensorMomentProcessor final : public IrsensorProcessorBase
{
public:
    IrsensorMomentProcessor() NN_NOEXCEPT :
        m_StatesId(0),
        m_ValidStatesCount(0)
    {
        for (int i = 0; i < IrMomentProcessorStateCountMax; i++)
        {
            m_pMomentProcessorStates[i] = nullptr;
        }
    }

    virtual ~IrsensorMomentProcessor() NN_NOEXCEPT;

    //!< IRセンサのMomentモードのデータをパースします。
    Result ParseMomentData(const uint8_t* pBuffer, size_t size, uint8_t sampleNumber) NN_NOEXCEPT;
    //!< IRセンサのMomentモードのワークバッファを設定します。
    Result SetupProcessor(IrCommonData* pIrCommonWorkBuffer, IrMomentProcessorState* pMomentProcessorWorkBuffer) NN_NOEXCEPT;
    //!< IRセンサのMomentモードを終了し、ワークバッファを解放します。
    Result StopProcessor() NN_NOEXCEPT;
    //!< IRセンサのMomentモードのデータを取得します。複数フレーム取得することができます。
    Result GetMomentStates(
        IrCommonData* pOutIrCommonData,
        IrMomentProcessorState* pOutMomentProcessorStates,
        int* pOutAckCount,
        int* pOutCount,
        int countMax) NN_NOEXCEPT;

private:
    //!< IRセンサのMomentブロックのデータを更新します。内部でデータチェックします。
    Result UpdateMomentBlockData(int index, int m00, int centroidX, int centroidY) NN_NOEXCEPT;

private:
    static const size_t MomentProcessorInputReportSize = 300; //!< MomentモードのInputReportのサイズ
    static const int MomentProcessorInputReportOffset = 10;    //!< MomentモードのデータのInputReporまでのオフセット
    static const int MomentPacketFormatRegionCount = 12; //!< Momentモードのパケット内の分割領域の数
    static const int MomentBlockCount = 48;          //!< Momentモードのブロック分割数
    static const size_t MomentPacketFormatRegionSize
        = MomentProcessorInputReportSize / MomentPacketFormatRegionCount; //!< Moment モードのパケット内の分割領域のサイズ
    static const int MomentBlockCountPerRegion =
        IrMomentProcessorBlockCount / MomentPacketFormatRegionCount;     //!< Moment モードの分割領域内に含まれるブロック数

private:
    IrMomentProcessorState* m_pMomentProcessorStates[IrMomentProcessorStateCountMax]; //!< Momentモードの出力結果のバッファポインタ
    int m_StatesId;                                                             //!< 最新のデータが読み込まれているバッファのインデックス
    int m_ValidStatesCount;                                                     //!< バッファ内で有効なデータの数
};

}} // namespace nn::xcd
