﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <mutex>
#include <nn/nn_Macro.h>
#include <nn/nn_Result.h>
#include <nn/nn_SdkAssert.h>
#include <nn/nn_SdkLog.h>
#include <nn/xcd/xcd_Result.h>
#include <nn/xcd/xcd_ResultForPrivate.h>
#include <nn/result/result_HandlingUtility.h>
#include "xcd_IrsensorBase.h"
#include "xcd_IrsensorMomentProcessor.h"

namespace nn { namespace xcd {

IrsensorMomentProcessor::~IrsensorMomentProcessor() NN_NOEXCEPT
{
    // 何もしない
}

Result IrsensorMomentProcessor::ParseMomentData(const uint8_t* pBuffer, size_t size, uint8_t sampleNumber) NN_NOEXCEPT
{
    std::lock_guard<nn::os::Mutex> lock(m_Mutex);

    NN_UNUSED(size);
    NN_UNUSED(sampleNumber);

    if (!IsProcessorReady())
    {
        return nn::xcd::ResultIrProcessorNotReady();
    }

    Result result = ResultSuccess();

    // ImageHeader の情報を取得する
    ParseIrsensorHeader(pBuffer, size, sampleNumber);

    // とりあえず常に先頭バッファを上書きする
    m_ValidStatesCount = 0;

    // バイト列のパースを行う
    for (int i = 0; i < MomentPacketFormatRegionCount; i++)
    {
        const uint8_t* blockData = pBuffer + i * MomentPacketFormatRegionSize + InputReportHeaderSize;

        int m00;
        int centroidX;
        int centroidY;
        int index = MomentBlockCountPerRegion * i;

        m00 = (static_cast<int>(blockData[0]) & 0xFF)
            | ((static_cast<int>(blockData[1]) & 0xFF) << 8)
            | ((static_cast<int>(blockData[2]) & 0x07) << 16);
        centroidX = ((static_cast<int>(blockData[2]) & 0xF8) >> 3)
            | ((static_cast<int>(blockData[3]) & 0xFF) << 5)
            | ((static_cast<int>(blockData[4]) & 0x07) << 13);
        centroidY = ((static_cast<int>(blockData[4]) & 0xF8) >> 3)
            | ((static_cast<int>(blockData[5]) & 0xFF) << 5)
            | ((static_cast<int>(blockData[6]) & 0x03) << 13);

        result = UpdateMomentBlockData(index, m00, centroidX, centroidY);
        if (result.IsFailure())
        {
            break;
        }

        m00 = ((static_cast<int>(blockData[6]) & 0xFC) >> 2)
            | ((static_cast<int>(blockData[7]) & 0xFF) << 6)
            | ((static_cast<int>(blockData[8]) & 0x1F) << 14);
        centroidX = ((static_cast<int>(blockData[8]) & 0xE0) >> 5)
            | ((static_cast<int>(blockData[9]) & 0xFF) << 3)
            | ((static_cast<int>(blockData[10]) & 0x1F) << 11);
        centroidY = ((static_cast<int>(blockData[10]) & 0xE0) >> 5)
            | ((static_cast<int>(blockData[11]) & 0xFF) << 3)
            | ((static_cast<int>(blockData[12]) & 0x0F) << 11);

        result = UpdateMomentBlockData(index + 1, m00, centroidX, centroidY);
        if (result.IsFailure())
        {
            break;
        }

        m00 = ((static_cast<int>(blockData[12]) & 0xF0) >> 4)
            | ((static_cast<int>(blockData[13]) & 0xFF) << 4)
            | ((static_cast<int>(blockData[14]) & 0x7F) << 12);
        centroidX = ((static_cast<int>(blockData[14]) & 0x80) >> 7)
            | ((static_cast<int>(blockData[15]) & 0xFF) << 1)
            | ((static_cast<int>(blockData[16]) & 0x7F) << 9);
        centroidY = ((static_cast<int>(blockData[16]) & 0x80) >> 7)
            | ((static_cast<int>(blockData[17]) & 0xFF) << 1)
            | ((static_cast<int>(blockData[18]) & 0x3F) << 9);

        result = UpdateMomentBlockData(index + 2, m00, centroidX, centroidY);
        if (result.IsFailure())
        {
            break;
        }

        m00 = ((static_cast<int>(blockData[18]) & 0xC0) >> 6)
            | ((static_cast<int>(blockData[19]) & 0xFF) << 2)
            | ((static_cast<int>(blockData[20]) & 0xFF) << 10)
            | ((static_cast<int>(blockData[21]) & 0x01) << 18);
        centroidX = ((static_cast<int>(blockData[21]) & 0xFE) >> 1)
            | ((static_cast<int>(blockData[22]) & 0xFF) << 7)
            | ((static_cast<int>(blockData[23]) & 0x01) << 15);
        centroidY = ((static_cast<int>(blockData[23]) & 0xFE) >> 1)
            | ((static_cast<int>(blockData[24]) & 0xFF) << 7);

        result = UpdateMomentBlockData(index + 3, m00, centroidX, centroidY);
        if (result.IsFailure())
        {
            break;
        }
    }

    // SampleNumberの更新
    m_pMomentProcessorStates[m_StatesId]->samplingNumber = sampleNumber;
    // diffTimestampの更新
    m_pMomentProcessorStates[m_StatesId]->diffTimeStampCount = m_DiffTimeStamp;

    if (result.IsSuccess())
    {
        // 最新のインデックスを更新
        m_StatesId = (m_StatesId + 1) % IrMomentProcessorStateCountMax;

        // 有効なバッファ数を更新
        if(m_ValidStatesCount < IrMomentProcessorStateCountMax)
        {
            m_ValidStatesCount++;
        }
    }
    else
    {
        // 途中までコピーされていた可能性があるため、クリアしておく。
        std::memset(m_pMomentProcessorStates[m_StatesId], 0, sizeof(IrMomentProcessorState));
    }
    NN_RESULT_SUCCESS;
}

Result IrsensorMomentProcessor::UpdateMomentBlockData(int index, int m00, int centroidX, int centroidY) NN_NOEXCEPT
{
    const int BlockSize = IrImageSizeMax / IrMomentProcessorBlockCount;

    if(m00 > 0)
    {
        float averageIntensity = m00 / static_cast<float>(BlockSize);
        if (averageIntensity < 0.0f || averageIntensity > 255.0f)
        {
            NN_XCD_IRSENSOR_LOG("MomentInvalidData: blockId:%d averageIntensity:%f\n", index, averageIntensity);
            nn::xcd::ResultIrInvalidPacketData();
        }
        m_pMomentProcessorStates[m_StatesId]->blocks[index].averageIntensity = averageIntensity;
        float centerX = centroidX / static_cast<float>(1 << 7);
        if (centerX < 0 || centerX >= nn::xcd::IrImageWidthMax)
        {
            NN_XCD_IRSENSOR_LOG("MomentInvalidData: blockId:%d centerX:%f\n", index, centerX);
            nn::xcd::ResultIrInvalidPacketData();
        }
        m_pMomentProcessorStates[m_StatesId]->blocks[index].centroid.x = centerX;
        float centerY = centroidY / static_cast<float>(1 << 7);
        if (centerY < 0 || centerY >= nn::xcd::IrImageHeightMax)
        {
            NN_XCD_IRSENSOR_LOG("MomentInvalidData: blockId:%d centerY:%f\n", index, centerY);
            nn::xcd::ResultIrInvalidPacketData();
        }
        m_pMomentProcessorStates[m_StatesId]->blocks[index].centroid.y = centerY;
    }
    else
    {
        m_pMomentProcessorStates[m_StatesId]->blocks[index].averageIntensity =  0.0f;
        m_pMomentProcessorStates[m_StatesId]->blocks[index].centroid.x =  0.0f;
        m_pMomentProcessorStates[m_StatesId]->blocks[index].centroid.y =  0.0f;
    }
    NN_RESULT_SUCCESS;
}

Result IrsensorMomentProcessor::SetupProcessor(IrCommonData* pIrCommonWorkBuffer, IrMomentProcessorState* pMomentProcessorWorkBuffer) NN_NOEXCEPT
{
    std::lock_guard<nn::os::Mutex> lock(m_Mutex);

    NN_UNUSED(pIrCommonWorkBuffer);
    NN_UNUSED(m_pMomentProcessorStates);
    // ワークバッファを保持する
    m_pMomentProcessorStates[0] = pMomentProcessorWorkBuffer;
    m_TimeStamp = 0;
    m_DiffTimeStamp = 0;
    m_IsProcessorReady = true;
    NN_RESULT_SUCCESS;
}

Result IrsensorMomentProcessor::StopProcessor() NN_NOEXCEPT
{
    std::lock_guard<nn::os::Mutex> lock(m_Mutex);

    if (!IsProcessorReady())
    {
        return nn::xcd::ResultIrProcessorNotReady();
    }

    // ワークバッファを解放する
    m_pMomentProcessorStates[0] = nullptr;
    m_IsProcessorReady = false;
    NN_RESULT_SUCCESS;
}

Result IrsensorMomentProcessor::GetMomentStates(
    IrCommonData* pOutIrCommonData,
    IrMomentProcessorState* pOutMomentProcessorStates,
    int* pOutAckCount,
    int* pOutCount,
    int countMax) NN_NOEXCEPT
{
    std::lock_guard<nn::os::Mutex> lock(m_Mutex);

    NN_UNUSED(pOutMomentProcessorStates);
    if (!IsProcessorReady())
    {
        return nn::xcd::ResultIrProcessorNotReady();
    }

    int counter = 0;
    int validDataCount = (countMax > m_ValidStatesCount) ? m_ValidStatesCount : countMax;
    int index = m_StatesId;

    // 最新データが先頭にくるようにコピーする
    for(int i = 0; i < validDataCount; i++)
    {
        *(pOutIrCommonData + i) = m_IrCommonData[index];
        std::memcpy((pOutMomentProcessorStates + i), m_pMomentProcessorStates[index], sizeof(IrMomentProcessorState) );
        counter++;
        index = (index - 1 + IrMomentProcessorStateCountMax) % IrMomentProcessorStateCountMax;
    }
    *pOutCount = counter;
    *pOutAckCount = m_LatestAckCount;

    // コピーが終わったらMutexを解く
    NN_RESULT_SUCCESS;
}

}} // namespace nn::xcd
