﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Result.h>
#include <nn/nn_Macro.h>
#include <nn/os/os_SystemEventTypes.h>
#include "xcd_IrsensorProcessorBase.h"

namespace nn { namespace xcd {

const int IMAGE_TRANSFER_FRAME_COUNT_MAX = 256; //!< 画像転送モードのフレーム分割数の最大値
/**
 * @brief   画像転送モードの受信完了フラグ
 */
typedef ::nn::util::BitFlagSet<IMAGE_TRANSFER_FRAME_COUNT_MAX> ReceivedFrameFlag;

//!< ImageTransfer Processor の実装クラス
class IrsensorImageTransferProcessor final : public IrsensorProcessorBase
{
public:
    IrsensorImageTransferProcessor() NN_NOEXCEPT :
        m_pImageTransferProcessorState(nullptr),
        m_ImageTransferFormat(IrImageTransferProcessorFormat::ImageSize_320x240),
        m_ImageId(0),
        m_PrevImageId(0),
        m_IsImageReady(false),
        m_IsFrameDropped(false),
        m_IsCompletionCommandAvailable(false),
        m_ContinuousFrameCount(0),
        m_TransferProtocol(IrImageTransferProtocol::FastMode)
    {
        // ALL FF にしておく
        m_ReceivedFrameFlag.Set();
    }

    virtual ~IrsensorImageTransferProcessor() NN_NOEXCEPT;

    //!< IRセンサの ImageTransfer モードのデータをパースします。
    Result ParseImageTransferData(const uint8_t* pBuffer, size_t size, uint8_t sampleNumber) NN_NOEXCEPT;
    //!< IRセンサーの ImageTransfer モードのワークバッファを設定します。
    Result SetupProcessor(IrCommonData* pIrCommonWorkBuffer, IrImageTransferProcessorState* pImageTransferProcessorWorkBuffer, IrImageTransferProcessorFormat size) NN_NOEXCEPT;
    //!< IRセンサーの ImageTransfer モードを停止して、ワークバッファを解放します。
    Result StopProcessor() NN_NOEXCEPT;
    //!< IRセンサの ImageTransfer モードのデータを取得します。複数フレーム取得することができます。
    Result GetImageTransferState(IrCommonData* pOutIrCommonData, IrImageTransferProcessorState* pOutImageTransferProcessorState, int* pOutAckCount) NN_NOEXCEPT;
    //!< 受信内容を元に次パケットに送信するコマンドを設定する
    Result SetDataReadCommand(uint8_t& resendFlag, uint8_t& resendId, uint8_t& imageId, ReceivedFrameFlag& receivedFrameFlag) NN_NOEXCEPT;
    //!< 設定されている画像サイズを取得します。
    int GetTransferFormat(IrImageTransferProcessorFormat* pSize) NN_NOEXCEPT;
    //!< 転送サイズに応じたパケット数を返します。
    int GetTransferPacketCount(IrImageTransferProcessorFormat size) NN_NOEXCEPT;
    //!< 転送サイズに応じた画像データサイズを返します。
    size_t GetTransferSize(IrImageTransferProcessorFormat size) NN_NOEXCEPT;
    //!< 転送プロトコルを設定します。
    Result SetTransferProtocol(IrImageTransferProtocol protocol) NN_NOEXCEPT;

public:
    struct ImageSizeInfo
    {
        int width;
        int height;
        int packetCnt;
    };

private:
    static const size_t ImageTransferProcessorInputReportSize = 300; //!< ImageTransferモードのInputReportのサイズ
    static const int ImageTransferProcessorInputReportOffset = 10;   //!< ImageTransferモードのデータのInputReportまでのオフセット
    static const int ImageTransferFormatTypeCountMax = 5;            //!< ImageTransferモードの画像設定サイズの最大数
    static const ImageSizeInfo TransferSizeInfo[ImageTransferFormatTypeCountMax];    //!< 画像サイズごとの設定値
    static const int ContinuousLastFrameCountMax = 15;               //!< 異常判定用の連続最終フレーム到着上限数

private:
    IrImageTransferProcessorState* m_pImageTransferProcessorState; //!< ImageTransferモードの出力結果のワークバッファポインタ
    IrImageTransferProcessorFormat m_ImageTransferFormat;   //!< 転送する画像サイズ
    int m_ImageId;     //!< 最新フレームの番号
    int m_PrevImageId; //!< 最新の転送済みフレームの番号
    bool m_IsImageReady; //!< 画像データが取得できるかどうかを判別するフラグ
    bool m_IsFrameDropped; //!< フレーム落ちが発生したかどうかのフラグ
    bool m_IsCompletionCommandAvailable; //!< 完了通知の送信可能フラグ
    int m_ContinuousFrameCount; //!< 同じ画像に対する分割データが連続到着した数
    IrImageTransferProtocol m_TransferProtocol; //!< 転送プロトコル
    ReceivedFrameFlag m_ReceivedFrameFlag; //!< フレーム到着状況をあらわすビットフィールド(0 が到着済み)
};

}} // namespace nn::xcd
