﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Result.h>
#include <nn/nn_Macro.h>
#include <nn/os/os_SystemEventTypes.h>
#include "xcd_IrsensorProcessorBase.h"

namespace nn { namespace xcd {

//!< Dpd Processor の実装クラス
class IrsensorDpdProcessor final : public IrsensorProcessorBase
{
public:
    IrsensorDpdProcessor() NN_NOEXCEPT :
        m_ValidStatesCnt(0)
    {
        for (int i = 0; i < IrDpdProcessorStateCountMax; i++)
        {
            m_pDpdProcessorStates[i] = nullptr;
        }
    }

    virtual ~IrsensorDpdProcessor() NN_NOEXCEPT;

    //!< IRセンサの Dpd モードのデータをパースします。
    Result ParseDpdData(const uint8_t* pBuffer, size_t size, uint8_t sampleNumber) NN_NOEXCEPT;
    //!< IRセンサーの Dpd モードのワークバッファを設定します。
    Result SetupProcessor(IrCommonData* pIrCommonWorkBuffer, IrDpdProcessorState* pDpdProcessorWorkBuffer) NN_NOEXCEPT;
    //!< IRセンサーの Dpd モードを終了し、ワークバッファを解放します。
    Result StopProcessor() NN_NOEXCEPT;
    //!< IRセンサの Dpd モードのデータを取得します。複数フレーム取得することができます。
    Result GetDpdStates(IrCommonData* pOutIrCommonData, IrDpdProcessorState* pOutDpdProcessorStates, int* pOutCount, int countMax) NN_NOEXCEPT;

private:
    //!< Dpd モードのクラスタ生データフォーマット
    struct NN_ALIGNAS(2) ClusterRawBlock
    {
        uint16_t averageIntensity; //!< 平均輝度
        uint16_t pixelCount;       //!< ピクセル数
        uint16_t centroidX;        //!< 重心点のX座標
        uint16_t centroidY;        //!< 重心点のY座標
        uint16_t left;             //!< クラスタ枠の左端X座標
        uint16_t right;            //!< クラスタ枠の右端X座標
        uint16_t up;               //!< クラスタ枠の上端Y座標
        uint16_t down;             //!< クラスタ枠の下端Y座標
    };

private:
    //!< Dpd モードのコピーを行います。内部でデータチェックします。
    Result CopyDpdPacketData(const uint8_t* pBuffer, size_t size, int index) NN_NOEXCEPT;

private:
    static const size_t DpdProcessorInputReportSize = 300; //!< DpdモードのInputReportのサイズ
    static const int DpdProcessorInputReportOffset = 10;   //!< DpdモードのデータのInputReportまでのオフセット
    static const int DpdProcessorDataOffset
        = DpdProcessorInputReportOffset + 1;               //!< Dpdモードのデータのクラスタデータ先頭のオフセット
    static const size_t DpdFrameDataSize = 49;             //!< Dpdモードの1フレームのデータサイズ
    static const float DpdAverageIntensityDivider;         //!< Dpdモードの輝度平均値を計算するための除数
    static const float DpdCentroidDivider;                 //!< Dpdモードの重心座標値を計算するための除数

private:
    IrDpdProcessorState* m_pDpdProcessorStates[IrDpdProcessorStateCountMax]; //!< Dpdモードの出力結果のバッファ
    int m_ValidStatesCnt;                                                    //!< 取得した有効な Dpd データの数
};

}} // namespace nn::xcd
