﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <algorithm>
#include <mutex>
#include <nn/nn_Macro.h>
#include <nn/nn_Result.h>
#include <nn/nn_SdkAssert.h>
#include <nn/nn_SdkLog.h>
#include <nn/xcd/xcd_Result.h>
#include <nn/xcd/xcd_ResultForPrivate.h>
#include <nn/result/result_HandlingUtility.h>
#include "xcd_IrsensorBase.h"
#include "xcd_IrsensorDpdProcessor.h"

namespace nn { namespace xcd {

const float IrsensorDpdProcessor::DpdAverageIntensityDivider = 256.0f;
const float IrsensorDpdProcessor::DpdCentroidDivider         = 64.0f;

IrsensorDpdProcessor::~IrsensorDpdProcessor() NN_NOEXCEPT
{
    // 何もしない
}

Result IrsensorDpdProcessor::ParseDpdData(const uint8_t* pBuffer, size_t size, uint8_t sampleNumber) NN_NOEXCEPT
{
    std::lock_guard<nn::os::Mutex> lock(m_Mutex);

    NN_UNUSED(size);
    NN_UNUSED(sampleNumber);

    if (!IsProcessorReady())
    {
        return nn::xcd::ResultIrProcessorNotReady();
    }

    // ImageHeader の情報を取得する
    ParseIrsensorHeader(pBuffer, size, sampleNumber);

    // 詰め込まれたデータの数を取得
    int dataCount = pBuffer[DpdProcessorInputReportOffset];
    if (dataCount < 0 || dataCount > nn::xcd::IrDpdProcessorStateCountMax)
    {
        NN_XCD_IRSENSOR_LOG("DpdInvalidData: dataCount:%d\n", dataCount);
        return nn::xcd::ResultIrInvalidPacketData();
    }
    m_ValidStatesCnt = dataCount;

    // 全て同じデータを埋めておく。
    // 現状正確には取得できないので、インターバルが等間隔であると仮定してライブラリ側でデータ補正をする。
    for (auto i = 0; i < dataCount; i++)
    {
        // Timestampの更新 (先頭から順に番号をふる)
        m_pDpdProcessorStates[i]->samplingNumber = m_IrCommonData[0].frameId - i;

        // diffTimestampの更新
        if (i == 0)
        {
            // 先頭は前回のフレームの最後からの差分フレーム
            m_pDpdProcessorStates[i]->diffTimeStampCount = std::max(m_DiffTimeStamp - dataCount + 1, 1);
        }
        else
        {
            m_pDpdProcessorStates[i]->diffTimeStampCount = 1;
        }
    }

    // コピーを行う。内部でデータチェックする。
    // (IAAA-2362) ヘッダは正しいが、中身が不正なパケットが送られてくる問題対策
    Result result = CopyDpdPacketData(pBuffer, size, m_ValidStatesCnt);
    if (result.IsFailure())
    {
        for (auto i = 0; i < m_ValidStatesCnt; i++)
        {
            // 途中までコピーされていた可能性があるため、クリアしておく。
            std::memset(m_pDpdProcessorStates[i], 0, sizeof(IrDpdProcessorState));
        }
    }
    NN_SDK_ASSERT(m_ValidStatesCnt <= IrDpdProcessorStateCountMax);
    NN_RESULT_SUCCESS;
}

Result IrsensorDpdProcessor::SetupProcessor(IrCommonData* pIrCommonWorkBuffer, IrDpdProcessorState* pDpdProcessorWorkBuffer) NN_NOEXCEPT
{
    std::lock_guard<nn::os::Mutex> lock(m_Mutex);

    NN_UNUSED(pIrCommonWorkBuffer);
    NN_SDK_REQUIRES_NOT_NULL(pDpdProcessorWorkBuffer);

    // ワークバッファを保持する
    for (int i = 0; i < IrDpdProcessorStateCountMax; i++)
    {
        m_pDpdProcessorStates[i] = pDpdProcessorWorkBuffer + i;
    }

    m_DiffTimeStamp = 0;
    m_TimeStamp = 0;
    m_IsProcessorReady = true;
    NN_RESULT_SUCCESS;
}

Result IrsensorDpdProcessor::StopProcessor() NN_NOEXCEPT
{
    std::lock_guard<nn::os::Mutex> lock(m_Mutex);

    if (!IsProcessorReady())
    {
        return nn::xcd::ResultIrProcessorNotReady();
    }

    // ワークバッファを解放する
    for (int i = 0; i < IrDpdProcessorStateCountMax; i++)
    {
        m_pDpdProcessorStates[i] = nullptr;
    }

    m_IsProcessorReady = false;
    NN_RESULT_SUCCESS;
}

Result IrsensorDpdProcessor::GetDpdStates(IrCommonData* pOutIrCommonData, IrDpdProcessorState* pOutDpdProcessorStates, int* pOutCount, int countMax) NN_NOEXCEPT
{
    std::lock_guard<nn::os::Mutex> lock(m_Mutex);

    if (!IsProcessorReady())
    {
        return nn::xcd::ResultIrProcessorNotReady();
    }

    int copyCountMax = std::min(countMax, m_ValidStatesCnt);

    // TODO : 最新のものから順に返すようにする
    for(int i = 0; i < copyCountMax; i++)
    {
        // Header については全て先頭の状態を返す
        *(pOutIrCommonData + i) = m_IrCommonData[0];
        std::memcpy((pOutDpdProcessorStates + i), m_pDpdProcessorStates[i], sizeof(IrDpdProcessorState));
    }
    *pOutCount = copyCountMax;

    // コピーが終わったらMutexを解いて、Eventを返す
    NN_RESULT_SUCCESS;
}

Result IrsensorDpdProcessor::CopyDpdPacketData(const uint8_t* pBuffer, size_t size, int index) NN_NOEXCEPT
{
    std::lock_guard<nn::os::Mutex> lock(m_Mutex);

    NN_UNUSED(size);

    if (!IsProcessorReady())
    {
        return nn::xcd::ResultIrProcessorNotReady();
    }

    // 各データ
    for (int i = 0; i < index; i++)
    {
        // 3つのクラスタ分の validity フラグを取得
        uint8_t validity = pBuffer[DpdProcessorDataOffset + i * DpdFrameDataSize];

        // 各クラスタの取得
        for (auto j = 0; j < IrDpdProcessorObjectCountMax; j++)
        {
            if ((validity >> j) & 0x01)
            {
                m_pDpdProcessorStates[i]->objects[j].isDataValid = true;
            }
            else
            {
                m_pDpdProcessorStates[i]->objects[j].isDataValid = false;
                continue;
            }

            // クラスタ先頭のオフセット
            size_t clusterStartOffset = DpdProcessorDataOffset + 1 + i * DpdFrameDataSize;
            ClusterRawBlock rawBlock = { 0 };
            std::memcpy(&rawBlock, &pBuffer[clusterStartOffset + j * sizeof(ClusterRawBlock)], sizeof(ClusterRawBlock));

            float averageIntensity = static_cast<float>(rawBlock.averageIntensity / DpdAverageIntensityDivider);
            if (averageIntensity < 0.0f || averageIntensity > 255.0f)
            {
                NN_XCD_IRSENSOR_LOG("DpdInvalidData: averageIntensity:%f\n", averageIntensity);
                return nn::xcd::ResultIrInvalidPacketData();
            }
            m_pDpdProcessorStates[i]->objects[j].averageIntensity = averageIntensity;

            // pixelCountについては全ての値を取りうるので、値チェックできない
            m_pDpdProcessorStates[i]->objects[j].pixelCount = rawBlock.pixelCount;

            float centroidX = static_cast<float>(rawBlock.centroidX) / DpdCentroidDivider;
            if (centroidX < 0 || centroidX >= nn::xcd::IrImageWidthMax)
            {
                NN_XCD_IRSENSOR_LOG("DpdInvalidData: centroidX:%f\n", centroidX);
                return nn::xcd::ResultIrInvalidPacketData();
            }
            m_pDpdProcessorStates[i]->objects[j].centroid.x = centroidX;

            float centroidY = static_cast<float>(rawBlock.centroidY) / DpdCentroidDivider;
            if (centroidY < 0 || centroidY >= nn::xcd::IrImageHeightMax)
            {
                NN_XCD_IRSENSOR_LOG("DpdInvalidData: centroidY:%f\n", centroidY);
                return nn::xcd::ResultIrInvalidPacketData();
            }
            m_pDpdProcessorStates[i]->objects[j].centroid.y = centroidY;

            int16_t boundX = rawBlock.left;
            if (boundX < 0 || boundX >= nn::xcd::IrImageWidthMax)
            {
                NN_XCD_IRSENSOR_LOG("DpdInvalidData: boundX:%d\n", boundX);
                return nn::xcd::ResultIrInvalidPacketData();
            }
            m_pDpdProcessorStates[i]->objects[j].bound.x = boundX;

            int16_t boundWidth = rawBlock.right - rawBlock.left + 1;
            if (boundWidth < 1 || boundWidth > nn::xcd::IrImageWidthMax)
            {
                NN_XCD_IRSENSOR_LOG("DpdInvalidData: boundWidth:%d\n", boundWidth);
                return nn::xcd::ResultIrInvalidPacketData();
            }
            m_pDpdProcessorStates[i]->objects[j].bound.width = boundWidth;

            int16_t boundY = rawBlock.up;
            if (boundY < 0 || boundY >= nn::xcd::IrImageHeightMax)
            {
                NN_XCD_IRSENSOR_LOG("DpdInvalidData: boundY:%d\n", boundY);
                return nn::xcd::ResultIrInvalidPacketData();
            }
            m_pDpdProcessorStates[i]->objects[j].bound.y = boundY;

            int16_t boundHeight = rawBlock.down - rawBlock.up + 1;
            if (boundHeight < 1 || boundHeight > nn::xcd::IrImageHeightMax)
            {
                NN_XCD_IRSENSOR_LOG("DpdInvalidData: boundHeight:%d\n", boundHeight);
                return nn::xcd::ResultIrInvalidPacketData();
            }
            m_pDpdProcessorStates[i]->objects[j].bound.height = boundHeight;
        }
    }

    NN_RESULT_SUCCESS;
}

}} // namespace nn::xcd
