﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Result.h>
#include <nn/nn_Macro.h>
#include <nn/os/os_SystemEventTypes.h>
#include "xcd_IrsensorProcessorBase.h"

namespace nn { namespace xcd {

//!< Clustering Processor の実装クラス
class IrsensorClusteringProcessor final : public IrsensorProcessorBase
{
public:
    IrsensorClusteringProcessor() NN_NOEXCEPT :
        m_ValidStatesCnt(0)
    {
        for (int i = 0; i < IrClusteringProcessorStateCountMax; i++)
        {
            m_pClusteringProcessorStates[i] = nullptr;
        }
    }

    virtual ~IrsensorClusteringProcessor() NN_NOEXCEPT;

    //!< IRセンサの Clustering モードのデータをパースします。
    Result ParseClusteringData(const uint8_t* pBuffer, size_t size, uint8_t sampleNumber) NN_NOEXCEPT;
    //!< IRセンサーの Clustering モードのワークバッファを設定します。
    Result SetupProcessor(IrCommonData* pIrCommonWorkBuffer, IrClusteringProcessorState* pClusteringProcessorWorkBuffer) NN_NOEXCEPT;
    //!< IRセンサーの Clustering モードを終了し、ワークバッファを解放します。
    Result StopProcessor() NN_NOEXCEPT;
    //!< IRセンサの Clustering モードのデータを取得します。複数フレーム取得することができます。
    Result GetClusteringStates(
        IrCommonData* pOutIrCommonData,
        IrClusteringProcessorState* pOutClusteringProcessorStates,
        int* pOutAckCount,
        int* pOutCount,
        int countMax) NN_NOEXCEPT;

private:
    //!< Clusteringモードの 生データフォーマット
    struct NN_ALIGNAS(2) ClusterRawBlock
    {
        uint16_t averageIntensity; //!< 平均輝度
        uint16_t pixelCount;       //!< ピクセル数
        uint16_t centroidX;        //!< 重心点のX座標
        uint16_t centroidY;        //!< 重心点のY座標
        uint16_t left;             //!< クラスタ枠の左端X座標
        uint16_t right;            //!< クラスタ枠の右端X座標
        uint16_t up;               //!< クラスタ枠の上端Y座標
        uint16_t down;             //!< クラスタ枠の下端Y座標
    };

private:
    //!< Clustering モードのコピーを行います。内部でデータチェックします。
    Result CopyClusteringPacketData(const uint8_t* pBuffer, size_t size, int index) NN_NOEXCEPT;

private:
    static const size_t ClusteringProcessorInputReportSize = 300; //!< ClusteringモードのInputReportのサイズ
    static const int ClusteringProcessorInputReportOffset = 10;   //!< ClusteringモードのデータのInputReportまでのオフセット
    static const int ClusteringProcessorDataOffset
        = ClusteringProcessorInputReportOffset + 2;               //!< Clusteringモードのデータのクラスタデータ先頭のオフセット
    static const int ClusteringSamplingCountMax = 3;              //!< ClusteringモードのデータがInputReportに含まれる最大フレーム数
    static const float ClusteringAverageIntensityDivider;         //!< Clusteringモードの輝度平均値を計算するための除数
    static const float ClusteringCentroidDivider;                 //!< Clusteringモードの重心座標値を計算するための除数

private:
    IrClusteringProcessorState* m_pClusteringProcessorStates[IrClusteringProcessorStateCountMax]; //!< Clusteringモードの出力結果のバッファ
    int m_ValidStatesCnt;                                                                         //!< 取得した有効な Clustering データの数
};

}} // namespace nn::xcd
