﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once


namespace nn { namespace xcd {

/**
 * @brief       IC 毎の Firmware バージョンを扱うための型です。
 */
struct FirmwareVersionUnit
{
    uint8_t major;       //!< メジャーバージョン
    uint8_t minor;       //!< マイナーバージョン

    uint16_t ToUint16() const NN_NOEXCEPT
    {
        return (static_cast<uint16_t>(major << 8) & 0xff00) |
               (static_cast<uint16_t>(minor) & 0x00ff);
    }

    /*!
     * @brief      値を比較します。
     */
    bool operator < (const FirmwareVersionUnit& value) const NN_NOEXCEPT
    {
        return this->ToUint16() < value.ToUint16();
    }

    /*!
     * @brief      値を比較します。
     */
    bool operator > (const FirmwareVersionUnit& value) const NN_NOEXCEPT
    {
        return this->ToUint16() > value.ToUint16();
    }

    /*!
     * @brief      値を比較します。
     */
    bool operator <= (const FirmwareVersionUnit& value) const NN_NOEXCEPT
    {
        return this->ToUint16() <= value.ToUint16();
    }

    /*!
     * @brief      値を比較します。
     */
    bool operator >= (const FirmwareVersionUnit& value) const NN_NOEXCEPT
    {
        return this->ToUint16() >= value.ToUint16();
    }
};

/**
 * @brief       ファームウェアのバージョンです。
 */
struct FirmwareVersionImpl
{
    FirmwareVersionUnit bluetooth;
    FirmwareVersionUnit tera;
};


// 特殊なバージョンの定義の一覧です。
// 実装分岐を行う場合などは、分岐を行う対象のバージョンを以下に追加してください。

/**
 * @brief       BT Firmware Version 01f4
 * @details     振動のデータフォーマットが変更されたバージョン
 */
const FirmwareVersionUnit FirmwareVersionBt_01f4 = { 0x01, 0xf4 };

/**
 * @brief       BT Firmware Version 0258
 * @details     アナログスティックのサンプリング処理の変更されたバージョン
 */
const FirmwareVersionUnit FirmwareVersionBt_025b = { 0x02, 0x5b };

/**
 * @brief       BT Firmware Version 0304
 * @details     ペアリング済みデバイスの有線ペアリングスキップ処理が追加されたバージョン
 */
const FirmwareVersionUnit FirmwareVersionBt_0304 = { 0x03, 0x04 };

/**
 * @brief       BT Firmware Version 0343
 * @details     サンプリングナンバーがセンサーのサンプリング依存して設定されるようになったバージョン
 */
const FirmwareVersionUnit FirmwareVersionBt_0343 = { 0x03, 0x43 };

/**
 * @brief       BT Firmware Version 0346
 * @details     Uart 経由での Firmware 更新をサポートするバージョン
 */
const FirmwareVersionUnit FirmwareVersionBt_0346 = { 0x03, 0x46 };

/**
 * @brief       BT Firmware Version 0348
 * @details     量産初期投入 Firmware
 */
const FirmwareVersionUnit FirmwareVersionBt_0348 = { 0x03, 0x48 };

/**
 * @brief       BT Firmware Version 0381
 * @details     MCU Polling の機能が追加されたバージョン
 */
const FirmwareVersionUnit FirmwareVersionBt_0381 = { 0x03, 0x81 };

/**
 * @brief       BT Firmware Version 0382
 * @details     MCU FW 高速更新機能が追加されたバージョン
 */
const FirmwareVersionUnit FirmwareVersionBt_0382 = { 0x03, 0x82 };

}} // namespace nn::xcd
