﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Macro.h>
#include <nn/nn_Result.h>
#include <nn/os/os_SystemEventTypes.h>
#include <nn/xcd/xcd_BluetoothSettings.h>
#include <nn/xcd/xcd_BleDevice.h>
#include <nn/xcd/xcd_Device.h>
#include <nn/xcd/xcd_Pairing.h>
#include <nn/xcd/xcd_Rail.h>

#include "xcd_IEventTask.h"
#include "xcd_BleDeviceHandler.h"
#include "xcd_DeviceHandler.h"
#include "xcd_MultiWaitEvent.h"
#include "xcd_PairingManager.h"
#include "detail/xcd_LinkMonitor.h"

namespace nn { namespace xcd {

const int MaxXcdDevices = 32;
const int MaxXcdBleDevices = 4;

//!< Nx デバイスを管理/制御するためのクラスです。
class DeviceListManager final : public IEventTask
{
private:
    //!< 新たなデバイスの接続を待ち受けるイベント
    ::nn::os::LightEventType m_DeviceUpdateEvent;

    //!< アプリに対して通知するためのイベントオブジェクト
    nn::os::SystemEventType* m_pStatusUpdateEvent;

    //!< Link upしているHIDデバイス
    detail::HidDeviceInfo m_LinkedDevices[MaxXcdDevices];

    //!< Link upしているBLEデバイス
    BleConnectionHandle m_LinkedBleDevices[MaxXcdBleDevices];

    //!< 制御対象のHIDデバイス
    DeviceHandler m_Handlers[MaxXcdDevices];

    //!< 制御対象のBLEデバイス
    BleDeviceHandler m_BleHandlers[MaxXcdBleDevices];

    bool m_Activated;

    //!< Suspend/Resume 処理の完了をシグナルするためのイベント
    nn::os::LightEventType m_SuspendCompleteEvent;

    //!< ペアリングマネージャー
    PairingManager m_PairingManager;

    bool m_WiredDeviceActivationEnabled;

    //!< Pro Controller 有線 USB 通信機能の状態
    bool m_FullKeyUsbEnabled;

    //!< Bluetooth 接続を試みるためのデバイスのアドレス
    ::nn::bluetooth::Address m_Address;

    //!< コントローラーへの接続トリガーがビジーだった場合にリトライするためのタイマー
    ::nn::os::TimerEventType m_BluetoothConnectionRetryEvent;

    //!< コントローラーへの接続トリガーが終了したことをまつためのタイマー
    ::nn::os::EventType m_BluetoothConnectionCompleteEvent;

    //!< コントローラーへの接続処理がタイムアウトした場合に通知するイベント
    ::nn::os::SystemEventType* m_pBluetoothConnectionTimeoutEvent;

    //!< ペアリング情報のクリアを待ち受けるイベント
    ::nn::os::LightEventType m_PairingDatabaseClearedEvent;

public:
    DeviceListManager() NN_NOEXCEPT;
    virtual ~DeviceListManager() NN_NOEXCEPT NN_OVERRIDE;

    //!< Eventが通知された際に呼ばれる関数
    virtual void EventFunction(const ::nn::os::MultiWaitHolderType* pMultiWaitHolder) NN_NOEXCEPT NN_OVERRIDE;

    //!< 定常処理を行う関数
    virtual void PeriodicEventFunction() NN_NOEXCEPT NN_OVERRIDE;

    void Activate() NN_NOEXCEPT;
    void Deactivate() NN_NOEXCEPT;

    //!< Suspend します。
    void Suspend() NN_NOEXCEPT;

    //!< Resume します。
    void Resume() NN_NOEXCEPT;

    //!< デバイスの接続状態に変更があった場合に通知されるイベントを登録します。
    void BindLinkUpdateEvent(nn::os::SystemEventType* pSystemEvent) NN_NOEXCEPT;

    //!< デバイスの接続状態に変更があった場合に通知されるイベントを解除します。
    void UnbindLinkUpdateEvent(nn::os::SystemEventType* pSystemEvent) NN_NOEXCEPT;

    //!< 接続されているデバイスを列挙します。
    Result ListDevices(DeviceList* pOutValue) NN_NOEXCEPT;

    //!< 接続されているデバイスを列挙します。
    Result ListDevices(BleDeviceList* pOutValue) NN_NOEXCEPT;

    //!< デバイスハンドルに割り当てられた DeviceHandler を取得します
    Result GetDeviceHandler(DeviceHandler** pOutValue, DeviceHandle deviceHandle) NN_NOEXCEPT;

    //!< デバイスハンドルに割り当てられた BleDeviceHandler を取得します
    Result GetBleDeviceHandler(BleDeviceHandler** pOutValue, BleConnectionHandle handle) NN_NOEXCEPT;

    //!< 新たに有線ペアリングでデバイスが登録された際にに通知されるイベントを登録します。
    void BindPairingCompletedEvent(nn::os::SystemEventType* pSystemEvent) NN_NOEXCEPT;

    //!< 新たに有線ペアリングでデバイスが登録された際にに通知されるイベントを解除します。
    void UnbindPairingCompletedEvent(nn::os::SystemEventType* pSystemEvent) NN_NOEXCEPT;

    //!< ペアリングで登録されたデバイスを列挙します
    size_t GetRegisteredDevices(RegisteredDeviceInfo* pOutValues, size_t length) NN_NOEXCEPT;

    //!< ペアリング情報を手動で追加します
    void AddRegisteredDeviceInfo(const RegisteredDeviceInfo& info) NN_NOEXCEPT;

    //!< ジョイコンのレール部の通信を有効にする
    void SetNwcpEnabled(bool enabled) NN_NOEXCEPT;

    //!< ジョイコンのレール部の接続状態に変化があった場合に通知するイベントを登録します
    void SetRailUpdateEvent(nn::os::SystemEventType* pEvent) NN_NOEXCEPT;

    //!< ジョイコンのレール部の接続状態に変化について、変化の内容を取得します
    RailUpdateEventInfo GetRailUpdateEventInfo() NN_NOEXCEPT;

    //!< レールにデバイスが接続されているかどうかを取得します
    bool IsDeviceAttachedOnRail(RailType type) NN_NOEXCEPT;

    //!< 左レール部のスリープ復帰要因を取得します
    AwakeTriggerReason GetAwakeTriggerReasonForLeftRail() NN_NOEXCEPT;

    //!< 右レール部のスリープ復帰要因を取得します
    AwakeTriggerReason GetAwakeTriggerReasonForRightRail() NN_NOEXCEPT;

    //!< 指定された Usb デバイスをサポートしているか評価します
    bool IsUsbHidSupported(UsbHidDeviceInfo deviceInfo) NN_NOEXCEPT;

    //!< USB デバイスを追加します
    Result AddUsbHidDevice(int index, UsbHidDeviceInfo deviceInfo) NN_NOEXCEPT;

    //!< USB デバイスを削除します
    Result RemoveUsbHidDevice(int index) NN_NOEXCEPT;

    //!< Input Report を処理します
    Result SetUsbHidInputReport(int index, uint8_t *pBuffer, size_t length) NN_NOEXCEPT;

    //!< Input Report を取得します
    size_t GetUsbHidOutputReport(int index, uint8_t *pOutBuffer, size_t length) NN_NOEXCEPT;

    //!< 有線デバイス登録を有効にします
    void SetWiredDeviceActivationEnabled(bool enabled) NN_NOEXCEPT;

    //!< Pro Controller 有線 USB 通信機能を有効にします
    void SetFullKeyUsbEnabled(bool enabled) NN_NOEXCEPT;

    //!< Pro Controller 有線 USB 通信機能の状態を取得します
    void GetFullKeyUsbEnabled(bool* pOutEnabled) NN_NOEXCEPT;

    //!< デバイスの USB 接続状態を取得します
    void GetUsbConnected(bool* pOutValue, DeviceHandle handle) NN_NOEXCEPT;

    //!< 指定したデバイスに接続する
    ::nn::Result SendConnectionTrigger(const ::nn::bluetooth::Address address) NN_NOEXCEPT;

    //!< デバイスへの接続処理がタイムアウトした場合に通知を受けるシステムイベントを登録する
    void SetConnectionTriggerTimeoutEvent(nn::os::SystemEventType* pSystemEvent) NN_NOEXCEPT;

    //!< Bluetooth の接続可能最大数を取得します
    int GetMaxBluetoothLinks() NN_NOEXCEPT;

    //!< ファームウェア更新中のデバイスを指定します
    Result SetFirmwareUpdatingDevice(DeviceHandle handle) NN_NOEXCEPT;

    //!< ファームウェア更新中のデバイス指定を解除します
    void UnsetFirmwareUpdatingDevice() NN_NOEXCEPT;

    //!< Ble を無効化します
    void DisableBle() NN_NOEXCEPT;

    //!< 全ての Palma のスキャンの有効無効を切り替える
    void SetIsPalmaAllConnectable(bool connectable) NN_NOEXCEPT;

    //!< 登録済みの Palma のスキャンの有効無効を切り替える
    void SetIsPalmaPairedConnectable(bool connectable) NN_NOEXCEPT;

    //!< 指定されたデバイスとペアリングする
    Result PairPalma(const nn::bluetooth::Address& address) NN_NOEXCEPT;

    //!< Fifty が接続されているかどうかを取得します
    bool IsFiftyConnected() NN_NOEXCEPT;

    //!< Fifty の入力状態を取得します
    PadState GetFiftyPadState() NN_NOEXCEPT;

    //!< Kuina の Fw バージョンを取得します
    Result GetKuinaVersion(KuinaVersionData* pOutMcuVersionData, int index) NN_NOEXCEPT;

    //!< Kuina の Fw バージョンを要求します
    Result RequestKuinaVersion(int index) NN_NOEXCEPT;

    //!< Kuina の Fw 更新モードに遷移させます
    Result SetKuinaToFirmwareUpdateMode(int index) NN_NOEXCEPT;

private:
    //!< デバイスリストが更新されたときに行う処理
    void HandleDeviceListUpdateEvent() NN_NOEXCEPT;

    //!< デバイスリストが更新されたときに行う処理
    void HandleBleDeviceListUpdateEvent() NN_NOEXCEPT;

    //!< 指定したデバイスに接続するための内部実装
    ::nn::Result TrySendConnectionTriggerImpl() NN_NOEXCEPT;

    //!< ペアリング処理を再度トリガーする
    void TriggerPairingOnAllDevice() NN_NOEXCEPT;
};

}} // namespace nn::xcd
