﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <algorithm>

namespace nn { namespace xcd {

//!< Commandの種類
enum CommandType
{
    CommandType_Out,
    CommandType_In,
};

//!< Commandの内容
struct CommandDescryptor
{
    uint8_t     Id;       //!< Command ID
    size_t      Size;     //!< コマンドのサイズ
    CommandType Type;     //!< コマンドの修理
    int         Response; //!< Outコマンドに続くInコマンドのCommand Id
};

// XCDでサポートされているHID Command 一覧
const CommandDescryptor Command_Ack                        = {0x80, 1, CommandType_In, 0};
const CommandDescryptor Command_PairingIn                  = {0x81, 37, CommandType_In, 0};
const CommandDescryptor Command_DeviceInfo                 = {0x82, 11, CommandType_In, 0};
const CommandDescryptor Command_LRButtonElapsedTime        = {0x83, 8, CommandType_In, 0};
const CommandDescryptor Command_SerialFlashData            = {0x90, 37, CommandType_In, 0};
const CommandDescryptor Command_McuData                    = {0xa0, 37, CommandType_In, 0};
const CommandDescryptor Command_AttachmentData             = {0xa8, 37, CommandType_In, 0};
const CommandDescryptor Command_IndicatorLed               = {0xb0, 1, CommandType_In, 0};
const CommandDescryptor Command_SensorData                 = {0xc0, 37, CommandType_In, 0};
const CommandDescryptor Command_ExtDevRead                 = {0xd8, 32, CommandType_In, 0 };
const CommandDescryptor Command_ExtDevInfo                 = {0xd9, 32, CommandType_In, 0};

const CommandDescryptor Command_PairingOut                 = {0x01, 38, CommandType_Out, Command_Ack.Id};
const CommandDescryptor Command_GetDeviceInfo              = {0x02, 0,  CommandType_Out, Command_PairingIn.Id};
const CommandDescryptor Command_SetDataFormat              = {0x03, 1,  CommandType_Out, Command_Ack.Id};
const CommandDescryptor Command_LRButtonDetection          = {0x04, 0,  CommandType_Out, Command_LRButtonElapsedTime.Id};
const CommandDescryptor Command_Page                       = {0x05, 0,  CommandType_Out, Command_Ack.Id};
const CommandDescryptor Command_Reset                      = {0x06, 1,  CommandType_Out, Command_Ack.Id};
const CommandDescryptor Command_Shipment                   = {0x08, 1,  CommandType_Out, Command_Ack.Id};
const CommandDescryptor Command_SerialFlashRead            = {0x10, 5,  CommandType_Out, Command_SerialFlashData.Id};
const CommandDescryptor Command_SerialFlashWrite           = {0x11, 39, CommandType_Out, Command_Ack.Id};
const CommandDescryptor Command_SerialFlashSectorErase     = {0x12, 4,  CommandType_Out, Command_Ack.Id};
const CommandDescryptor Command_McuReset                   = {0x20, 0,  CommandType_Out, Command_Ack.Id};
const CommandDescryptor Command_McuWrite                   = {0x21, 38, CommandType_Out, Command_McuData.Id};
const CommandDescryptor Command_McuResume                  = {0x22, 1,  CommandType_Out, Command_Ack.Id};
const CommandDescryptor Command_McuPollingEnable           = {0x24, 38, CommandType_Out, Command_Ack.Id};
const CommandDescryptor Command_McuPollingDisable          = {0x25, 0,  CommandType_Out, Command_Ack.Id};
const CommandDescryptor Command_AttachmentWrite            = {0x28, 39, CommandType_Out, Command_AttachmentData.Id};
const CommandDescryptor Command_AttachmentEnable           = {0x2a, 1,  CommandType_Out, Command_Ack.Id };
const CommandDescryptor Command_SetIndicatorLed            = {0x30, 1,  CommandType_Out, Command_Ack.Id};
const CommandDescryptor Command_GetIndicatorLed            = {0x31, 0,  CommandType_Out, Command_IndicatorLed.Id};
const CommandDescryptor Command_SensorSleep                = {0x40, 1,  CommandType_Out, Command_Ack.Id};
const CommandDescryptor Command_SensorConfig               = {0x41, 2,  CommandType_Out, Command_Ack.Id};
const CommandDescryptor Command_SensorWrite                = {0x42, 39, CommandType_Out, Command_Ack.Id};
const CommandDescryptor Command_SensorRead                 = {0x43, 2,  CommandType_Out, Command_SensorData.Id};
const CommandDescryptor Command_MotorEnable                = {0x48, 1,  CommandType_Out, Command_Ack.Id};
const CommandDescryptor Command_GetBatteryVoltage          = {0x50, 1,  CommandType_Out, Command_Ack.Id};
const CommandDescryptor Command_WriteChargeSetting         = {0x51, 1,  CommandType_Out, Command_Ack.Id};
const CommandDescryptor Command_ExtDevWrite                = {0x58, 38, CommandType_Out, Command_ExtDevRead.Id };
const CommandDescryptor Command_GetExtDevInfo              = {0x59, 0,  CommandType_Out, Command_ExtDevInfo.Id };
const CommandDescryptor Command_ExtDevPollingEnable        = {0x5a, 38, CommandType_Out, Command_Ack.Id };
const CommandDescryptor Command_ExtDevPollingDisable       = {0x5b,  0, CommandType_Out, Command_Ack.Id };
const CommandDescryptor Command_ExtDevInFormatConfig       = {0x5c, 38, CommandType_Out, Command_Ack.Id };

const uint16_t CommandByte_Base                     = 0;                        //!< 基準となるバイト
const uint16_t CommandByte_CommandId                = CommandByte_Base;         //!< コマンドID

// CommandIn のデータフォーマットに関する定義
const uint16_t CommandInByte_OutputId               = CommandByte_Base + 1;     //!< 送信レポートのID
const uint16_t CommandInByte_Payload                = CommandByte_Base + 2;     //!< ペイロード
const uint16_t CommandInSize_Payload                = 35;                       //!< ペイロードのサイズ

// CommandOut のデータフォーマットに関する定義
const uint16_t CommandOutByte_Payload               = CommandByte_Base + 1;     //!< ペイロード
const uint16_t CommandOutSize_Payload               = 38;                       //!< ペイロードのサイズ

// Pairing に関する定義
const uint16_t PairingOutSize_Data                   = 38;                       //!< ペイロードのサイズ
const uint16_t PairingOutByte_Payload                = CommandOutByte_Payload;   //!< ペイロード
const uint16_t PairingByteOffset_Status              = 0;                        //!< Status
const uint16_t PairingByteOffset_BdAddr              = 1;                        //!< Bluetooth アドレス
const uint16_t PairingSize_BdAddr                    = 6;                        //!< Bluetooth アドレスのサイズ
const uint16_t PairingByteOffset_ClassOfDevice       = 7;                        //!< Class Of Device
const uint16_t PairingSize_ClassOfDevice             = 3;                        //!< Class Of Device のサイズ
const uint16_t PairingByteOffset_DeviceName          = 10;                       //!< デバイスネーム
const uint16_t PairingSize_DeviceName                = 20;                       //!< デバイスネームの最大長
const uint16_t PairingByteOffset_FeatureSet          = 30;                       //!< Feature Set
const uint16_t PairingByteOffset_LinkKey             = 1;                        //!< LinkKey
enum PairingStatus : uint8_t
{
    PairingStatus_Initiate = 0x01,                                                 //!< ペアリング処理の開始
    PairingStatus_LinkKey = 0x02,                                                  //!< ペアリング処理の開始
    PairingStatus_Ack = 0x03,                                                      //!< リンクキー受領に対する Ack
    PairingStatus_InitiateWithPairedDevice = 0x04,                                 //!< 登録済みのデバイスとペアリング処理を開始
};

// SetDataFormat に関する定義
const uint16_t GetDeviceInfoByte_ReportId           = CommandOutByte_Payload;   //!< ReportId

// Reset に関する定義
const uint16_t ResetByte_Page                       = CommandOutByte_Payload;   //!< Reset
enum ResetPageType
{
    ResetPageType_Disabled = 0,                                                 //!< Reset後のPageは行わない
    ResetPageType_Enabled,                                                      //!< Reset後のPageを行う
};

// SerialFlashRead に関する定義
const uint16_t SerialFlashReadByte_Address          = CommandOutByte_Payload;   //!< 読み出すデータのアドレス
const uint16_t SerialFlashReadByte_Size             = CommandOutByte_Payload + 4; //!< 読み出すデータのサイズ
const uint16_t SerialFlashReadSize_Data             = 0x1d;                       //!< 読み出せるデータの最大長

// SerialFlashWrite に関する定義
const uint16_t SerialFlashWriteByte_Address          = CommandOutByte_Payload;   //!< 書き込むデータのアドレス
const uint16_t SerialFlashWriteByte_Size             = CommandOutByte_Payload + 4; //!< 書き込むデータのサイズ
const uint16_t SerialFlashWriteByte_Data             = CommandOutByte_Payload + 5; //!< 書き込むデータ
const uint16_t SerialFlashWriteSize_Data             = std::min(static_cast<uint16_t>(0x21), SerialFlashReadSize_Data);  //!< 書き込めるデータの最大長。ベリファイを行うため読み込みサイズのほうが小さい場合はそちらにリミットされる

// SerialFlashSectorErase に関する定義
const uint16_t SerialFlashSectorEraseByte_Address    = CommandOutByte_Payload;   //!< イレースするセクタデータのアドレス

// MCUWrite に関する定義
const uint16_t McuResumeByte_Value                   = CommandOutByte_Payload;   //!< MCUのResume/Suspend
enum McuResumeValueType
{
    McuResumeValueType_Suspend = 0,
    McuResumeValueType_Resume = 1,
    McuResumeValueType_ResumeForUpdate = 2
};

// MCUWrite に関する定義
const uint16_t McuWriteByte_Data                     = CommandOutByte_Payload;   //!< MCUに送るデータ
const uint16_t McuWriteSize_Data                     = 38;                       //!< MCUに送るデータのサイズ

// AttachMentEnable に関する定義
enum AttachmentEnableType
{
    AttachmentEnableType_Disable = 0,
    AttachmentEnableType_Enable  = 1,
};

// ChargeEnable に関する定義
enum ChargeEnableType
{
    ChargeEnableType_Disable = 0x00,
    ChargeEnableType_Enable = 0x04,
};

// ExtDevInfo に関する定義
const uint16_t ExtDevInfoOffset_DeviceType           = CommandOutByte_Payload + 2; //!< DeviceType のアドレス

// ExtDevRead に関する定義
const uint16_t ExtDevReadOffset_Status               = CommandOutByte_Payload + 1; //!< Status のアドレス
const uint16_t ExtDevReadOffset_Size                 = CommandOutByte_Payload + 2; //!< Size のアドレス
const uint16_t ExtDevReadOffset_Data                 = CommandOutByte_Payload + 3; //!< Data のアドレス
enum ExtDevReadStatus
{
    ExtDevReadStatus_Succsess         = 0x00,
    ExtDevReadStatus_CrcMismatch      = 0xfd,
    ExtDevReadStatus_NoExtDevAttached = 0xfe,
    ExtDevReadStatus_Timeout          = 0xff,
};

// SetIndicatorに関する定義
const uint16_t SetIndicatorByte_Pattern              = CommandOutByte_Payload;   //!< LEDの点灯パターン

// SensorSleepに関する定義
const uint16_t SensorSleepByte_Value                 = CommandOutByte_Payload;   //!< センサーのスリープ
enum SensorSleepValueType
{
    SensorSleepValueType_Sleep = 0,
    SensorSleepValueType_Active = 1,
    SensorSleepValueType_Dscale1 = 2,
    SensorSleepValueType_Dscale1Payload6 = 3,
    SensorSleepValueType_Dscale1Payload12 = 4,
    SensorSleepValueType_Dscale2 = 5,
};

// SensorConfigに関する定義
const uint16_t SensorConfigByte_Gyroscope            = CommandOutByte_Payload;   //!< ジャイロセンサーの設定
const uint16_t SensorConfigByte_Accelerometer        = CommandOutByte_Payload + 1;//!< 加速度センサーの設定
enum SensorConfigGyroscopeType
{
    SensorConfigGyroscopeType_250dps  = 0,                                         //!< FSR 250dps
    SensorConfigGyroscopeType_500dps  = 1,                                         //!< FSR 500dps
    SensorConfigGyroscopeType_1000dps = 2,                                         //!< FSR 1000dps
    SensorConfigGyroscopeType_2000dps = 3,                                         //!< FSR 2000dps
};

enum SensorConfigAccelerometerType
{
    SensorConfigAccelerometerType_8G = 0,                                         //!< FSR 8G
    SensorConfigAccelerometerType_4G = 1,                                         //!< FSR 4G
    SensorConfigAccelerometerType_2G = 2,                                         //!< FSR 2G
};

// SensorWriteに関する定義
const uint16_t SensorWriteByte_Register               = CommandOutByte_Payload;   //!< センサーのレジスタ
const uint16_t SensorWriteByte_Size                   = CommandOutByte_Payload + 1; //!< センサーに書き込む値のサイズ
const uint16_t SensorWriteByte_Data                   = CommandOutByte_Payload + 2; //!< センサーに書き込む値
const uint16_t SensorWriteDataSizeMax                 = 0x24;

// SensorReadに関する定義
const uint16_t SensorReadByte_Register                = CommandOutByte_Payload;   //!< センサーのレジスタ
const uint16_t SensorReadByte_Size                    = CommandOutByte_Payload + 1; //!< センサーから読み出す値のサイズ
const uint16_t SensorReadDataSizeMax                  = 0x21;

// MotorEnable に関する定義
const uint16_t MotorEnableByte_Value                  = CommandOutByte_Payload;   //!< MotorのEnable/Disable
enum MotorEnableValueType
{
    MotorEnableValueType_Disable = 0,
    MotorEnableValueType_Enable = 1,
};

// Shipment に関する定義
const uint16_t ShipmentEnableByte_Value               = CommandOutByte_Payload;   //!< Shipment のEnable/Disable
enum ShipmentEnableValueType
{
    ShipmentEnableValueType_Disable = 0,
    ShipmentEnableValueType_Enable = 1,
};

// Ack に関連する定義
const uint16_t AckByte_Result                       = CommandInByte_Payload;   //!< Firmwareバージョン
enum ActResultType
{
    AckResultType_Success = 0,                                                      //!< 成功
    AckResultType_Fail,                                                             //!< 失敗 (汎用)
    AckResultType_NoDevicePaired,                                                   //!< ペアリング済みのデバイスがない
    AckResultType_NotSupported,                                                     //!< サポートされていないHID Command
};

// Pairing に関する定義
const uint16_t PairingInSize_Payload                = 34;                            //!< ペイロード
const uint16_t PairingInByte_Payload                = CommandInByte_Payload;         //!< ペイロード

// DeviceInfo に関連する定義
const uint16_t DeviceInfoByte_FwVersion         = CommandInByte_Payload;           //!< Firmwareバージョン
const uint16_t DeviceInfoByte_DeviceType        = CommandInByte_Payload + 2;       //!< デバイスの種類
const uint16_t DeviceInfoByte_BoardRevision     = CommandInByte_Payload + 3;       //!< ボードリビジョン
const uint16_t DeviceInfoByte_BdAddr            = CommandInByte_Payload + 4;       //!< BDADDR
const uint16_t DeviceInfoByte_SensorType        = CommandInByte_Payload + 10;       //!< センサーの種類
const uint16_t DeviceInfoByte_FlashFormatVersion = CommandInByte_Payload + 11;      //!< シリアルフラッシュのフォーマットバージョン
enum DeviceInfoDeviceType
{
    DeviceInfoDeviceType_Reserved = 0,                                           //!< Reserved
    DeviceInfoDeviceType_LeftJoy,                                                //!< DeviceType: 左ジョイコン
    DeviceInfoDeviceType_RightJoy,                                               //!< DeviceType: 右ジョイコン
    DeviceInfoDeviceType_Fullkey,                                                //!< DeviceType: フルキーコン
    DeviceInfoDeviceType_MiyabiLeft,                                             //!< DeviceType: USB 左ジョイコン
    DeviceInfoDeviceType_MiyabiRight,                                            //!< DeviceType: USB 右ジョイコン
    DeviceInfoDeviceType_Tarragon,                                               //!< DeviceType: フルキーコン
};
enum DeviceInfoSensorType
{
    DeviceInfoSensorType_Reserved = 0,                                           //!< Reserved
    DeviceInfoSensorType_LSM6DS3,                                                //!< SensorType: LSM6DS3
    DeviceInfoSensorType_BMI160,                                                 //!< SensorType: BMI160
    DeviceInfoSensorType_ICM20630,                                               //!< SensorType: ICM20630
};

// LRButtonElapsedTime に関連する定義
const uint16_t LRButtonElapsedTimeByte_L             = CommandInByte_Payload;     //!< Elapsed time of L
const uint16_t LRButtonElapsedTimeByte_R             = CommandInByte_Payload + 2; //!< Elapsed time of R
const uint16_t LRButtonElapsedTimeByte_ZL            = CommandInByte_Payload + 4; //!< Elapsed time of ZL
const uint16_t LRButtonElapsedTimeByte_ZR            = CommandInByte_Payload + 6; //!< Elapsed time of ZR

// SerialFlashData に関連する定義
const uint16_t SerialFlashDataByte_Address           = CommandInByte_Payload;     //!< 読み出したシリアルフラッシュのアドレス
const uint16_t SerialFlashDataByte_Size              = CommandInByte_Payload + 4; //!< 読み出したシリアルフラッシュのデータのサイズ
const uint16_t SerialFlashDataByte_Data              = CommandInByte_Payload + 5; //!< 読み出したシリアルフラッシュのデータ

// SensorData に関連する定義
const uint16_t SensorDataByte_Registor               = CommandInByte_Payload;     //!< 読み出したセンサーのレジスター
const uint16_t SensorDataByte_Size                   = CommandInByte_Payload + 1; //!< 読み出したセンサーのデータのサイズ
const uint16_t SensorDataByte_Data                   = CommandInByte_Payload + 2; //!< 読み出したセンサーのデータ

// McuRead に関連する定義
const uint16_t McuReadByte_Data                      = CommandInByte_Payload;     //!< 読み出したMCUのデータ
const uint16_t McuReadSize_Data                      = 34;                        //!< 読み出したMCUのデータのサイズ

// IndicatorLed に関連する定義
const uint16_t IndicatorLedByte_Pattern              = CommandInByte_Payload;     //!< インジケータLEDの点灯パターン


}} // namespace nn::xcd
