﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <vector>
#include <nn/nn_Result.h>
#include <nn/nn_Macro.h>
#include <nn/xcd/xcd_Input.h>
#include <nn/xcd/xcd_DataFormat.h>
#include <nn/xcd/xcd_Pairing.h>
#include "xcd_CommandTypes.h"
#include "xcd_ICommandListener.h"

namespace nn { namespace xcd {

const int MaxCommandCount = 32;
const int MaxCompareSequenceCount = 10;

//!< HID Commandを制御するためのクラスです。
class CommandHandler
{
private:
    enum CommandStatus
    {
        CommandStatus_Empty,             //!< Commandがキューない状態
        CommandStatus_Queued,            //!< Commandがキューにセットされた状態
        CommandStatus_Sent,              //!< Commandが送信されInputを待っている状態
        CommandStatus_Completed,         //!< Commandの送信が完了した状態
    };

    struct HidCommand
    {
        CommandStatus status;
        uint8_t commandId;
        uint8_t payload[CommandOutSize_Payload];
        ICommandListener* pListener;
        bool isSerialFlashVerify;
        bool isRawSerialFlashRead;
        bool isDuplicatable;
    };

    // HidCommand を格納するキュー
    HidCommand m_CommandQueue[MaxCommandCount];
    int m_CommandCount;

    //!< 次にコマンドを追加する Index
    int m_CommandIndex;

    //!< 送信中のコマンドの Index
    int m_SentCommandIndex;

    //!< Busyの期間(送信スロット単位)
    uint16_t m_Busy;

    //!< 有線ペアリング中かどうか
    bool m_IsOnWiredPairing;

    //!< 有線ペアリング済みの機器との再ペアリングかどうか
    bool m_IsPairingWithRegisteredDevice;

    //!< 有線ペアリングデバイスの情報
    BluetoothDeviceInfo m_PairingDeviceInfo;

    //!< SerialFlashWrite 用のバッファ
    uint8_t m_BufferForSerialFlashWrite[SerialFlashWriteSize_Data + 5];

    //!< SerialFlashWrite 中かどうか
    bool m_IsOnSerialFlashWrite;

    //!< SerialFlashWrite の書き込みが任意のアドレスに対する操作かどうか
    bool m_IsOnSerialFlashManualWrite;

    //!< IndicatorPattern
    uint8_t m_IndicatorPattern;

    bool m_Activated;

    FirmwareVersionUnit m_BluetoothFirmwareVersion;
public:

    CommandHandler() NN_NOEXCEPT;
    ~CommandHandler() NN_NOEXCEPT;

    void ParseInputReport(const uint8_t* pBuffer, size_t size) NN_NOEXCEPT;
    size_t GetOutputReport(uint8_t* pOutValue, size_t size) NN_NOEXCEPT;

    void Activate() NN_NOEXCEPT;
    void Deactivate() NN_NOEXCEPT;

    /**
    * @brief  コントローラーと有線ペアリングを行います。有線コントローラーのみ有効です。
    */
    void Pairing(const BluetoothDeviceInfo& deviceInfo, ICommandListener* pListener, bool pairingInfoExist) NN_NOEXCEPT;
    /**
    * @brief  デバイスの情報を取得します。ICommandListener::NotifyDeviceInfoで通知されます。pListenerは必須です。
    */
    void GetDeviceInfo(ICommandListener* pListener) NN_NOEXCEPT;
    /**
    * @brief  データフォーマットの変更。ICommandListener::NotifyAckで通知されます。
    */
    void SetDataFormat(PeriodicDataFormat format, ICommandListener* pListener) NN_NOEXCEPT;
    /*
    * @brief  デバイスを再起動します。
    */
    void Reset(bool page) NN_NOEXCEPT;
    /**
    * @brief  McuのReset。ICommandListener::NotifyAckで通知されます。
    */
    void McuReset(ICommandListener* pListener) NN_NOEXCEPT;
    /**
    * @brief  Mcuへのone shotのWrite/Read。ICommandListener::NotifyMcuReadで通知されます。pListenerは必須です。
    */
    void McuWrite(uint8_t* pBuffer, size_t size, ICommandListener* pListener) NN_NOEXCEPT;
    /**
    * @brief  Mcuへのone shotのWrite/Read のコマンド重複禁止版。ICommandListener::NotifyMcuReadで通知されます。pListenerは必須です。
    */
    void McuWriteWithoutDuplication(uint8_t* pBuffer, size_t size, ICommandListener* pListener) NN_NOEXCEPT;
    /**
    * @brief  McuのResume/Suspendの切り替え。ICommandListener::NotifyAckで通知されます。
    */
    void McuResume(McuResumeValueType resumeType, ICommandListener* pListener) NN_NOEXCEPT;
    /**
    * @brief  McuへのPollingモードを有効化。指定したコマンドをPollingします。ICommandListener::NotifyAckで通知されます。
    */
    void McuPollingEnable(uint8_t* pBuffer, size_t size, ICommandListener* pListener) NN_NOEXCEPT;
    /**
    * @brief  McuへのPollingモードを無効化。ICommandListener::NotifyAckで通知されます。
    */
    void McuPollingDisable(ICommandListener* pListener) NN_NOEXCEPT;
    /**
    * @brief  Attachment の有効化。ICommandListener::NotifyAckで通知されます。
    */
    void AttachmentEnable(bool enabled, ICommandListener* pListener) NN_NOEXCEPT;

    /**
    * @brief  拡張デバイスの情報を取得します。ICommandListener::NotifyExtDevInfo で通知されます。pListenerは必須です。
    */
    void GetExtDevInfo(ICommandListener* pListener) NN_NOEXCEPT;

    /**
    * @brief  拡張デバイスへコマンドを送信します。ICommandListener::NotifySendCommandToAttachmentDevice で通知されます。pListenerは必須です。
    */
    void SendCommandToAttachmentDevice(const uint8_t* pInData, size_t size, ICommandListener* pListener) NN_NOEXCEPT;

    /**
    * @brief  Attachment Device 向けに 受信データを Basic In にどうマップするか設定します。ICommandListener::NotifyAckで通知されます。
    */
    void ConfigBasicInFormatForAttachment(const uint8_t* pInData, size_t size, ICommandListener* pListener) NN_NOEXCEPT;

    /**
    * @brief  Attachment Device の Polling Mode の有効化。ICommandListener::NotifyAckで通知されます。
    */
    void EnablePollingReceiveModeForAttachmentDevice(const uint8_t* pInData, size_t size, ICommandListener* pListener) NN_NOEXCEPT;

    /**
    * @brief  Attachment Device の Polling Mode の無効化。ICommandListener::NotifyAckで通知されます。
    */
    void DisablePollingReceiveModeForAttachmentDevice(ICommandListener* pListener) NN_NOEXCEPT;

    /**
    * @brief  LEDを制御します。ICommandListener::NotifyAckで通知されます。
    */
    void SetIndicatorLed(uint8_t pattern, ICommandListener* pListener) NN_NOEXCEPT;

    /**
    * @brief  センサーのスリープ制御。ICommandListener::NotifyAckで通知されます。
    */
    void SensorSleep(bool sleep, ICommandListener* pListener) NN_NOEXCEPT;
    /**
    * @brief  センサーの設定変更。ICommandListener::NotifyAckで通知されます。
    */
    void SensorConfig(AccelerometerFsr accelerometerFsr, GyroscopeFsr gyroscopeFsr, ICommandListener* pListener) NN_NOEXCEPT;
    /**
    * @brief  センサーのペイロード領域制御。ICommandListener::NotifyAckで通知されます。
    */
    void SensorPayload(bool enable, ICommandListener* pListener) NN_NOEXCEPT;

    /**
    * @brief  モーターのEnable/Disable切り替え。ICommandListener::NotifyAckで通知されます。
    */
    void MotorEnable(bool enabled, ICommandListener* pListener) NN_NOEXCEPT;

    /**
    * @brief  充電の設定変更。ICommandListener::NotifyAckで通知されます。
    */
    void WriteChargeSetting(bool chargeEnabled, ICommandListener* pListener) NN_NOEXCEPT;

    /**
    * @brief  Shipment Mode の切り替え。 ICommandListener::NotifyAck で通知されます。
    */
    void SetShipment(bool enabled, ICommandListener* pListener) NN_NOEXCEPT;

    /*
    * @brief  ボタンが押下されてから経過した時間を取得します。
    */
    void GetButtonTriggerElapsedTime(ICommandListener* pListener) NN_NOEXCEPT;

    /**
    * @brief  シリアルナンバーを読み出します
    */
    void ReadIdentificationCode(ICommandListener* pListener) NN_NOEXCEPT;

    /**
    * @brief  6軸センサーのユーザー CAL を更新します
    */
    Result UpdateSixAxisSensorUserCal(const SensorCalibrationValue& value, ICommandListener* pListener) NN_NOEXCEPT;

    /**
    * @brief  6軸センサーのユーザー CAL を破棄します
    */
    Result DeleteSixAxisSensorUserCal(ICommandListener* pListener) NN_NOEXCEPT;

    /**
    * @brief  左アナログスティックのユーザー CAL を更新します
    */
    Result UpdateLeftAnalogStickUserCal(const AnalogStickValidRange& value, ICommandListener* pListener) NN_NOEXCEPT;

    /**
    * @brief  左アナログスティックのユーザー CAL を破棄します
    */
    Result DeleteLeftAnalogStickUserCal(ICommandListener* pListener) NN_NOEXCEPT;

    /**
    * @brief  右アナログスティックのユーザー CAL を更新します
    */
    Result UpdateRightAnalogStickUserCal(const AnalogStickValidRange& value, ICommandListener* pListener) NN_NOEXCEPT;

    /**
    * @brief  右アナログスティックのユーザー CAL を破棄します
    */
    Result DeleteRightAnalogStickUserCal(ICommandListener* pListener) NN_NOEXCEPT;

    /**
    * @brief  CAL1 領域(センサー)
    */
    void ReadCal1(ICommandListener* pListener) NN_NOEXCEPT;

    /**
    * @brief  CAL2 領域(スティックの CAL 値, 色情報)
    */
    void ReadCal2(ICommandListener* pListener) NN_NOEXCEPT;

    /**
    * @brief デザイン情報 領域(色情報)
    */
    void ReadDesign(ICommandListener* pListener) NN_NOEXCEPT;

    /**
    * @brief  MODEL1 領域(センサー, 左スティックのモデル値)
    */
    void ReadModel1(ICommandListener* pListener) NN_NOEXCEPT;

    /**
    * @brief  MODEL2 領域(右スティックのモデル値)
    */
    void ReadModel2(ICommandListener* pListener) NN_NOEXCEPT;

    /**
    * @brief  USERCAL1 領域(スティックのCAL値 + センサーのマジックナンバー)
    */
    void ReadUserCal1(ICommandListener* pListener) NN_NOEXCEPT;

    /**
    * @brief  USERCAL2 領域(センサーの CAL 値)
    */
    void ReadUserCal2(ICommandListener* pListener) NN_NOEXCEPT;

    /**
    * @brief  コントローラーの色情報を更新します
    */
    Result UpdateControllerColor(const nn::util::Color4u8Type& mainColor, const nn::util::Color4u8Type& subColor, ICommandListener* pListener) NN_NOEXCEPT;

    /**
    * @brief  コントローラーのデザイン情報全体を更新します。
    */
    Result UpdateDesignInfo(const DeviceColor& color, uint8_t variation, ICommandListener* pListener) NN_NOEXCEPT;

    /**
    * @brief  シリアルフラッシュのフォーマットバージョンを更新します。
    */
    Result UpdateFormatVersion(uint8_t version, ICommandListener* pListener) NN_NOEXCEPT;

    /**
    * @brief  シリアルフラッシュのへの読み込み最大長を取得します
    */
    static int GetSerialFlashReadUnitSize() NN_NOEXCEPT;

    /**
    * @brief  シリアルフラッシュのへの書き込み最大長を取得します
    */
    static int GetSerialFlashWriteUnitSize() NN_NOEXCEPT;

    /**
    * @brief  シリアルフラッシュの任意のアドレスから読み込みを行います
    */
    void ReadSerialFlash(const uint32_t address, uint8_t size, ICommandListener* pListener) NN_NOEXCEPT;

    /**
    * @brief  シリアルフラッシュの任意のアドレスへの書き込みを行います
    */
    Result WriteSerialFlash(const uint32_t address, const uint8_t* pBuffer, uint8_t size, ICommandListener* pListener) NN_NOEXCEPT;

private:
    //!< Hid Command がキューの末尾と重複しているか取得する
    bool IsSameAsTailCommand(CommandDescryptor command, uint8_t* payload, ICommandListener* pListener) NN_NOEXCEPT;

    //!< Hid Command の重複をキューの末尾から検索し、indexを返す (指定したサイズまで検索を続け、見つからない場合は -1 を返す)
    int FindSameCommandFromTail(CommandDescryptor command, uint8_t* payload, ICommandListener* pListener, uint8_t compareSizeMax) NN_NOEXCEPT;

    //!< Hid Command のシーケンスの重複をキューの末尾から検索し、重複した(削除可能な)シーケンスサイズを返す。 (見つからない場合は -1 を返す)
    int FindSameSequenceFromTail(CommandDescryptor command, uint8_t* payload, ICommandListener* pListener, uint8_t compareSizeMax) NN_NOEXCEPT;

    //!< Hid Commandをキューに追加する
    Result PushCommandToQueue(CommandDescryptor command, uint8_t* payload, ICommandListener* pListener) NN_NOEXCEPT;

    //!< Hid Commandをキューに重複禁止コマンドとして追加する
    Result PushCommandToQueueWithoutDuplication(CommandDescryptor command, uint8_t* payload, ICommandListener* pListener) NN_NOEXCEPT;

    //!< Hid Commandをキューに追加する (Serial Flash Read 用)
    Result PushCommandToQueueSerialFlashRead(CommandDescryptor command, uint8_t* payload, ICommandListener* pListener, bool isVerify, bool isDuplicatable, bool isRawAccess) NN_NOEXCEPT;

    //!< Command キューを空にする
    void ClearCommandToQueue() NN_NOEXCEPT;

    //!< Command キューを末尾から指定した数分削除する
    void RemoveCommandFromTail(int commandCount) NN_NOEXCEPT;

    //!< Serial FlashのRead
    void ReadSerialFlashImpl(uint32_t address, uint8_t size, ICommandListener* pListener, bool isVerify, bool isRawAccess) NN_NOEXCEPT;

    //!< Serial Flash の Write
    Result PushSerialFlashWriteCommandToQueue(ICommandListener* pListener) NN_NOEXCEPT;

    void HandleAck(const uint8_t* pBuffer, ICommandListener* pListener, uint8_t* pOutputReportBuffer) NN_NOEXCEPT;
    void HandlePairing(const uint8_t* pBuffer, ICommandListener* pListener) NN_NOEXCEPT;
    void HandleDeviceInfo(const uint8_t* pBuffer, ICommandListener* pListener) NN_NOEXCEPT;
    void HandleGetButtonTriggerElapsedTime(const uint8_t* pBuffer, ICommandListener* pListner) NN_NOEXCEPT;
    void HandleMcuRead(const uint8_t* pBuffer, ICommandListener* pListener) NN_NOEXCEPT;
    bool HandleSerialFlashData(const uint8_t* pBuffer, ICommandListener* pListener, bool isVerify, bool isRawSerialFlashRead, uint8_t* pOutputReportBuffer) NN_NOEXCEPT;

    void HandleReadIdentificationCode(const uint8_t* pBuffer, uint32_t baseAddress, ICommandListener* pListener) NN_NOEXCEPT;
    void HandleReadCal1(const uint8_t* pBuffer, uint32_t baseAddress, ICommandListener* pListener) NN_NOEXCEPT;
    void HandleReadCal2(const uint8_t* pBuffer, uint32_t baseAddress, ICommandListener* pListener) NN_NOEXCEPT;
    void HandleReadDesign(const uint8_t* pBuffer, uint32_t baseAddress, ICommandListener* pListener) NN_NOEXCEPT;
    void HandleReadModel1(const uint8_t* pBuffer, uint32_t baseAddress, ICommandListener* pListener) NN_NOEXCEPT;
    void HandleReadModel2(const uint8_t* pBuffer, uint32_t baseAddress, ICommandListener* pListener) NN_NOEXCEPT;
    void HandleReadUserCal1(const uint8_t* pBuffer, uint32_t baseAddress, ICommandListener* pListener) NN_NOEXCEPT;
    void HandleReadUserCal2(const uint8_t* pBuffer, uint32_t baseAddress, ICommandListener* pListener) NN_NOEXCEPT;
    void HandleExtDevInfo(const uint8_t* pBuffer, ICommandListener* pListener) NN_NOEXCEPT;
    void HandleExtDevRead(const uint8_t* pBuffer, ICommandListener* pListener) NN_NOEXCEPT;
};

}} // namespace nn::xcd
