﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <vector>
#include <nn/nn_Result.h>
#include <nn/nn_Macro.h>
#include <nn/os/os_SdkMutex.h>
#include <nn/os/os_Tick.h>
#include <nn/xcd/xcd_Input.h>
#include "xcd_PadInput.h"
#include "xcd_BleReportTypes.h"

namespace nn { namespace xcd {

//!< Ble レポートのフォーマット情報
struct BleReportFormatVersion
{
    BleInputReportFormatVersion inputReportFormatVer;
    BleOutputCommandFormatVersion outputCommandFormatVer;
    BleCommandResponseFormatVersion commandResponseFormatVer;
};

//!< Ble デバイスのボタン, アナログスティック, 6軸センサーを扱うクラス
class BlePadInput
{

private:
    //!< ボタン、アナログスティックの状態
    PadState m_State;

    //!< 6軸センサーの状態
    SixAxisSensorState m_SensorStates[SixAxisSensorSampleCountMax];
    //!< センサーを排他するための Mutex
    ::nn::os::SdkMutex m_SensorMutex;
    //!< バッファ内の現在のインデックス
    int m_SensorBufferIndex;
    //!< バッファの中身のデータ数
    int m_SensorBufferCount;
    //!< センサーデータに付与するサンプル番号
    int64_t m_LastSampleNumberForBuffer;

    //!< 初めてのサンプルかどうか
    bool m_IsFirstSample;

    //!< 前回の InputReport に含まれるサンプル番号の生値
    int64_t m_LastSampleNumberRaw;

    BleReportFormatVersion m_FormatVersion;

public:
    BlePadInput() NN_NOEXCEPT;
    virtual ~BlePadInput() NN_NOEXCEPT;

    //!< デバイスが接続されたときに呼ばれる関数
    void Activate(const BleReportFormatVersion& version) NN_NOEXCEPT;
    //!< 接続直後の初期化処理が完了したかどうかを確認する関数。初期化が完了したら必ずtrueを返してください
    bool IsActivated() NN_NOEXCEPT;
    //!< デバイスが切断された時に呼ばれる関数
    void Deactivate() NN_NOEXCEPT;

    void ParseInputReport(const uint8_t* pBuffer, size_t size) NN_NOEXCEPT;

    //!< 最新のPadの入力を1つ取得する
    Result GetPadState(PadState* pOutValue) NN_NOEXCEPT;

    //!< センサーの入力を取得する
    Result GetSensorStates(int* pOutCount, SixAxisSensorState* pOutValue, int length) NN_NOEXCEPT;

private:
    //!< サンプル数の差分を算出する
    int GetDeltaSampleNumber(uint8_t sampleNumber) NN_NOEXCEPT;

    //!< BleInputReportFormatVersion_JoyRight フォーマットのパースを行う
    void ParseButtonStateJoyRight(const uint8_t* pBuffer, size_t size) NN_NOEXCEPT;
    void ParseAnalogStickJoyRight(const uint8_t* pBuffer, size_t size) NN_NOEXCEPT;
    void ParseSensorDataJoyRight(const uint8_t* pBuffer, size_t size, int samplesSinceLast) NN_NOEXCEPT;

    //!< BleInputReportFormatVersion_Palma フォーマットのパースを行う
    void ParseButtonStatePalma(const uint8_t* pBuffer, size_t size) NN_NOEXCEPT;
    void ParseAnalogStickPalma(const uint8_t* pBuffer, size_t size) NN_NOEXCEPT;
    void ParseSensorDataPalma(const uint8_t* pBuffer, size_t size, int samplesSinceLast) NN_NOEXCEPT;

    void PrintDeviceInformation() NN_NOEXCEPT;
};

}} // namespace nn::xcd
