﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <mutex>
#include <nn/nn_Abort.h>
#include <nn/nn_Macro.h>
#include <nn/nn_Result.h>
#include <nn/nn_SdkAssert.h>
#include <nn/nn_SdkLog.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/util/util_BitPack.h>
#include <nn/xcd/xcd_Input.h>
#include <nn/xcd/xcd_Result.h>
#include <nn/xcd/detail/xcd_Log.h>

#include "xcd_BlePadInput.h"
#include "xcd_ParserUtil.h"

#if 0
#define PAD_DUMP_LOG(...) NN_SDK_LOG(__VA_ARGS__)
#else
#define PAD_DUMP_LOG(...)
#endif

namespace nn { namespace xcd {

namespace
{
    // デフォルト定義一覧
    const uint8_t SampleNumberFromBleDeviceMax = 0xFF;

    // モデル値カウントを FSR = 8G から FSR = 2G の単位系に変換
    //!< アナログスティックの最大値

    inline void ConvertPalmaAxis(nn::xcd::SixAxisSensorState* pState)
    {
        nn::xcd::SixAxisSensorState state = *pState;

        pState->accelerometer.x = state.accelerometer.y;
        pState->gyroscope.x = - state.gyroscope.y;

        pState->accelerometer.y = - state.accelerometer.x;
        pState->gyroscope.y = state.gyroscope.x;

        pState->accelerometer.z = - state.accelerometer.z;
    }

    void DumpStickValue(const AnalogStickState& state)
    {
        NN_UNUSED(state);
        PAD_DUMP_LOG("x:%5d y:%5d", state.x, state.y);
    }

    void DumpSensorValue(const nn::util::Float3& state)
    {
        NN_UNUSED(state);
        PAD_DUMP_LOG("x:%7.1f y:%7.1f z:%7.1f\n", state.x, state.y, state.z);
    }
}

BlePadInput::BlePadInput() NN_NOEXCEPT
    : m_SensorMutex()
    , m_SensorBufferIndex(0)
    , m_SensorBufferCount(0)
    , m_LastSampleNumberForBuffer(0)
    , m_IsFirstSample(true)
    , m_LastSampleNumberRaw(0)
{
    for (auto& state : m_SensorStates)
    {
        state = SixAxisSensorState();
    }
}

BlePadInput::~BlePadInput() NN_NOEXCEPT
{
    // 何もしない
}

void BlePadInput::Activate(const BleReportFormatVersion& version) NN_NOEXCEPT
{
    m_FormatVersion = version;
    m_State.buttons.Reset();
}

bool BlePadInput::IsActivated() NN_NOEXCEPT
{
    // TODO: CAL 値の読み出しが完了していれば、終了
    return true;
}

void BlePadInput::Deactivate() NN_NOEXCEPT
{
    m_SensorBufferIndex = 0;
    m_SensorBufferCount = 0;
    m_LastSampleNumberForBuffer = 0;
    m_IsFirstSample = true;
    m_LastSampleNumberRaw = 0;
}

void BlePadInput::ParseInputReport(const uint8_t* pBuffer, size_t size) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pBuffer);

    switch (m_FormatVersion.inputReportFormatVer)
    {
        case BleInputReportFormatVersion_JoyRight:
        {
            NN_SDK_REQUIRES(size >= BleInputReportFormat_JoyRight_Size, "Invalid size(%d) set", size);
            m_State.sampleNumber = pBuffer[BleInputReportFormat_JoyRight::SampleNumber];
            int samplesSinceLast = GetDeltaSampleNumber(m_State.sampleNumber);
            ParseButtonStateJoyRight(pBuffer, size);
            ParseAnalogStickJoyRight(pBuffer, size);
            ParseSensorDataJoyRight(pBuffer, size, samplesSinceLast);
            break;
        }
        case BleInputReportFormatVersion_Palma:
        {
            NN_SDK_REQUIRES(size >= BleInputReportFormat_Palma_Size, "Invalid size(%d) set", size);
            m_State.sampleNumber = pBuffer[BleInputReportFormat_Palma::SampleNumber];
            int samplesSinceLast = GetDeltaSampleNumber(m_State.sampleNumber);
            ParseButtonStatePalma(pBuffer, size);
            ParseAnalogStickPalma(pBuffer, size);
            ParseSensorDataPalma(pBuffer, size, samplesSinceLast);
            break;
        }
        // 何もしない
        case BleInputReportFormatVersion_Unknown:
        case BleInputReportFormatVersion_JoyLeft:
            break;
        default:
            NN_UNEXPECTED_DEFAULT;
    }

    PAD_DUMP_LOG("[xcd] PadState");
    PAD_DUMP_LOG("\n    -> Sample# %d", m_State.sampleNumber);
    PAD_DUMP_LOG("\n    -> AnalogStick ");
    DumpStickValue(m_State.analogStickR);
    PAD_DUMP_LOG("\n    -> SixAxisSensor %d index=%d", m_SensorStates[m_SensorBufferIndex].sampleNumber, m_SensorBufferIndex);
    PAD_DUMP_LOG("\n       accelerometer: ");
    DumpSensorValue(m_SensorStates[m_SensorBufferIndex].accelerometer);
    PAD_DUMP_LOG("\n       gyroscope: ");
    DumpSensorValue(m_SensorStates[m_SensorBufferIndex].gyroscope);
    PAD_DUMP_LOG("\n");
}

Result BlePadInput::GetPadState(PadState* pOutValue) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutValue);

    memcpy(pOutValue, &m_State, sizeof(PadState));
    NN_RESULT_SUCCESS;
}

Result BlePadInput::GetSensorStates(int* pOutCount, SixAxisSensorState* pOutValue, int length) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutCount);
    NN_SDK_REQUIRES_NOT_NULL(pOutValue);
    NN_SDK_REQUIRES_GREATER(length, 0);
    NN_SDK_REQUIRES_LESS_EQUAL(length, SixAxisSensorSampleCountMax);

    ::std::lock_guard<decltype(m_SensorMutex)
                      > locker(m_SensorMutex);

    int64_t lastSamplingNumber = 0;
    *pOutCount = 0;

    if (m_SensorBufferCount == 0)
    {
        NN_RESULT_SUCCESS;
    }

    for (int i = 0; i < length; i++)
    {
        auto sensorIndex = (m_SensorBufferIndex + SixAxisSensorSampleCountMax - i) % SixAxisSensorSampleCountMax;
        auto sensorState = m_SensorStates[sensorIndex];

        if (lastSamplingNumber < sensorState.sampleNumber && lastSamplingNumber != 0)
        {
            break;
        }

        pOutValue[i] = sensorState;
        (*pOutCount)++;

        if (length == *pOutCount)
        {
            break;
        }

        lastSamplingNumber = sensorState.sampleNumber;
    }

    NN_ABORT_UNLESS_GREATER(*pOutCount, 0);
    NN_ABORT_UNLESS_LESS_EQUAL(*pOutCount, length);

    NN_RESULT_SUCCESS;
}

int BlePadInput::GetDeltaSampleNumber(uint8_t sampleNumber) NN_NOEXCEPT
{
    if (m_IsFirstSample)
    {
        m_IsFirstSample = false;
        m_LastSampleNumberRaw = sampleNumber;
        return 1;
    }
    // サンプルナンバーの差分を計算する
    int outSamples = ((sampleNumber + SampleNumberFromBleDeviceMax) - m_LastSampleNumberRaw) % SampleNumberFromBleDeviceMax;
    m_LastSampleNumberRaw = sampleNumber;
    return outSamples;
}

void BlePadInput::ParseButtonStateJoyRight(const uint8_t* pBuffer, size_t size) NN_NOEXCEPT
{
    NN_UNUSED(size);

    auto bit = nn::util::BitPack8();

    bit.SetMaskedBits(0xFF, pBuffer[BleInputReportFormat_JoyRight::Button]);
    m_State.buttons.Set<PadButton::Y>       (bit.GetBit(0));
    m_State.buttons.Set<PadButton::X>       (bit.GetBit(1));
    m_State.buttons.Set<PadButton::B>       (bit.GetBit(2));
    m_State.buttons.Set<PadButton::A>       (bit.GetBit(3));
    m_State.buttons.Set<PadButton::SR>      (bit.GetBit(4));
    m_State.buttons.Set<PadButton::SL>      (bit.GetBit(5));
    m_State.buttons.Set<PadButton::R>       (bit.GetBit(6));
    m_State.buttons.Set<PadButton::ZR>      (bit.GetBit(7));

    bit.SetMaskedBits(0xFF, pBuffer[BleInputReportFormat_JoyRight::Button + 1]);
    m_State.buttons.Set<PadButton::Start>   (bit.GetBit(0)); // Plus
    m_State.buttons.Set<PadButton::StickR>  (bit.GetBit(1));
    m_State.buttons.Set<PadButton::Home>    (bit.GetBit(2));
}

void BlePadInput::ParseAnalogStickJoyRight(const uint8_t* pBuffer, size_t size) NN_NOEXCEPT
{
    NN_UNUSED(size);

    ParseAnalogStick(&(m_State.analogStickR), &pBuffer[BleInputReportFormat_JoyRight::AnalogStick]);
}

void BlePadInput::ParseSensorDataJoyRight(const uint8_t* pBuffer, size_t size, int samplesSinceLast) NN_NOEXCEPT
{
    NN_UNUSED(size);

    ::std::lock_guard<decltype(m_SensorMutex)
                      > locker(m_SensorMutex);

    // サンプルナンバーを更新する
    m_LastSampleNumberForBuffer += samplesSinceLast;

    // BufferIndex を更新する
    m_SensorBufferIndex++;
    if (m_SensorBufferIndex == SixAxisSensorSampleCountMax)
    {
        m_SensorBufferIndex = 0;
    }
    NN_ABORT_UNLESS_LESS(m_SensorBufferIndex, SixAxisSensorSampleCountMax);

    // PadState の方は旧実装なため、センサーデータを含めない
    // SensorState を更新する
    auto& sensorState = m_SensorStates[m_SensorBufferIndex];
    sensorState.sampleNumber = m_LastSampleNumberForBuffer;

    // FSR = 8G から FSR = 2G の単位系に変換
    ParseSensorData(&sensorState.accelerometer, 4, &pBuffer[BleInputReportFormat_JoyRight::Accelerometer]);
    // FSR = 2000dps のまま
    ParseSensorData(&sensorState.gyroscope, 1, &pBuffer[BleInputReportFormat_JoyRight::Gyroscope]);

    if (m_SensorBufferCount < SixAxisSensorSampleCountMax)
    {
        m_SensorBufferCount++;
    }
}

void BlePadInput::ParseButtonStatePalma(const uint8_t* pBuffer, size_t size) NN_NOEXCEPT
{
    NN_UNUSED(size);

    auto bit = nn::util::BitPack8();

    bit.SetMaskedBits(0xFF, pBuffer[BleInputReportFormat_Palma::Button]);
    // TOP ボタンを A にしておく
    m_State.buttons.Set<PadButton::A>       (bit.GetBit(0));
    // Stick ボタンを StickL にしておく
    m_State.buttons.Set<PadButton::StickL>  (bit.GetBit(1));
    // ほかは Reset のまま何もしない
}

void BlePadInput::ParseAnalogStickPalma(const uint8_t* pBuffer, size_t size) NN_NOEXCEPT
{
    NN_UNUSED(size);

    ParseAnalogStick(&m_State.analogStickL, &pBuffer[BleInputReportFormat_Palma::AnalogStick]);
}

void BlePadInput::ParseSensorDataPalma(const uint8_t* pBuffer, size_t size, int samplesSinceLast) NN_NOEXCEPT
{
    NN_UNUSED(size);

    ::std::lock_guard<decltype(m_SensorMutex)
                      > locker(m_SensorMutex);

    // サンプルナンバーを更新する
    m_LastSampleNumberForBuffer += samplesSinceLast;

    // BufferIndex を更新する
    m_SensorBufferIndex++;
    if (m_SensorBufferIndex == SixAxisSensorSampleCountMax)
    {
        m_SensorBufferIndex = 0;
    }
    NN_ABORT_UNLESS_LESS(m_SensorBufferIndex, SixAxisSensorSampleCountMax);

    // PadState の方は旧実装なため、センサーデータを含めない
    // SensorState を更新する
    auto& sensorState = m_SensorStates[m_SensorBufferIndex];
    sensorState.sampleNumber = m_LastSampleNumberForBuffer;

    // FSR = 8G から FSR = 2G の単位系に変換
    ParseSensorData(&sensorState.accelerometer, 4, &pBuffer[BleInputReportFormat_Palma::Accelerometer]);
    // FSR = 2000dps のまま
    ParseSensorData(&sensorState.gyroscope, 1, &pBuffer[BleInputReportFormat_Palma::Gyroscope]);

    ConvertPalmaAxis(&sensorState);

    if (m_SensorBufferCount < SixAxisSensorSampleCountMax)
    {
        m_SensorBufferCount++;
    }
}

void BlePadInput::PrintDeviceInformation() NN_NOEXCEPT
{
}

}} // namespace nn::xcd
