﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <mutex>
#include <new>
#include <nn/nn_Macro.h>
#include <nn/nn_Result.h>
#include <nn/nn_SdkAssert.h>
#include <nn/nn_StaticAssert.h>
#include <nn/os.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/xcd/xcd_Result.h>
#include <nn/xcd/xcd_Tera.h>
#include <nn/xcd/xcd_TeraFirmware.h>
#include <nn/xcd/detail/xcd_Log.h>
#include "xcd_CommandHandler.h"
#include "xcd_Peripheral.h"
#include "xcd_AttachmentBase.h"
#include "xcd_ReportTypes.h"

namespace nn { namespace xcd {

void AttachmentBase::Activate(DeviceType type, FirmwareVersionImpl firmwareVersion) NN_NOEXCEPT
{
    Peripheral::Activate(type, firmwareVersion);

}

bool AttachmentBase::IsExtDevAttached() NN_NOEXCEPT
{
    switch (m_AttachmentStatus)
    {
    case AttachmentStatus_Detached :
        return false;
    case AttachmentStatus_Attached:
    case AttachmentStatus_Ready:
    case AttachmentStatus_CommunicationAvailable:
        return true;
    default: NN_UNEXPECTED_DEFAULT;
    }
}

void AttachmentBase::UpdateAttachmentStatus(DeviceStatus status) NN_NOEXCEPT
{
    std::lock_guard<nn::os::Mutex> lock(m_AttachmentStatusMutex);

    // CONDET 抜けが最も優先度が高い
    if (status.connector != Connector_AttachedToEXDevice)
    {
        m_AttachmentStatus = AttachmentStatus_Detached;
        m_ExtDevType = 0x00;
        //NN_DETAIL_XCD_ERROR("m_AttachmentStatus = AttachmentStatus_Detached\n");
        return;
    }

    // 次点で電源供給を切ったときの処理
    if (status.isAttachmentReady == false)
    {
        m_AttachmentStatus = AttachmentStatus_Attached;
        m_ExtDevType = 0x00;
        //NN_DETAIL_XCD_ERROR("m_AttachmentStatus = AttachmentStatus_Attached\n");
        return;
    }

    // CONDET抜け・電源断以外でステートは戻らないので、CommunicationAvailable なら状態を更新しない
    if (m_AttachmentStatus == AttachmentStatus_CommunicationAvailable)
    {
        return;
    }

    // 他の２つのステートは Status を確認して処理
    if (status.connector == Connector_AttachedToEXDevice)
    {
        if (status.isAttachmentReady == true)
        {
            m_AttachmentStatus = AttachmentStatus_Ready;
        }
        else
        {
            m_AttachmentStatus = AttachmentStatus_Attached;
        }
    }
    return;
}

AttachmentStatus AttachmentBase::GetAttachmentStatus() NN_NOEXCEPT
{
    return m_AttachmentStatus;
}

void AttachmentBase::SetAttachmentStatus(AttachmentStatus status) NN_NOEXCEPT
{
    std::lock_guard<nn::os::Mutex> lock(m_AttachmentStatusMutex);
    m_AttachmentStatus = status;
}

void AttachmentBase::SetAttachmentType(uint8_t type) NN_NOEXCEPT
{
    m_ExtDevType = type;
}

Result AttachmentBase::GetAttachmentType(uint8_t* pOutAttachmentType) NN_NOEXCEPT
{
    if (m_AttachmentStatus != AttachmentStatus_CommunicationAvailable)
    {
        return  nn::xcd::ResultAttachmentDeviceNotReady();
    }

    *pOutAttachmentType = m_ExtDevType;

    NN_RESULT_SUCCESS;
}

Result AttachmentBase::SendInquiryCommand() NN_NOEXCEPT
{
    const nn::TimeSpan PollingInterval = nn::TimeSpan::FromMilliSeconds(15);

    if (!IsExtDevAttached())
    {
        return nn::xcd::ResultAttachmentDeviceNotAttached();
    }

    // UART 通信が Ready になるまで待機
    nn::os::Tick startTick = nn::os::GetSystemTick();
    while (m_AttachmentStatus == AttachmentStatus_Attached)
    {
        nn::os::SleepThread(PollingInterval);
        if ((nn::os::GetSystemTick() - startTick).ToTimeSpan() > AttachmentCommandResponseTimeout)
        {
            NN_DETAIL_XCD_INFO("UART Communication Ready Timeout\n");
            return nn::xcd::ResultAttachmentDeviceAttachmentReadyTimeout();
        }
    }
    NN_DETAIL_XCD_INFO("Attachment Ready Waiting Time = %4d ms\n", (nn::os::GetSystemTick() - startTick).ToTimeSpan().GetMilliSeconds());

    // 上記の while を Detach で抜けた時は NotAttached を返す
    if (!IsExtDevAttached())
    {
        return nn::xcd::ResultAttachmentDeviceNotAttached();
    }

    // Inquiry Command の発行
    m_pCommand->GetExtDevInfo(this);

    // Inquiry Command の受信待ち
    startTick = nn::os::GetSystemTick();
    while (m_ExtDevType == 0x00)
    {
        nn::os::SleepThread(PollingInterval);
        if ((nn::os::GetSystemTick() - startTick).ToTimeSpan() > AttachmentCommandResponseTimeout)
        {
            NN_DETAIL_XCD_INFO("Inquiry Response Timeout\n");
            return nn::xcd::ResultAttachmentDeviceInquiryTimeout();
        }
    }
    NN_DETAIL_XCD_INFO("Inquiry Command Waiting Time = %4d ms\n", (nn::os::GetSystemTick() - startTick).ToTimeSpan().GetMilliSeconds());

    std::lock_guard<nn::os::Mutex> lock(m_AttachmentStatusMutex);
    m_AttachmentStatus = AttachmentStatus_CommunicationAvailable;

    NN_RESULT_SUCCESS;
}

Result AttachmentBase::GetReceivedCommandFromAttachmentDevice(size_t* pOutSize, uint8_t* pOutExtDevData, uint8_t size) NN_NOEXCEPT
{
    NN_ABORT_UNLESS(size <= nn::xcd::AttachmentReceiveCommandSizeMax);
    if (m_AttachmentStatus != AttachmentStatus_CommunicationAvailable)
    {
        return  nn::xcd::ResultAttachmentDeviceNotReady();
    }

    if (m_ExtDataStatus == ExtDevReadStatus_Timeout)
    {
        *pOutSize = 0;
        return nn::xcd::ResultAttachmentDeviceTimeout();
    }

    size_t cpySize = (size <= m_ExtDataSize) ? size : m_ExtDataSize;
    *pOutSize = cpySize;
    memcpy(pOutExtDevData, m_ExtData, cpySize);

    NN_RESULT_SUCCESS;
}

Result AttachmentBase::SendCommandToAttachmentDevice(const uint8_t* pInData, size_t size) NN_NOEXCEPT
{
    NN_ABORT_UNLESS(size <= nn::xcd::AttachmentSendCommandSizeMax);

    if (m_AttachmentStatus != AttachmentStatus_CommunicationAvailable)
    {
        return  nn::xcd::ResultAttachmentDeviceNotReady();
    }

    m_pCommand->SendCommandToAttachmentDevice(pInData, size, this);
    NN_RESULT_SUCCESS;
}

Result AttachmentBase::SetAttachmentDataReceiveEvent(nn::os::SystemEventType* pEvent) NN_NOEXCEPT
{
    m_pAttachmentDataReceiveEvent = pEvent;
    NN_RESULT_SUCCESS;
}

void AttachmentBase::NotifyExtDevInfo(uint8_t extDeviceType) NN_NOEXCEPT
{
    NN_DETAIL_XCD_INFO("Get ExDevInfo ID = %x\n", extDeviceType);
    m_ExtDevType = extDeviceType;
}

void AttachmentBase::NotifyExtDevRead(uint8_t status, const uint8_t* pData, uint8_t size) NN_NOEXCEPT
{
    NN_ABORT_UNLESS(size <= AttachmentReceiveCommandSizeMax);
    m_ExtDataStatus = status;
    switch (static_cast<ExtDevReadStatus>(status))
    {
    case ExtDevReadStatus_Succsess:
        memcpy(m_ExtData, pData, size);
        m_ExtDataSize = size;
        break;
    case ExtDevReadStatus_CrcMismatch:
        NN_DETAIL_XCD_WARN("ExtDevReadData CRC Mismatch\n");
        memset(m_ExtData, 0, sizeof(m_ExtData));
        m_ExtDataSize = 0;
        break;
    case ExtDevReadStatus_NoExtDevAttached:
        NN_DETAIL_XCD_INFO("NoExtDevAttached\n");
        memset(m_ExtData, 0, sizeof(m_ExtData));
        m_ExtDataSize = 0;
        break;
    case ExtDevReadStatus_Timeout:
        NN_DETAIL_XCD_INFO("ExtDev Timeout\n");
        memset(m_ExtData, 0, sizeof(m_ExtData));
        m_ExtDataSize = 0;
        break;
    default:
        // 想定外のステータスだが、実装ミスだけでなく、
        // おかしなデータが入れられた時にも発生しうるので、 ABORT はしない
        NN_DETAIL_XCD_WARN("ExtDev UNEXPECTED DEFAULT\n");
        memset(m_ExtData, 0, sizeof(m_ExtData));
        m_ExtDataSize = 0;
        break;
    }

    if (m_pAttachmentDataReceiveEvent != nullptr)
    {
        nn::os::SignalSystemEvent(m_pAttachmentDataReceiveEvent);
    }

}

Result AttachmentBase::EnablePollingReceiveModeForAttachmentDevice(const uint8_t* pInCommand, size_t inCommandSize, AttachmentPollingMode mode, nn::xcd::DeviceType type) NN_NOEXCEPT
{
    // 大きいデータが来たときは丸める。
    size_t commandSize = inCommandSize <= nn::xcd::AttachmentSendCommandSizeMax ? inCommandSize : nn::xcd::AttachmentSendCommandSizeMax;
    if (inCommandSize > nn::xcd::AttachmentSendCommandSizeMax)
    {
        NN_DETAIL_XCD_ERROR("inCommandSize is over nn::xcd::AttachmentSendCommandSizeMax\n");
    }

    uint8_t analogStickOffset;

    // 左右ジョイコンで使うアナログスティック領域を変える
    if (type == nn::xcd::DeviceType_Left)
    {
        analogStickOffset = InputReportByte_AnalogStickSub;
        m_DeviceType = type;
    }
    else if (type == nn::xcd::DeviceType_Right)
    {
        analogStickOffset = InputReportByte_AnalogStickMain;
        m_DeviceType = type;
    }
    else
    {
        // NotAttached を返しておく
        return nn::xcd::ResultAttachmentDeviceNotAttached();
    }

    // 最初の 3byte を使ってない側のアナログスティックにマップし、残りの 36 byte を Accelerometer / Gyroscope 全部(36 バイト)を潰す
    uint8_t disableSixAxisSensorCommand[] = { analogStickOffset,               0x03,
                                              InputReportByte_Accelerometer,   0x24,
                                              0x00,                            0x00};

    // 最初の 3byte を使ってない側のアナログスティックにマップし、残りの 6 byte を Accelerometer02 を潰す
    uint8_t enableSixAxisSensorCommand[] = { analogStickOffset,                   0x03,
                                             InputReportByte_Accelerometer + 24,  0x06,
                                             0x00,                                0x00 };

    NN_SDK_LOG("[xcd] EnablePollingReceiveModeForAttachmentDevice()\n");

    if (!IsExtDevAttached())
    {
        return nn::xcd::ResultAttachmentDeviceNotAttached();
    }
    else if (m_AttachmentStatus != AttachmentStatus_CommunicationAvailable)
    {
        return  nn::xcd::ResultAttachmentDeviceNotReady();
    }

    m_PollingMode = mode;

    if (m_PollingMode == AttachmentPollingMode_SixAxisSendorEnable)
    {
        m_pCommand->SensorPayload(true, this);
        m_AckEventForSensorPayload.TimedWait(AttachmentCommandResponseTimeout);
    }

    switch (m_PollingMode)
    {
    case AttachmentPollingMode_SixAxisSendorDisable:
        m_pCommand->ConfigBasicInFormatForAttachment(disableSixAxisSensorCommand, sizeof(disableSixAxisSensorCommand), this);
        break;
    case AttachmentPollingMode_SixAxisSendorEnable:
        m_pCommand->ConfigBasicInFormatForAttachment(enableSixAxisSensorCommand, sizeof(enableSixAxisSensorCommand), this);
        break;
    default: NN_UNEXPECTED_DEFAULT;
    }

    // Config の完了を待つ
    if (m_AckEventForConfigExtDevIn.TimedWait(AttachmentCommandResponseTimeout))
    {
        m_pCommand->EnablePollingReceiveModeForAttachmentDevice(pInCommand, commandSize, this);
        if (m_AckEventForEnablePollingMode.TimedWait(AttachmentCommandResponseTimeout))
        {
            m_IsEnablePollingMode = true;
            NN_RESULT_SUCCESS;
        }
        else
        {
            NN_DETAIL_XCD_INFO("EnablePollingMode Timeout\n");
            return nn::xcd::ResultAttachmentDeviceTimeout();
        }
    }
    else
    {
        NN_DETAIL_XCD_INFO("ConfigExtDevIn Timeout\n");
        return  nn::xcd::ResultAttachmentDeviceTimeout();
    }
}

Result AttachmentBase::DisablePollingReceiveModeForAttachmentDevice() NN_NOEXCEPT
{
    m_pCommand->DisablePollingReceiveModeForAttachmentDevice(this);
    m_AckEventForDisablePollingMode.TimedWait(AttachmentCommandResponseTimeout);

    if (m_PollingMode == AttachmentPollingMode_SixAxisSendorEnable)
    {
        m_pCommand->SensorPayload(false, this);
        m_AckEventForSensorPayload.TimedWait(AttachmentCommandResponseTimeout);
    }

    // BasicIn の Config を明示的に disable にして、六軸のデータが来るようにする。
    uint8_t disableSixAxisSensorCommand[] = { 0x00, 0x00 };
    m_pCommand->ConfigBasicInFormatForAttachment(disableSixAxisSensorCommand, sizeof(disableSixAxisSensorCommand), this);
    if (m_AckEventForConfigExtDevIn.TimedWait(AttachmentCommandResponseTimeout))
    {
        m_IsEnablePollingMode = false;
        NN_RESULT_SUCCESS;
    }
    else
    {
        NN_DETAIL_XCD_INFO("Disable PollingMode Timeout\n");
        return nn::xcd::ResultAttachmentDeviceTimeout();
    }
}

Result AttachmentBase::GetPollingDataForAttachmentDevice(size_t* pOutSize, uint8_t* pOutCommand, size_t outCommandSize) NN_NOEXCEPT
{
    if (!IsExtDevAttached())
    {
        return nn::xcd::ResultAttachmentDeviceNotAttached();
    }
    else if (m_AttachmentStatus != AttachmentStatus_CommunicationAvailable)
    {
        return  nn::xcd::ResultAttachmentDeviceNotReady();
    }

    size_t cpySize = (outCommandSize <= m_ExtPollingDataSize) ? outCommandSize : m_ExtPollingDataSize;
    *pOutSize = cpySize;
    memcpy(pOutCommand, m_ExtPollingData, cpySize);

    NN_RESULT_SUCCESS;
}

bool AttachmentBase::GetStabledAttachmentExBitValue(bool immediateAttachmentExBitValue, nn::TimeSpan tickValue) NN_NOEXCEPT
{
    // AttachementExBit の安定時間
    const nn::TimeSpan AttachementExBitStableTime = nn::TimeSpan::FromMilliSeconds(30);

    if (m_AttachmentExBitTemp == immediateAttachmentExBitValue)
    {
        // 現在の tick を取得して、AttachementExBitStableTime 経っていたら、その値を返す
        if ((nn::os::GetSystemTick().ToTimeSpan() - m_AttachmentExBitTempGetTime) > AttachementExBitStableTime)
        {
            return immediateAttachmentExBitValue;
        }
        else
        {
            // この時は、まだ安定時間になってないので、反転した値を返す
            return !immediateAttachmentExBitValue;
        }
    }
    else
    {
        // AttachmentExBit の変化が起きた可能性があるので、temp を更新して、取得時間も更新する。
        m_AttachmentExBitTemp        = immediateAttachmentExBitValue;
        m_AttachmentExBitTempGetTime = tickValue;

        // この時は、まだ安定時間になってないので、反転した値を返す
        return !immediateAttachmentExBitValue;
    }
}

void AttachmentBase::ParseInputReport(const uint8_t* pBuffer, size_t size, uint8_t sampleNumber) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pBuffer);
    NN_UNUSED(sampleNumber);
    NN_UNUSED(size);

    if (m_IsEnablePollingMode)
    {
        uint8_t analogStickOffset;
        if (m_DeviceType == nn::xcd::DeviceType_Left)
        {
            analogStickOffset = InputReportByte_AnalogStickSub;
        }
        else if (m_DeviceType == nn::xcd::DeviceType_Right)
        {
            analogStickOffset = InputReportByte_AnalogStickMain;
        }
        else
        {
            NN_ABORT();
        }

        switch (m_PollingMode)
        {
        case AttachmentPollingMode_SixAxisSendorDisable:
            if (static_cast<size_t>(pBuffer[analogStickOffset]) <= MaxPollingModeReceiveDataSizeForSixAxisSensorDisable)
            {
                m_ExtPollingDataSize = pBuffer[analogStickOffset];
            }
            else
            {
                m_ExtPollingDataSize = MaxPollingModeReceiveDataSizeForSixAxisSensorDisable;
            }
            memcpy(m_ExtPollingData, &pBuffer[analogStickOffset + 1], 2);
            memcpy(&m_ExtPollingData[2], &pBuffer[InputReportByte_Accelerometer], 0x24);
            break;
        case AttachmentPollingMode_SixAxisSendorEnable:
            if (static_cast<size_t>(pBuffer[analogStickOffset]) <= MaxPollingModeReceiveDataSizeForSixAxisSensorEnable)
            {
                m_ExtPollingDataSize = pBuffer[analogStickOffset];
            }
            else
            {
                m_ExtPollingDataSize = MaxPollingModeReceiveDataSizeForSixAxisSensorEnable;
            }
            memcpy(m_ExtPollingData, &pBuffer[analogStickOffset + 1], 2);
            memcpy(&m_ExtPollingData[2], &pBuffer[InputReportByte_Accelerometer + 24], 0x06);
            break;
        default: NN_UNEXPECTED_DEFAULT;
        }
    }
}

void AttachmentBase::NotifyAck(Result result, uint8_t id) NN_NOEXCEPT
{
    m_LatestAckResult = result;
    m_LatestAckId = id;
    if (id == Command_ExtDevInFormatConfig.Id)
    {
        m_AckEventForConfigExtDevIn.Signal();
    }
    else if (id == Command_ExtDevPollingEnable.Id)
    {
        m_AckEventForEnablePollingMode.Signal();
    }
    else if (id == Command_ExtDevPollingDisable.Id)
    {
        m_AckEventForDisablePollingMode.Signal();
    }
    else if (id == Command_SensorSleep.Id)
    {
        m_AckEventForSensorPayload.Signal();
    }
};

void AttachmentBase::NotifyMcuRead(const uint8_t* pBuffer, size_t size) NN_NOEXCEPT
{
    NN_UNUSED(pBuffer);
    NN_UNUSED(size);

    // AttachmentBase で McuRead の Notify を受ける場合は、必ず電源操作だけなので、
    // イベントをシグナルだけして、パースはしない。
    m_AckEventForExtControl.Signal();
}

Result AttachmentBase::WaitExtControlAck() NN_NOEXCEPT
{
    if (m_AckEventForExtControl.TimedWait(AttachmentCommandResponseTimeout))
    {
        NN_RESULT_SUCCESS;
    }
    else
    {
        NN_DETAIL_XCD_INFO("ExtControl Timeout\n");
        return  nn::xcd::ResultAttachmentDeviceTimeout();
    }
}

}} // namespace nn::xcd
