﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <mutex>
#include <nn/nn_Common.h>
#include <nn/nn_SdkAssert.h>
#include <nn/os/os_MessageQueue.h>
#include <nn/os/os_Mutex.h>
#include "xcd_TeraNfc.h"

#if defined(NN_BUILD_CONFIG_COMPILER_SUPPORTS_VC)
    // C4351: 配列メンバの初期化が規定値で行われる旨の警告を抑止
    #pragma warning(push)
    #pragma warning(disable: 4351)
#endif

namespace nn { namespace xcd { namespace detail {

/**
 * @brief   MessageQueue を用いて、任意の型に対するキューを構築するクラス。
 *
 * @tparam  Type        キューに格納するデータの型
 * @tparam  Capacity    キューに格納できるデータ数の上限
 */
template<class Type, int Capacity>
class TypedQueue
{
    NN_DISALLOW_COPY(TypedQueue);

public:
    TypedQueue() NN_NOEXCEPT :
        m_Mutex(true),
        m_QueueBuffer(),
        m_DataBuffer(),
        m_Queue(),
        m_DataCount(0),
        m_NextDataIndex(0)
    {
        nn::os::InitializeMessageQueue(&m_Queue, m_QueueBuffer, Capacity);
    }

    ~TypedQueue() NN_NOEXCEPT
    {
        nn::os::FinalizeMessageQueue(&m_Queue);
    }

    /**
     * @brief   キューにデータを追加する
     *
     * @param[in]   data    追加するデータ
     *
     * @retval  true    成功
     * @retval  false   キューが一杯で追加できない
     *
     * @post
     *  キューの末尾にデータが追加される。
     */
    bool TryEnqueue(const Type& data) NN_NOEXCEPT
    {
        std::lock_guard<nn::os::Mutex> lock(m_Mutex);

        if (IsFull())
        {
            return false;
        }

        // 内部のバッファに実体をコピーし、キューには実体をコピーしたバッファ要素のアドレスを登録する
        auto* pNextData = &m_DataBuffer[m_NextDataIndex];
        *pNextData = data;

        auto dataPtr = reinterpret_cast<uintptr_t>(pNextData);
        if (nn::os::TrySendMessageQueue(&m_Queue, dataPtr))
        {
            m_NextDataIndex = (m_NextDataIndex + 1) % Capacity;
            m_DataCount++;
            return true;
        }
        else
        {
            return false;
        }
    }

    /**
     * @brief   キューからデータを取り出す
     *
     * @param[out]  pOutData    取り出したデータの格納先
     *
     * @retval  true    成功
     * @retval  false   キューにデータがない
     *
     * @pre
     *  - pOutData != nullptr
     *
     * @post
     *  キューの先頭データが削除される。
     */
    bool TryDequeue(Type* pOutData) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_NOT_NULL(pOutData);

        std::lock_guard<nn::os::Mutex> lock(m_Mutex);

        uintptr_t data;
        bool isSuccess = nn::os::TryReceiveMessageQueue(&data, &m_Queue);
        if (isSuccess)
        {
            // データの本体を渡す
            *pOutData = *reinterpret_cast<Type*>(data);
            m_DataCount--;
        }

        return isSuccess;
    }

    /**
     * @brief   キューを空にする
     */
    void Clear() NN_NOEXCEPT
    {
        std::lock_guard<nn::os::Mutex> lock(m_Mutex);

        // バッファが空になるまで読み出す
        uintptr_t data;
        while (nn::os::TryReceiveMessageQueue(&data, &m_Queue)) {}
        m_DataCount = 0;
    }

    /**
     * @brief   キューが一杯か
     */
    bool IsFull() NN_NOEXCEPT
    {
        std::lock_guard<nn::os::Mutex> lock(m_Mutex);

        return m_DataCount >= Capacity;
    }

private:
    nn::os::Mutex            m_Mutex;                   //!< キューへのアクセスを排他する Mutex
    uintptr_t                m_QueueBuffer[Capacity];   //!< メッセージキューのデータ (データ本体のアドレス)
    Type                     m_DataBuffer[Capacity];    //!< キューに溜め込むデータの本体
    nn::os::MessageQueueType m_Queue;                   //!< メッセージキュー
    int                      m_DataCount;               //!< キュー内のデータの数
    int                      m_NextDataIndex;           //!< 次に使用するデータ本体の番号
};

}}}  // nn::xcd::detail

#if defined(NN_BUILD_CONFIG_COMPILER_SUPPORTS_VC)
    #pragma warning(pop)
#endif
