﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <algorithm>
#include <nn/nn_Macro.h>
#include <nn/util/util_BitPack.h>
#include "../xcd_ReportTypes.h"

#if defined(NN_BUILD_CONFIG_COMPILER_VC)
    // C4351: 配列メンバの初期化が規定値で行われる旨の警告を抑止
    #pragma warning(push)
    #pragma warning(disable: 4351)
#endif

namespace nn { namespace xcd { namespace detail {

/**
 * @brief   Flash ROM の末尾に書き込む CRC のサイズ
 */
const size_t McuFirmwareFooterCrcSize = 4;

/**
 * @brief   Flash ROM の末尾に書き込むデータブロック数のサイズ
 */
const size_t McuFirmwareFooterBlockNumberSize = 2;

/**
 * @brief   Flash ROM の末尾に書き込む情報のサイズ
 */
const size_t McuFirmwareFooterSize = McuFirmwareFooterCrcSize + McuFirmwareFooterBlockNumberSize;

/**
 * @brief   FW イメージ情報のサイズ
 */
const size_t McuFirmwareImageHeaderSize = 8;

/**
 * @brief   FW イメージの最大サイズ
 */
const size_t McuFirmwareImageSizeMax = 512 * 1024 - McuFirmwareFooterSize;

/**
 * @brief   消去する Flash ROM のセクタ (Customer code 用)
 */
const int EraseSectorsCustomer[] = { 1, 2, 3, 4, 5, 6, 7 };

/**
 * @brief   消去する Flash ROM のセクタ (全域消去用)
 */
const int EraseSectorsFull[] = { 0, 1, 2, 3, 4, 5, 6, 7 };

/**
 * @brief   消去セクタ数の上限
 */
const int EraseSectorCountMax = sizeof(EraseSectorsFull) / sizeof(EraseSectorsFull[0]);

/**
 * @brief   MCU Update Out で送出できるイメージの最大サイズ
 */
const size_t McuUpdateDataSizeMax = 256;

/**
 * @brief   Start of Frame バイト
 */
const nn::Bit8 SofByte = 0x5A;

/**
 * @brief   データ書き込み時のパディングデータ
 */
const nn::Bit8 PaddingByte = 0xFF;

/**
 * @brief   Erase コマンドコード
 */
const nn::Bit8 EraseCommand = 0x44;

/**
 * @brief   Write Memory コマンドコード
 */
const nn::Bit8 WriteCommand = 0x31;

/**
 * @brief   Tera の Flash ROM 開始アドレス
 */
const uint32_t FlashRomStartAddress = 0x08000000;

/**
 * @brief   Flash ROM の IAP アドレスオフセット
 */
const uint32_t AddressOffsetIap = 0x00000000;

/**
 * @brief   Flash ROM の NFC Chip FW (通常書き込み開始位置) オフセット
 */
const uint32_t AddressOffsetNfcChipFirmware = 0x00004000;

// Flash ROM の Customer code オフセット
//const uint32_t AddressOffsetCustomerCode = 0x00010000;

/**
 * @brief   CRC32 のアドレスオフセット
 */
const uint32_t AddressOffsetCrc32 = 0x0007FFF8;

// 更新データブロック数のアドレスオフセット
//const uint32_t AddressOffsetBlockCount = 0x0007FFFC;

/**
 * @brief   起動フラグ書き込みアドレス
 */
const uint32_t AddressOffsetActivationFlag = 0x0007FFFE;

/**
 * @brief   起動フラグ領域のサイズ
 */
const size_t ActivationFlagSize = 2;

/**
 * @brief   FW イメージの種類
 */
enum McuFirmwareImageType : uint8_t
{
    McuFirmwareImageType_JoyRight = 0xCC,   //!< 右コン用イメージ
    McuFirmwareImageType_FullKey  = 0xCD    //!< フルキーコン用イメージ
};

/**
 * @brief   FW 更新中のイベント
 */
enum McuUpdateEventType
{
    McuUpdateEventType_None             = 0,    //!< イベント未発生
    McuUpdateEventType_OueueOverflow    = 1,    //!< キュー溢れ
    McuUpdateEventType_Nack             = 2,    //!< NACK 応答
    McuUpdateEventType_Timeout          = 3,    //!< 規定時間 MCU Update Out が届かなかった
    McuUpdateEventType_InvalidReport    = 4,    //!< 無効な MCU Update Out
    McuUpdateEventType_DuplicateReport  = 5,    //!< 同じ MCU Update Out が連続して届いた
    McuUpdateEventType_OldReport        = 6,    //!< 既に処理済みの MCU Update Out が届いた
    McuUpdateEventType_ReportLost       = 7     //!< MCU Update Out に抜けが発生した
};

/**
 * @brief   FW イメージの情報
 */
struct McuFirmwareImageInfo
{
    McuFirmwareImageType    type;           //!< イメージの種類
    uint8_t                 majorVersion;   //!< メジャーバージョン
    uint8_t                 minorVersion;   //!< マイナーバージョン
    uint8_t                 _reserve[5];
};

/**
 * @brief   ペイロードの属性です。
 */
struct McuUpdateInPayloadAttributePack
{
    typedef ::nn::util::BitPack8::Field<0, 1, uint8_t> QueueFull;   //!< コマンドキューが一杯か
    typedef ::nn::util::BitPack8::Field<1, 1, uint8_t> QueueEmpty;  //!< コマンドキューが空か
    typedef ::nn::util::BitPack8::Field<2, 1, uint8_t> Error;       //!< エラーが発生したか
    typedef ::nn::util::BitPack8::Field<3, 1, uint8_t> ReportLost;  //!< Output report の喪失が発生した
};

/**
 * @brief   MCU Update In のペイロード
 *
 * @note    高速版と旧版の両方で使用するため、共通するメンバの扱いを変更してはならない。
 *          互換がとれなくなる場合は、それぞれで専用の定義を使用するようにする。
 */
struct NN_ALIGNAS(1) McuUpdateInPayload
{
    ::nn::util::BitPack8 _flags;        // 状態フラグ (高速版 / 旧版で共通)
    uint8_t _completeCount[2];          // 完了済みコマンド数
    uint8_t _receiveCount[2];           // 到達済みコマンド数
    uint8_t _lastSequenceNumber[2];     // 最後に受信したレポートの sequence number
    uint8_t _eventType;                 // 直近に発生したイベント

    /**
     * @brief   メンバをクリアします。
     */
    void Clear() NN_NOEXCEPT
    {
        _flags.Clear();
        std::memset(_completeCount, 0, sizeof(_completeCount));
        std::memset(_receiveCount, 0, sizeof(_receiveCount));
        std::memset(_lastSequenceNumber, 0, sizeof(_lastSequenceNumber));
        _eventType = 0;
    }

    /**
     * @brief   コマンドキューが埋まっているかどうかを返します。
     */
    bool IsQueueFull() const NN_NOEXCEPT
    {
        return _flags.Get<McuUpdateInPayloadAttributePack::QueueFull>() != 0;
    }

    /**
     * @brief   コマンドキューが空かどうかを返します。
     */
    bool IsQueueEmpty() const NN_NOEXCEPT
    {
        return _flags.Get<McuUpdateInPayloadAttributePack::QueueEmpty>() != 0;
    }

    /**
     * @brief   エラーが発生したかどうかを返します。
     */
    bool HasError() const NN_NOEXCEPT
    {
        return _flags.Get<McuUpdateInPayloadAttributePack::Error>() != 0;
    }

    /**
     * @brief   Output report の喪失が発生したかどうかを返します。
     */
    bool IsReportLost() const NN_NOEXCEPT
    {
        return _flags.Get<McuUpdateInPayloadAttributePack::ReportLost>() != 0;
    }

    /**
     * @brief   コントローラー側で処理が完了したコマンド数を取得します。
     */
    int GetCompleteCount() const NN_NOEXCEPT
    {
        // Little endian で格納されている
        return (_completeCount[1] << 8) + _completeCount[0];
    }

    /**
     * @brief   コントローラー側に到達したコマンド数を取得します。
     */
    int GetReceiveCount() const NN_NOEXCEPT
    {
        // Little endian で格納されている
        return (_receiveCount[1] << 8) + _receiveCount[0];
    }

    /**
     * @brief   コントローラー側に届いた最新 sequence number を取得します。
     */
    int GetLastSequenceNumber() const NN_NOEXCEPT
    {
        // Little endian で格納されている
        return (_lastSequenceNumber[1] << 8) + _lastSequenceNumber[0];
    }

    /**
     * @brief   コントローラー側で発生したイベントを取得します。
     */
    McuUpdateEventType GetEventType() const NN_NOEXCEPT
    {
        return static_cast<McuUpdateEventType>(_eventType);
    }
};

/**
 * @brief   MCU Update Out のペイロード
 *
 * @note    高速版と旧版の両方で使用するため、ペイロードのフォーマットを変更してはならない。
 *          互換がとれなくなる場合は、それぞれで専用の定義を使用するようにする。
 */
struct McuUpdateOutPayload
{
    size_t  payloadSize;                                //!< ペイロード長
    int     commandCount;                               //!< コマンド数
    int     sequenceNumber;                             //!< シーケンス番号
    uint8_t payload[McuUpdateOutReportSize_Payload];    //!< ペイロード

    /**
     * @brief   値のクリア
     */
    void Clear() NN_NOEXCEPT
    {
        payloadSize    = 0;
        commandCount   = 0;
        sequenceNumber = 0;
    }

    /**
     * @brief   既にデータ設定済みか
     */
    bool HasData() const NN_NOEXCEPT
    {
        return payloadSize > 0;
    }

    /**
     * @brief   ペイロードの設定
     */
    void SetPayload(const uint8_t* pData, size_t dataSize) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_NOT_NULL(pData);
        NN_SDK_REQUIRES_LESS_EQUAL(dataSize, sizeof(payload) - 2);

        payloadSize = dataSize + 2;

        // 先頭にデータサイズ (16 bit little endian)、続けてデータ部を格納
        payload[0] = static_cast<uint8_t>(dataSize & 0xFF);
        payload[1] = static_cast<uint8_t>((dataSize >> 8) & 0xFF);
        if (dataSize > 0)
        {
            std::memcpy(payload + 2, pData, dataSize);
        }

        // 残りは念のため 0 埋め
        size_t remainBytes = sizeof(payload) - payloadSize;
        if (remainBytes > 0)
        {
            std::memset(payload + payloadSize, 0, remainBytes);
        }
    }
};

/**
 * @brief   MCU Update Out Report で送出するコマンドを生成するクラス
 *
 * @note    高速版でのみ使用。
 */
class TeraUpdaterCommandBuilder final
{
    NN_DISALLOW_COPY(TeraUpdaterCommandBuilder);
    NN_DISALLOW_MOVE(TeraUpdaterCommandBuilder);

public:
    TeraUpdaterCommandBuilder() NN_NOEXCEPT :
        m_Buffer(),
        m_UsedBufferSize(0),
        m_CommandCount(0),
        m_SequenceNumber(0)
    {
    }

    /**
     * @brief   コマンドを追加
     */
    bool Append(const uint8_t* pCommand, size_t commandSize) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_NOT_NULL(pCommand);

        if (commandSize + 2 >= GetFreeSize())
        {
            return false;
        }

        auto* pBuffer = &m_Buffer[m_UsedBufferSize];

        // コマンドサイズ (Little endian)
        pBuffer[0] = static_cast<uint8_t>(commandSize & 0xFF);
        pBuffer[1] = static_cast<uint8_t>((commandSize >> 8) & 0xFF);

        std::memcpy(&pBuffer[2], pCommand, commandSize);

        m_UsedBufferSize += commandSize + 2;
        m_CommandCount++;

        return true;
    }

    /**
     * @brief   コマンドが空か
     */
    bool IsEmpty() const NN_NOEXCEPT
    {
        return GetCommandCount() == 0;
    }

    /**
     * @brief   登録済みのコマンド数を取得
     */
    int GetCommandCount() const NN_NOEXCEPT
    {
        return m_CommandCount;
    }

    /**
     * @brief   生成した Output Report の数
     */
    int GetSequenceNumber() const NN_NOEXCEPT
    {
        return m_SequenceNumber;
    }

    /**
     * @brief   Output Report のペイロードを生成
     */
    int CreateReport(size_t* pOutSize, uint8_t* pOutReport, size_t bufferSize) NN_NOEXCEPT
    {
        const size_t BufferSizeMax = sizeof(m_Buffer) + 4;
        NN_SDK_REQUIRES_NOT_NULL(pOutReport);
        NN_SDK_REQUIRES_GREATER_EQUAL(bufferSize, BufferSizeMax);
        NN_UNUSED(BufferSizeMax);

        m_SequenceNumber++;

        // 先頭はシーケンス番号
        pOutReport[0] = m_SequenceNumber & 0xFF;
        pOutReport[1] = (m_SequenceNumber >> 8) & 0xFF;

        std::memcpy(pOutReport + 2, m_Buffer, std::min(m_UsedBufferSize, bufferSize));

        // 終端は 0
        pOutReport[m_UsedBufferSize + 2] = 0;
        pOutReport[m_UsedBufferSize + 3] = 0;

        *pOutSize = m_UsedBufferSize + 4;

        ClearBuffer();

        return m_SequenceNumber;
    }

    /**
     * @brief   メンバのクリア
     */
    void Clear() NN_NOEXCEPT
    {
        ClearBuffer();
        m_SequenceNumber = 0;
    }

    /**
     * @brief   バッファのクリア
     */
    void ClearBuffer() NN_NOEXCEPT
    {
        std::memset(m_Buffer, 0, sizeof(m_Buffer));
        m_UsedBufferSize = 0;
        m_CommandCount   = 0;
    }

private:
    /**
     * @brief   バッファ空き容量の取得
     */
    size_t GetFreeSize() const NN_NOEXCEPT
    {
        return sizeof(m_Buffer) - m_UsedBufferSize;
    }

private:
    uint8_t m_Buffer[McuUpdateOutReportSize_Payload - 4];   //!< コマンドバッファ (4 バイトはシーケンス番号と終端用)
    size_t  m_UsedBufferSize;                               //!< バッファの使用済みサイズ
    int     m_CommandCount;                                 //!< 登録済みのコマンド数
    int     m_SequenceNumber;                               //!< 生成済みの Output Report 数
};

/**
 * @brief   指定した値を Big Endian でバイト列に格納
 */
template <typename OutType, typename InType>
void SetBigEndianValue(OutType *pOutData, size_t outSize, InType value)
{
    NN_STATIC_ASSERT(sizeof(OutType) == 1);  // 1 バイトの型しか受け付けない

    NN_SDK_REQUIRES_NOT_NULL(pOutData);

    size_t dataSize = std::min(sizeof(InType), outSize);
    for (size_t i = 0; i < dataSize; i++)
    {
        pOutData[i] = static_cast<OutType>((value >> (8 * (dataSize - i - 1))) & 0xFF);
    }
}

}}}  // nn::xcd::detail

#if defined(NN_BUILD_CONFIG_COMPILER_VC)
    #pragma warning(pop)
#endif
