﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <mutex>
#include <nn/nn_Result.h>
#include <nn/nn_Macro.h>
#include <nn/fs.h>
#include <nn/os/os_Mutex.h>
#include <nn/xcd/xcd_Device.h>
#include <nn/xcd/xcd_Tera.h>
#include <nn/xcd/xcd_TeraFirmware.h>
#include <nn/xcd/xcd_Result.h>
#include "xcd_TeraCommon.h"
#include "xcd_TimeCounter.h"
#include "xcd_TeraUpdaterTypes.h"
#include "xcd_TeraUpdaterStream.h"
#include "../xcd_ITeraUpdater.h"

#if defined(NN_BUILD_CONFIG_COMPILER_VC)
    // C4351: 配列メンバの初期化が規定値で行われる旨の警告を抑止
    #pragma warning(push)
    #pragma warning(disable: 4351)
#endif

namespace nn { namespace xcd { namespace detail {

/**
 * @brief   Tera MCU の FW アップデート処理を行うクラス
 */
class TeraUpdaterLegacy final : public ITeraUpdater
{
    NN_DISALLOW_COPY(TeraUpdaterLegacy);
    NN_DISALLOW_MOVE(TeraUpdaterLegacy);

public:
    /**
     * @brief   アップデートの段階
     */
    enum class UpdatePhaseLegacy
    {
        Stopped,                //!< アップデートを行っていない
        BootPrepare,            //!< システムブート準備
        Boot,                   //!< システムブート中
        BootFinish,             //!< システムブート完了
        InvalidateSendCommand,  //!< 起動フラグ無効化コマンド送信
        InvalidateSendAddress,  //!< 起動フラグアドレス送信
        InvalidateSendData,     //!< 起動フラグ消去データ送信
        EraseSendCommand,       //!< 消去コマンド送信
        EraseSendSectorCount,   //!< 消去セクタ数送信
        EraseSendSectors,       //!< 消去セクタ番号送信
        EraseFinish,            //!< 消去完了
        WriteSendCommand,       //!< 書き込みコマンド送信
        WriteSendAddress,       //!< 書き込み先アドレス送信
        WriteSendData,          //!< 書き込みデータ送信
        FinalizeSendCommand,    //!< ファイナライズコマンド送信
        FinalizeSendAddress,    //!< ファイナライズ先アドレス送信
        FinalizeSendData,       //!< ファイナライズデータ送信
        FinalizeWaiting,        //!< ファイナライズ完了待ち
        FinalizeFinish,         //!< ファイナライズ完了
        Error                   //!< エラー発生
    };

    /**
     * @brief   Flash の消去状況
     */
    enum class EraseState
    {
        None,                   //!< 消去開始前
        Started,                //!< 消去コマンド送出
        WaitingQueueNotEmpty,   //!< Queue が埋まる (= Erase が始まる) のを待つ
        WaitingQueueEmpty,      //!< Queue が空く (= Erase が終わる) のを待つ
        Finished                //!< 完了
    };

public:
    TeraUpdaterLegacy() NN_NOEXCEPT :
        m_Mutex(true),
        m_IsActive(false),
        m_PhaseChangeCallback(nullptr),
        m_pCallbackArgument(nullptr),
        m_Phase(UpdatePhaseLegacy::Stopped),
        m_Mode(UpdateMode::Normal),
        m_EraseState(EraseState::None),
        m_ProgressRate(0),
        m_ReceivedInput(),
        m_OutputData(),
        m_SentCommandCount(0),
        m_TotalCommandCount(0),
        m_RestartCounter(),
        m_TimeoutCounter(),
        m_Stream(),
        m_Crc32(detail::Crc32InitialValue)
    {}

    void Activate(DeviceType deviceType, PhaseChangeCallbackType callback, void* pCallbackArgument) NN_NOEXCEPT;
    void Deactivate() NN_NOEXCEPT;

    /**
     * @brief   書き込む FW イメージを設定
     *
     * @param[in]   image       FW イメージ
     * @param[in]   mode        アップデートモード
     */
    nn::Result SetFirmwareImage(
        const FirmwareImage& image,
        UpdateMode mode) NN_NOEXCEPT;

    /**
     * @brief   アップデート処理の中断要求
     */
    nn::Result Abort() NN_NOEXCEPT;

    /**
     * @brief   アップデート状況の取得
     *
     * @param[out]  pOutStateInfo   状態の取得先
     */
    nn::Result GetState(McuUpdateStateInfo* pOutStateInfo) const NN_NOEXCEPT;

    /**
     * @brief   Firmware Update 中に受け取ったデータをパースします
     */
    void ParseMcuUpdateInputReport(const uint8_t* pBuffer, size_t size, uint8_t sampleNumber) NN_NOEXCEPT;

    /**
     * @brief   Firmware Update でコントローラーに送信するデータを取得します。
     */
    size_t GetMcuUpdateOutputReport(uint8_t* pOutValue, size_t size) NN_NOEXCEPT;

    /**
     * @brief   システムブートローダとの同期を開始
     */
    void StartSyncWithBootLoader() NN_NOEXCEPT;

    /**
     * @brief   ROM の消去を開始
     */
    void StartEraseRom() NN_NOEXCEPT;

    /**
     * @brief   ROM の書き込みを開始
     */
    void StartWriteRom() NN_NOEXCEPT;

    /**
     * @brief   アップデートの終了
     */
    void Finish() NN_NOEXCEPT;

private:
    struct RomFooter
    {
        uint8_t value[McuFirmwareFooterSize];
    };

private:
    /**
     * @brief   内部状態のクリア
     */
    void ClearInternalState() NN_NOEXCEPT;

    /**
     * @brief   Flash ROM の末尾に書き込むデータを取得
     */
    void GetRomFooter(RomFooter* pOutRomFooter) NN_NOEXCEPT;

    /**
     * @brief   アップデート処理が中断したか
     */
    bool IsAborted() const NN_NOEXCEPT;

    /**
     * @brief   アップデート段階の更新
     */
    void SetPhase(UpdatePhaseLegacy phase) NN_NOEXCEPT;

    /**
     * @brief   アップデート段階の更新通知
     */
    void NotifyPhaseChanged() NN_NOEXCEPT;

    /**
     * @brief   MCU Update In の処理
     */
    void ProcessMcuUpdateInput() NN_NOEXCEPT;

    /**
     * @brief   消去待ちの処理
     */
    void ProcessWaitErase() NN_NOEXCEPT;

    /**
     * @brief   最大数のパケット送出後の処理
     */
    void ProcessMaximumSent() NN_NOEXCEPT;

    /**
     * @brief   アップデートの進行処理
     */
    void Proceed() NN_NOEXCEPT;

    /**
     * @brief   システムブート処理の進行処理
     */
    void ProceedBootPhase() NN_NOEXCEPT;

    /**
     * @brief   起動フラグ無効化の進行処理
     */
    void ProceedInvalidatePhase() NN_NOEXCEPT;

    /**
     * @brief   ROM 消去の進行処理
     */
    void ProceedErasePhase() NN_NOEXCEPT;

    /**
     * @brief   ROM 書き込みの進行処理
     */
    void ProceedWritePhase() NN_NOEXCEPT;

    /**
     * @brief   次に書き込むデータの取得
     */
    nn::Result FetchNextWriteData(
        size_t* pOutBytes,
        uint8_t* pOutBuffer,
        size_t bufferSize,
        uint32_t nextAddress) NN_NOEXCEPT;

    /**
     * @brief   ROM 書き込み最終処理の進行処理
     */
    void ProceedFinalizePhase() NN_NOEXCEPT;

private:
    mutable nn::os::Mutex       m_Mutex;                //!< メンバへのアクセスを排他するための Mutex

    bool                        m_IsActive;             //!< Activate 済みフラグ
    DeviceType                  m_Type;                 //!< デバイスの種類
    PhaseChangeCallbackType     m_PhaseChangeCallback;  //!< アップデート段階進行時に呼ぶコールバック
    void*                       m_pCallbackArgument;    //!< コールバック関数に渡す引数
    UpdatePhaseLegacy           m_Phase;                //!< アップデートの段階
    UpdateMode                  m_Mode;                 //!< アップデートモード
    EraseState                  m_EraseState;           //!< 消去状況
    int                         m_ProgressRate;         //!< 現在の進捗率
    McuUpdateInPayload          m_ReceivedInput;        //!< 直近の MCU Update In のペイロード
    McuUpdateOutPayload         m_OutputData;           //!< 次に送信する MCU Update Out のペイロード
    int                         m_SentCommandCount;     //!< 送出済みのコマンド数
    int                         m_TotalCommandCount;    //!< 合計送出コマンド数
    detail::TimeCounter         m_RestartCounter;       //!< 処理再開までの時間カウンタ
    detail::TimeCounter         m_TimeoutCounter;       //!< Queue empty 検知用のタイムアウトカウンタ
    detail::TeraUpdaterStream   m_Stream;               //!< FW イメージ読み込みクラス
    nn::Bit32                   m_Crc32;                //!< FW イメージの CRC32
};

}}} // namespace nn::xcd

#if defined(NN_BUILD_CONFIG_COMPILER_VC)
    #pragma warning(pop)
#endif
