﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/xcd/xcd_TeraFirmware.h>
#include <nn/xcd/xcd_NfcTypes.h>

namespace nn { namespace xcd { namespace detail {

/**
 * @brief   Tera MCU のファームウェアバージョン情報を格納する構造体です。
 */
class TeraFirmwareVersion
{
public:
    TeraFirmwareVersion() :
        m_HasValidVersion(false),
        m_CorruptionCheckCount(0),
        m_Version()
    {}

    /**
     * @brief   デバイスの種類を設定します。
     */
    void SetDeviceType(DeviceType type) NN_NOEXCEPT
    {
        m_Version.deviceType = type;
    }

    /**
     * @brief   有効なバージョン情報があるか返します。
     */
    bool IsValid() const NN_NOEXCEPT
    {
        return m_HasValidVersion;
    }

    /**
     * @brief   FW が破損しているか返します。
     */
    bool IsCorrupted() const NN_NOEXCEPT
    {
        return IsValid() && m_Version.isCorrupted;
    }

    /**
     * @brief   IAP が破損しているか返します。
     */
    bool IsIapCorrupted() const NN_NOEXCEPT
    {
        return IsValid() && m_Version.isIapCorrupted;
    }

    /**
     * @brief   バージョンを取得します。
     */
    bool GetVersion(McuVersionData* pOutVersion) NN_NOEXCEPT;

    /**
     * @brief   バージョンを取得します。 (NFC 向け)
     */
    bool GetVersion(McuVersionDataForNfc* pOutVersion) NN_NOEXCEPT;

    /**
     * @brief   バージョン情報を無効化します。
     */
    void Invalidate() NN_NOEXCEPT;

    /**
     * @brief   Customer code のバージョンを設定します。
     */
    void SetVersion(uint16_t major, uint16_t minor) NN_NOEXCEPT;

    /**
     * @brief   Cusromter code 破損判定カウンタを進めます。
     */
    void ProceedCustomerCodeCorruptionCount() NN_NOEXCEPT;

    /**
     * @brief   Cusromter code 破損判定カウンタを初期化します。
     */
    void ClearCustomerCodeCorruptionCount() NN_NOEXCEPT;

    /**
     * @brief   IAP 破損状態に設定します。
     */
    void SetIapCorruption() NN_NOEXCEPT;

private:
    /**
     * @brief   Cusromter code 破損状態に設定します。
     */
    void SetCustomerCodeCorruption() NN_NOEXCEPT;

private:
    bool            m_HasValidVersion;          //!< 有効な情報を保持しているか
    int             m_CorruptionCheckCount;     //!< Customer code 破損判定確定のためのカウンタ
    McuVersionData  m_Version;                  //!< バージョン情報
};

}}}
