﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/nn_Macro.h>
#include <nn/os/os_SystemEvent.h>
#include <nn/xcd/xcd_NfcTypes.h>
#include <nn/xcd/xcd_TeraFirmware.h>
#include <nn/xcd/detail/xcd_ISystemServer.sfdl.h>
#include "../xcd_DeviceListManager.h"

namespace nn { namespace xcd { namespace detail {

/**
 * XCD システムサービス (xcd:sys) を扱うクラスです。
 */
class SystemServerImpl
{
    NN_DISALLOW_COPY(SystemServerImpl);
    NN_DISALLOW_MOVE(SystemServerImpl);

public:
    SystemServerImpl() NN_NOEXCEPT {}

    /**
     * @brief   デバイスから受け取るデータのフォーマットを取得します。
     */
    nn::Result GetDataFormat(nn::sf::Out<std::uint8_t> pOutDataFormat, DeviceHandle deviceHandle) NN_NOEXCEPT;

    /**
     * @brief   デバイスから受け取るデータのフォーマットを指定します。
     */
    nn::Result SetDataFormat(std::uint8_t dataFormat, DeviceHandle deviceHandle) NN_NOEXCEPT;

    /**
     * @brief   Tera MCU の現在のステートを取得します。
     */
    nn::Result GetMcuState(nn::sf::Out<std::uint8_t> pOutMcuState, DeviceHandle deviceHandle) NN_NOEXCEPT;

    /**
     * @brief   Tera MCU のステート遷移要求を発行します。
     */
    nn::Result SetMcuState(std::uint8_t mcuState, DeviceHandle deviceHandle) NN_NOEXCEPT;

    /**
     * @brief   HID コマンドによる Tera MCU のステート遷移要求を発行します。
     */
    nn::Result SetMcuStateImmediate(std::uint8_t mcuState, DeviceHandle deviceHandle) NN_NOEXCEPT;

    /**
     * @brief   Tera MCU のファームウェアバージョンを取得します。
     */
    nn::Result GetMcuVersionForNfc(nn::sf::Out<nn::xcd::McuVersionDataForNfc> pOutMcuVersionData, DeviceHandle deviceHandle) NN_NOEXCEPT;

    /**
     * @brief   NFC 機能を使用できる電源状態であるかを確認します。
     */
    nn::Result CheckNfcDevicePower(DeviceHandle deviceHandle) NN_NOEXCEPT;

    /**
     * @brief   NFC に関連する通知を受け取るイベントを指定します。
     */
    nn::Result SetNfcEvent(nn::sf::Out<nn::sf::NativeHandle> nfcEventHandle, nn::sf::Out<nn::sf::NativeHandle> nfcDetectEventHandle, DeviceHandle deviceHandle) NN_NOEXCEPT;

    /**
     * @brief   NFC の状態を取得します。
     */
    nn::Result GetNfcInfo(nn::sf::Out<nn::xcd::NfcInfo> pOutValue, DeviceHandle deviceHandle) NN_NOEXCEPT;

    /**
     * @brief   NFC タグの検出を開始します。
     */
    nn::Result StartNfcDiscovery(nn::xcd::NfcDiscoveryParameter discoveryParameter, DeviceHandle deviceHandle) NN_NOEXCEPT;

    /**
     * @brief   NFC タグの検出動作を停止します。
     */
    nn::Result StopNfcDiscovery(DeviceHandle deviceHandle) NN_NOEXCEPT;

    /**
     * @brief   NTAG シリーズのタグの読み取りを開始します。
     */
    nn::Result StartNtagRead(const nn::xcd::NtagReadParameter& readParameter, DeviceHandle deviceHandle) NN_NOEXCEPT;

    /**
     * @brief   NTAG シリーズのタグへの書き込みを開始します。
     */
    nn::Result StartNtagWrite(const nn::xcd::NtagWriteParameter& writeParameter, DeviceHandle deviceHandle) NN_NOEXCEPT;

    /**
     * @brief   パススルーコマンドを発行します。
     */
    nn::Result SendNfcRawData(const nn::xcd::NfcPassThruParameter& passThruParameter, DeviceHandle deviceHandle) NN_NOEXCEPT;

    /**
     * @brief   MIFARE 鍵を登録します。
     */
    nn::Result RegisterMifareKey(const nn::xcd::MifareKeyWriteParameter& keyWriteParameter, DeviceHandle deviceHandle) NN_NOEXCEPT;

    /**
     * @brief   登録されている MIFARE 鍵を消去します。
     */
    nn::Result ClearMifareKey(const nn::xcd::MifareKeyClearParameter& keyClearParameter, DeviceHandle deviceHandle) NN_NOEXCEPT;

    /**
     * @brief   MIFARE タグの読み取りを行います。
     */
    nn::Result StartMifareRead(const nn::xcd::MifareReadParameter& readParameter, DeviceHandle deviceHandle) NN_NOEXCEPT;

    /**
     * @brief   MIFARE タグへ書き込みを行います。
     */
    nn::Result StartMifareWrite(const nn::xcd::MifareWriteParameter& writeParameter, DeviceHandle deviceHandle) NN_NOEXCEPT;

    /**
     * @brief   左レールのスリープ復帰要因を取得します。
     */
    nn::Result GetAwakeTriggerReasonForLeftRail(nn::sf::Out<uint64_t> pOutValue) NN_NOEXCEPT;

    /**
     * @brief   右レールのスリープ復帰要因を取得します。
     */
    nn::Result GetAwakeTriggerReasonForRightRail(nn::sf::Out<uint64_t> pOutValue) NN_NOEXCEPT;
};

}}} // namespace nn::xcd::detail
