﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/nn_Macro.h>
#include <nn/btm/btm_Types.h>
#include <nn/os/os_MemoryHeapCommon.h>
#include <nn/os/os_EventTypes.h>
#include <nn/os/os_LightEventTypes.h>
#include <nn/os/os_SystemEventTypes.h>
#include <nn/os/os_MultipleWaitTypes.h>
#include <nn/os/os_Tick.h>
#include <nn/uart/uart.h>
#include <nn/xcd/xcd_Device.h>
#include <nn/xcd/xcd_DeviceState.h>
#include <nn/xcd/xcd_Pairing.h>
#include <nn/xcd/xcd_Sleep.h>
#include <nn/xcd/xcd_Input.h>

#include "xcd_IHidListener.h"
#include "xcd_NwcpProtocolTypes.h"
#include "../xcd_BluetoothTypes.h"
#include "../xcd_IEventTask.h"
#include "../xcd_MultiWaitEvent.h"
#include "../xcd_TaskManager.h"

namespace nn { namespace xcd { namespace detail{

//!< 外部から与えてもらう必要のある Uart ドライバ用のワークバッファのサイズ
const size_t UartWorkBufferSize = nn::os::MemoryPageSize;

//!< 内部で使用する受信バッファサイズ
const int UartReceiveBufferSize = 512;

class NwcpUartDriver final : public IEventTask
{
private:
    //!< Uart の割り込みを待ち受ける多重待ちイベント
    MultiWaitSystemEvent m_Event;

    //!< Uart のデータ送信をハンドリングするタイマー
    MultiWaitTimerEvent m_SendTimerEvent;

    //!< Reset 完了を待ち受けるタイマー
    nn::os::TimerEventType m_ResetTimerEvent;

    //!< Hid の受信待ちタイマー
    nn::os::TimerEventType m_HidTimeoutTimerEvent;

    //!< DeviceCommadn の受信待ちタイマー
    nn::os::TimerEventType m_DeviceCommandTimeoutTimerEvent;

    //!< Sync の受信待ちタイマー
    nn::os::TimerEventType m_SyncTimeoutTimerEvent;

    //!< Disconnect の完了待ちタイマー
    nn::os::TimerEventType m_DisconnectTimeoutTimerEvent;

    //!< Uart ドライバを有効にしているかどうか
    bool m_IsDriverEnabled;

    //!< Uart の通信周期 (msec)
    static const ::nn::TimeSpan UartHidInterval;

    //!< Uart 高速通信時の通信周期 (msec)
    static const ::nn::TimeSpan UartHidIntervalFast;

    //!< UART の初期 BaudRate
    static const nn::uart::BaudRate s_BaudRateDefault = nn::uart::BaudRate_1M;

    //!< UART の高速通信 BaudRate
    static const nn::uart::BaudRate s_BaudRateHigh = nn::uart::BaudRate_3M;

    //!< UART のポート名
    nn::uart::PortName m_PortName;

    //!< UART の変化を通知するためのイベントオブジェクト
    nn::os::LightEventType* m_pUpdateEvent;

    //!< Inquiry までが完了したら通知するイベントオブジェクト
    nn::os::LightEventType* m_pInquiryCompleteEvent;

    //!< Uart 通信に使用するセッション
    nn::uart::PortSession m_UartSession;

    //!< Uart ドライバで使用するワークバッファ
    char* m_pSendWorkBuffer;

    //!< Uart ドライバで使用するワークバッファ
    char* m_pReceiveWorkBuffer;

    //!< Uart の受信用バッファ
    uint8_t m_ReceiveBuffer[UartReceiveBufferSize];

    //!< 読み込み中の Nwcp パケットの読み込み完了済みサイズ
    size_t m_ReceivedDataSize;

    //!< 接続されたデバイスのアドレス
    ::nn::bluetooth::Address m_Address;

    //!< 接続されたデバイスの種類
    NwcpDeviceType m_DeviceType;

    //!< 受信済み InputReport の長さ
    uint16_t m_HidInputReportLength;

    //!< 受信済み InputReport
    uint8_t m_HidInputReportBuffer[UartReceiveBufferSize];

    //!< パケットが受信されたかどうか
    bool m_IsPacketReceived;

    //!< 送信待ち OutputReport 用リングバッファのサイズ
    static const size_t OutputReportRingBufferCountMax = 32;

    struct HidOutputReportEntity
    {
        size_t size;
        uint8_t buffer[UartReceiveBufferSize];
    };

    //!< 送信待ち OutputReport 用リングバッファ
    HidOutputReportEntity m_HidOutputReportBuffer[OutputReportRingBufferCountMax];

    //!< 送信待ち OutputReport 用リングバッファの書き込み位置
    uint8_t m_HidOutputReportBufferIndex;

    //!< 送信待ち OutputReport 用リングバッファ内の数
    uint8_t m_HidOutputReportBufferCount;

    enum UartState
    {
        UartState_Detached,
        UartState_Sync,
        UartState_Inquiry,
        UartState_WriteChargerSettings,
        UartState_Pairing,
        UartState_CreateHid,
        UartState_Attached,
    };
    UartState m_UartState;

    // リトライ回数をカウントする
    int m_RetryCount;

    //!< パケット内の CRC が有効かどうか
    bool m_IsCrcEnabled;

    //!< ペアリング情報
    BluetoothDeviceInfo m_PairingDeviceInfo;

    //!< 電池残量が変化したときに通知するイベント
    ::nn::os::LightEventType* m_pBatteryLevelEvent;

    //!< 現在の電池残量
    BatteryLevel m_BatteryLevel;

    //!< 現在の充電状態
    bool m_IsCharging;

    //!< DeviceCommand のパケット最大長
    static const int DeviceCommandLengthMax = 64;

    //!< DeviceCommand のキュー最大数
    static const int DeviceCommandQueueCountMax = 4;

    //!< DeviceCommand の定義
    struct DeviceCommand
    {
        bool isAvailable;                           //!< コマンドが有効かどうか
        bool isSent;                                //!< コマンドが送信されたかどうか
        int length;                                 //!< コマンドの長さ
        uint8_t opCode;                             //!< opCode
        uint8_t buffer[DeviceCommandLengthMax];     //!< コマンドの中身
    };

    //!< DeviceCommand 用のキュー
    DeviceCommand m_DeviceCommands[DeviceCommandQueueCountMax];

    //!< DeviceCommand のキュー上の Index
    int m_DeviceCommandsQueueIndex;

    //!< DeviceCommand のビジー状態
    bool m_IsDeviceCommandBusy;

    //!< 起動要因取得が完了したときにシグナルするイベント
    ::nn::os::LightEventType* m_pAwakeTriggerReasonReadCompleteEvent;

    //!< 起動要因
    AwakeTriggerReason m_AwakeTriggerReason;

    //!< SetReport / GetReport が完了したときに通知するためのリスナー
    IHidListener* m_pHidListener;

    enum FirmwareUpdateMode
    {
        FirmwareUpdateMode_Disable,
        FirmwareUpdateMode_Enabling,
        FirmwareUpdateMode_Idle,
        FirmwareUpdateMode_SetReport,
        FirmwareUpdateMode_GetReport,
    };

    FirmwareUpdateMode m_FirmwareUpdateMode;

    ::nn::TimeSpan m_UartHidInterval;

    //!< 最後に Attach した時の Tick
    nn::os::Tick m_LastAttached;

    //!< 最後に送信したときの Tick
    nn::os::Tick m_LastSendTick;

    //!< 送信遅延が検出されたか
    bool m_IsSendDelayDetected;

    //!< Disconnect 時に Reboot するかどうか
    bool m_IsRebootEnabled;

    //!< InputReport を解析する関数
    InputReportParserFunc m_pInputReportParserFunc;
    void* m_pInputReportParserArg;

    //!< fifty の入力状態
    nn::xcd::PadState m_FiftyPadState;

public:
    NN_IMPLICIT NwcpUartDriver(nn::uart::PortName portName, char* pReceiveBuffer, char* pSendBuffer) NN_NOEXCEPT;
    virtual ~NwcpUartDriver() NN_NOEXCEPT NN_OVERRIDE;

    bool Activate() NN_NOEXCEPT;
    void Deactivate() NN_NOEXCEPT;

    void StartUartCommunication(nn::os::LightEventType* pEvent) NN_NOEXCEPT;

    //!< サンプル受信時に解析する処理を登録する
    void SetSampleParserFunction(InputReportParserFunc func, void* pArg) NN_NOEXCEPT;

    // デバイス情報の呼び出しまでが完了したら呼ばれるイベントを登録する
    void SetInquiryCompleteEvent(nn::os::LightEventType* pEvent) NN_NOEXCEPT;

    //!< 接続されているデバイスの情報を取得する
    Result GetDeviceInfo(::nn::bluetooth::Address* pOutAddress, NwcpDeviceType* pOutDeviceType) NN_NOEXCEPT;

    //!< Eventが通知された際に呼ばれる関数
    virtual void EventFunction(const ::nn::os::MultiWaitHolderType* pMultiWaitHolder) NN_NOEXCEPT NN_OVERRIDE;

    //!< 定常的な処理を行う関数
    virtual void PeriodicEventFunction() NN_NOEXCEPT NN_OVERRIDE;

    //!< Page を開始する
    void ConnectWithBluetooth() NN_NOEXCEPT;

    //!< 最新の受信済み HidReport を取得する
    size_t GetInputReport(uint8_t* pBuffer, size_t size) NN_NOEXCEPT;

    //!< 送信する HidReport をセットする
    Result SetOutputReport(const uint8_t* pBuffer, size_t size) NN_NOEXCEPT;

    //!< Uart 通信を終了する
    void DetachUart() NN_NOEXCEPT;

    //!< Uart の通信が確立されているかどうか
    bool IsUartAttached() NN_NOEXCEPT;

    //!< 充電を開始する
    void SetBatteryCharing(bool enabled, bool is200mA) NN_NOEXCEPT;

    //!< 電池残量を取得する
    BatteryLevel GetBatteryLevel() NN_NOEXCEPT;

    //!< 充電状態を取得する
    bool IsCharging() NN_NOEXCEPT;

    //!< 電池残量が変化した際に通知を受けるイベントオブジェクトをバインドします
    void BindBatteryLevelChangeEvent(::nn::os::LightEventType* pEventType) NN_NOEXCEPT;

    //!< 電池残量が変化した際に通知を受けるイベントオブジェクトを解除します
    void UnbindBatteryLevelChangeEvent(::nn::os::LightEventType* pEventType) NN_NOEXCEPT;

    //!< 起動要因を取得します
    void StartReadingAwakeTriggerReason(::nn::os::LightEventType* pEventType) NN_NOEXCEPT;

    //!< 起動要因の取得結果を返します。
    AwakeTriggerReason GetAwakeTriggerReason() NN_NOEXCEPT;

    //!< デバイス再起動で Uart 通信を終了する
    void DetachUartByReboot() NN_NOEXCEPT;

    //!< 通信周期を取得します
    TimeSpan GetInterval() NN_NOEXCEPT;

    //!< SetReport を送信する
    void SetReport(uint8_t* pBuffer, size_t size, IHidListener* pListener) NN_NOEXCEPT;

    //!< GetReport を送信する
    void GetReport(uint8_t reportId, IHidListener* pListener) NN_NOEXCEPT;

    //!< Firmware 更新モード用に通信モードを変更する
    void SetFirmwareUpdateModeEnabled(bool enabled) NN_NOEXCEPT;

    //!< 高速通信モードを切り替える
    void SetFastModeEnabled(bool enabled) NN_NOEXCEPT;

    //!< Fifty 用の入力状態を取得する
    PadState GetFiftyPadState() NN_NOEXCEPT;

private:
    bool OpenUart(::nn::uart::BaudRate baudRate) NN_NOEXCEPT;

    void CloseUart() NN_NOEXCEPT;

    void ReadUartData() NN_NOEXCEPT;
    void ParseUartData(uint8_t* pBuffer, size_t lenght) NN_NOEXCEPT;

    void WriteUartData(uint8_t* pBuffer, size_t size) NN_NOEXCEPT;

    void CreateUartConfig(::nn::uart::PortConfigType* pOutValue, ::nn::uart::BaudRate baudRate) NN_NOEXCEPT;

    size_t CalculatePacketSizeFull(size_t payloadSize) NN_NOEXCEPT;

    void GenerateHciHeader(uint8_t* pOutValue, size_t payloadLength) NN_NOEXCEPT;

    void GenerateNwcpPacket(uint8_t *pOutValue, size_t size, uint8_t opcode) NN_NOEXCEPT;

    uint8_t CalculateCrc8(const uint8_t* pData, size_t size) NN_NOEXCEPT;
    void AddCrcForNwcpPacket(uint8_t *pOutValue, size_t size) NN_NOEXCEPT;
    bool CheckCrcInNwcpPacket(const uint8_t *pData, size_t size) NN_NOEXCEPT;

    size_t CalculatePacketLength(uint8_t *pBuffer) NN_NOEXCEPT;

    void BindUartReceiveEvent() NN_NOEXCEPT;

    void UnbindUartReceiveEvent() NN_NOEXCEPT;

    void SendMagicPacket() NN_NOEXCEPT;

    void HandleDeviceAttach() NN_NOEXCEPT;
    void HandleDeviceDetach() NN_NOEXCEPT;

    void ResetDevice() NN_NOEXCEPT;
    void HandleResetComplete() NN_NOEXCEPT;

    void SendSync() NN_NOEXCEPT;
    void ReceivedSync() NN_NOEXCEPT;

    void SendInquiry() NN_NOEXCEPT;
    void HandleInquiryResult(uint8_t* pBuffer, size_t size) NN_NOEXCEPT;

    void UpdateUartBaudRate() NN_NOEXCEPT;
    void HandleUpdateUartBaudRateResult(uint8_t* pBuffer, size_t size) NN_NOEXCEPT;

    void QueueDeviceCommand(uint8_t* pBuffer, int length, uint8_t opCode) NN_NOEXCEPT;
    void TrySendDeviceCommandFromQueue() NN_NOEXCEPT;
    void HandleDeviceCommandTimeout() NN_NOEXCEPT;
    void ReceivedDeviceCommand(uint8_t* pBuffer, size_t size) NN_NOEXCEPT;
    void ReceivedHid(uint8_t* pBuffer, size_t size) NN_NOEXCEPT;

    void ReadWakeUpReason() NN_NOEXCEPT;
    void HandleReadWakeUpReasonResult(uint8_t* pBuffer, size_t size) NN_NOEXCEPT;

    void CreateHidConnection() NN_NOEXCEPT;
    void HandleCreateHidConnectionResult(uint8_t* pBuffer, size_t size) NN_NOEXCEPT;

    void StartPairing() NN_NOEXCEPT;
    void HandlePairingResult(uint8_t* pBuffer, size_t size) NN_NOEXCEPT;

    void DisconnectHid() NN_NOEXCEPT;
    void HandleDisconnectHidResult(uint8_t* pBuffer, size_t size) NN_NOEXCEPT;

    void HandleWriteChargerSettings(uint8_t* pBuffer, size_t size) NN_NOEXCEPT;

    void SetHidInterval() NN_NOEXCEPT;
    void HandleSetHidIntervalResult(uint8_t* pBuffer, size_t size) NN_NOEXCEPT;

    void SendHidOutput() NN_NOEXCEPT;

    void SendFiftyPoll() NN_NOEXCEPT;
    void HandleFiftyData(uint8_t* pBuffer, size_t size) NN_NOEXCEPT;

    void DisconnectHidByReboot() NN_NOEXCEPT;
};

}}} // namespace nn::xcd::detail
