﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Macro.h>
#include <nn/nn_Result.h>
#include <nn/nn_SdkAssert.h>
#include <nn/nn_SdkLog.h>
#include <nn/util/util_ScopeExit.h>
#include <nn/os/os_Event.h>
#include <nn/os/os_SystemEvent.h>
#include <nn/os/os_MultipleWait.h>
#include <nn/os/os_LightEvent.h>
#include <nn/os/os_Thread.h>
#include <nn/pinmux/pinmux.h>
#include <nn/result/result_HandlingUtility.h>

// ペアリングのための暫定呼び出し
#include <nn/xcd/xcd_Pairing.h>

#include "xcd_NwcpProtocolTypes.h"
#include "xcd_NwcpManager-os.horizon.h"
#include "../xcd_DeviceHandleGenerator.h"

namespace nn { namespace xcd { namespace detail {

namespace
{
    NN_ALIGNAS(nn::os::MemoryPageSize) char g_SendWorkBuffer[2][UartWorkBufferSize];
    NN_ALIGNAS(nn::os::MemoryPageSize) char g_ReceiveWorkBuffer[2][UartWorkBufferSize];
}

NwcpManager::NwcpManager() NN_NOEXCEPT :
    m_LeftDriver(nn::pinmux::AssignablePinGroupName_ExtConSTx,
                 nn::gpio::GpioPadName_ExtconDetS,
                 nn::gpio::GpioPadName_ExtUart3Cts,
                 nn::uart::PortName::PortName_ExtConS,
                 nn::gpio::GpioPadName_ExtconChgS,
                 g_ReceiveWorkBuffer[0],
                 g_SendWorkBuffer[0],
                 NwcpDeviceType_JoyLeft),
    m_RightDriver(nn::pinmux::AssignablePinGroupName_ExtConUTx,
                  nn::gpio::GpioPadName_ExtconDetU,
                  nn::gpio::GpioPadName_ExtUart2Cts,
                  nn::uart::PortName::PortName_ExtConU,
                  nn::gpio::GpioPadName_ExtconChgU,
                  g_ReceiveWorkBuffer[1],
                  g_SendWorkBuffer[1],
                 NwcpDeviceType_JoyRight),
     m_Activated(false)
{
    // 何もしない
}

NwcpManager::~NwcpManager() NN_NOEXCEPT
{
    // 何もしない
}

void NwcpManager::EventFunction(const ::nn::os::MultiWaitHolderType* pMultiWaitHolder) NN_NOEXCEPT
{
    NN_UNUSED(pMultiWaitHolder);
}

void NwcpManager::PeriodicEventFunction() NN_NOEXCEPT
{
    NwcpHidAccessor* pHidAccessor;
    NwcpDriver* pDriver;
    if (nn::os::TryWaitLightEvent(&m_SuspendCompleteEvent))
    {
        nn::os::ClearLightEvent(&m_SuspendCompleteEvent);
        if (m_LeftDriver.IsSuspended() && m_RightDriver.IsSuspended())
        {
            ::nn::os::SignalLightEvent(m_pSuspendCompletedEvent);
        }
        return;
    }
    else if (nn::os::TryWaitLightEvent(&m_LeftEvent))
    {
        nn::os::ClearLightEvent(&m_LeftEvent);
        pHidAccessor = &m_LeftAccessor;
        pDriver = &m_LeftDriver;
    }
    else if (nn::os::TryWaitLightEvent(&m_RightEvent))
    {
        nn::os::ClearLightEvent(&m_RightEvent);
        pHidAccessor = &m_RightAccessor;
        pDriver = &m_RightDriver;
    }
    else
    {
        return;
    }

    ::nn::bluetooth::Address address;
    NwcpDeviceType deviceType;
    if (pDriver->GetDeviceInfo(&address, &deviceType).IsSuccess())
    {
        auto handle = DeviceHandleGenerator::Get().GetDeviceHandle();
        pHidAccessor->Activate(handle, pDriver);
    }
    else
    {
        pHidAccessor->Deactivate();
    }

    nn::os::SignalLightEvent(m_pDeviceUpdateEvent);
}

void NwcpManager::StartMonitoring(nn::os::LightEventType* pEvent) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_EQUAL(m_Activated, false);
    NN_SDK_REQUIRES_NOT_NULL(pEvent);

    m_pDeviceUpdateEvent = pEvent;

    nn::os::InitializeLightEvent(&m_LeftEvent, false, nn::os::EventClearMode_ManualClear);
    nn::os::InitializeLightEvent(&m_RightEvent, false, nn::os::EventClearMode_ManualClear);

    m_LeftDriver.SetPowerSupplyManager(&m_PowerSupplyManager);
    m_RightDriver.SetPowerSupplyManager(&m_PowerSupplyManager);

    m_LeftDriver.StartMonitoring(&m_LeftEvent);
    m_RightDriver.StartMonitoring(&m_RightEvent);

    nn::os::InitializeLightEvent(&m_SuspendCompleteEvent, false, nn::os::EventClearMode_ManualClear);

    GetTaskManager().RegisterPeriodicTask(this);

    m_Activated = true;
}

void NwcpManager::StopMonitoring() NN_NOEXCEPT
{
    GetTaskManager().UnregisterPeriodicTask(this);

    nn::os::FinalizeLightEvent(&m_SuspendCompleteEvent);

    m_LeftDriver.StopMonitoring();
    m_RightDriver.StopMonitoring();

    nn::os::FinalizeLightEvent(&m_LeftEvent);
    nn::os::FinalizeLightEvent(&m_RightEvent);

    m_Activated = false;
}

size_t NwcpManager::GetDevices(HidDeviceInfo* pOutValue, size_t deviceCount) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutValue);

    size_t returnCount = 0;

    NwcpDeviceType deviceType;
    HidDeviceInfo deviceInfo;

    deviceInfo.vid = 0x57E;
    deviceInfo.pid = 0x2006;
    deviceInfo.interfaceType = InterfaceType_Uart;

    if (m_LeftDriver.GetDeviceInfo(&(deviceInfo.address), &deviceType).IsSuccess())
    {
        if (deviceType == NwcpDeviceType_JoyLeft)
        {
            deviceInfo.deviceHandle = m_LeftAccessor.GetDeviceHandle();
            pOutValue[returnCount] = deviceInfo;
            returnCount++;
        }
    }
    if (m_RightDriver.GetDeviceInfo(&(deviceInfo.address), &deviceType).IsSuccess())
    {
        if (deviceType == NwcpDeviceType_JoyRight)
        {
            deviceInfo.deviceHandle = m_RightAccessor.GetDeviceHandle();
            pOutValue[returnCount] = deviceInfo;
            returnCount++;
        }
    }

    return returnCount;
}

HidAccessor* NwcpManager::GetHidAccessor(DeviceHandle deviceHandle) NN_NOEXCEPT
{
    if (m_LeftAccessor.GetDeviceHandle() == deviceHandle && m_LeftAccessor.IsActivated())
    {
        return &m_LeftAccessor;
    }
    else if (m_RightAccessor.GetDeviceHandle() == deviceHandle && m_RightAccessor.IsActivated())
    {
        return &m_RightAccessor;
    }

    // みつからなかった
    return nullptr;
}

bool NwcpManager::IsRightNwcpAttached() NN_NOEXCEPT
{
    return m_RightDriver.IsDeviceAttached();
}

bool NwcpManager::IsLeftNwcpAttached() NN_NOEXCEPT
{
    return m_LeftDriver.IsDeviceAttached();
}

void NwcpManager::SetNwcpEnabled(bool enabled) NN_NOEXCEPT
{
    m_LeftDriver.SetNwcpEnabled(enabled);
    m_RightDriver.SetNwcpEnabled(enabled);
}

void NwcpManager::SetRailUpdateEvent(nn::os::SystemEventType* pEvent) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pEvent);
    m_LeftDriver.SetRailUpdateEvent(pEvent);
    m_RightDriver.SetRailUpdateEvent(pEvent);
}

void NwcpManager::GetRailUpdateEventType(RailUpdateEventType* pOutEventType, ::nn::bluetooth::Address* pOutAddress) NN_NOEXCEPT
{
    // 左レールのイベント状態をチェック
    m_LeftDriver.GetRailUpdateEventType(pOutEventType, pOutAddress);
    if (*pOutEventType != RailUpdateEventType_None)
    {
        return;
    }
    // 左側にイベントがなければ右レールのイベントを返す
    return m_RightDriver.GetRailUpdateEventType(pOutEventType, pOutAddress);
}

void NwcpManager::TriggerSuspend(::nn::os::LightEventType* pEvent) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pEvent);

    m_pSuspendCompletedEvent = pEvent;

    m_LeftDriver.TriggerSuspend(&m_SuspendCompleteEvent);
    m_RightDriver.TriggerSuspend(&m_SuspendCompleteEvent);
}

void NwcpManager::Resume() NN_NOEXCEPT
{
    m_LeftDriver.Resume();
    m_RightDriver.Resume();
}

bool NwcpManager::IsConsolePowered() NN_NOEXCEPT
{
    return m_PowerSupplyManager.IsPowered();
}

AwakeTriggerReason NwcpManager::GetAwakeTriggerReasonForLeftRail() NN_NOEXCEPT
{
    return m_LeftDriver.GetAwakeTriggerReason();
}

AwakeTriggerReason NwcpManager::GetAwakeTriggerReasonForRightRail() NN_NOEXCEPT
{
    return m_RightDriver.GetAwakeTriggerReason();
}

void NwcpManager::SetRebootEnabled(bool enabled) NN_NOEXCEPT
{
    m_LeftDriver.SetRebootEnabled(enabled);
    m_RightDriver.SetRebootEnabled(enabled);
}

bool NwcpManager::IsFiftyConnected() NN_NOEXCEPT
{
    ::nn::bluetooth::Address address;
    NwcpDeviceType deviceType;
    if (m_LeftDriver.GetDeviceInfo(&address, &deviceType).IsSuccess())
    {
        if (deviceType == NwcpDeviceType_Fifty)
        {
            return true;
        }
    }

    return false;
}

PadState NwcpManager::GetFiftyPadState() NN_NOEXCEPT
{
    return m_LeftDriver.GetFiftyPadState();
}

}}} // namespace nn::xcd::detail
