﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/nn_Macro.h>
#include <nn/os/os_LightEventTypes.h>
#include <nn/pinmux/pinmux.h>
#include <nn/xcd/xcd_Device.h>
#include <nn/xcd/xcd_Input.h>
#include <nn/xcd/xcd_Rail.h>
#include <nn/xcd/xcd_Sleep.h>

#include "xcd_GpioMonitorTask-os.horizon.h"
#include "xcd_IHidListener.h"
#include "xcd_NwcpCharger-os.horizon.h"
#include "xcd_NwcpUartDriver-os.horizon.h"
#include "xcd_NwcpProtocolTypes.h"
#include "xcd_PowerSupplyManager-os.horizon.h"

namespace nn { namespace xcd { namespace detail{

struct NwcpDriverFlag
{
    typedef ::nn::util::BitFlagSet<32, NwcpDriverFlag>::Flag<0> IsActivated;             //!< ドライバがアクティベートされているかどうか
    typedef ::nn::util::BitFlagSet<32, NwcpDriverFlag>::Flag<1> IsUartAllowed;           //!< Uart が許可されているかどうか
    typedef ::nn::util::BitFlagSet<32, NwcpDriverFlag>::Flag<2> IsUartUsed;              //!< Uart が有効かどうか。無効の場合は Nwcp は GPIO として動作している
    typedef ::nn::util::BitFlagSet<32, NwcpDriverFlag>::Flag<3> IsPhysicallyAttached;    //!< レール部に物理接続されているかどうか
    typedef ::nn::util::BitFlagSet<32, NwcpDriverFlag>::Flag<4> IsNwcpDeviceActivated;   //!< Nwcp のデバイスが接続され通信可能な状態にあるかどうか
    typedef ::nn::util::BitFlagSet<32, NwcpDriverFlag>::Flag<5> IsFromBoot;              //!< 起動直後のデバイス状態のチェックかどうか。起動直後の場合レールへのデバイス接続の通知を行わない
    typedef ::nn::util::BitFlagSet<32, NwcpDriverFlag>::Flag<6> ReadAwakeTriggerReason;  //!< 起動理由を読み出すかどうか
    typedef ::nn::util::BitFlagSet<32, NwcpDriverFlag>::Flag<7> IsInquiryCompleted;      //!< Inquiry 処理が完了しているかどうか
    typedef ::nn::util::BitFlagSet<32, NwcpDriverFlag>::Flag<8> IsRebootEnabled;         //!< 切断時再起動が有効かどうか
    typedef ::nn::util::BitFlagSet<32, NwcpDriverFlag>::Flag<9> IsNwcpEnabled;           //!< Nwcp が有効かどうか。
};


/**
 * @brief       NwcpDriver で扱うフラグの集合です
 */
typedef ::nn::util::BitFlagSet<32, NwcpDriverFlag> NwcpDriverFlagSet;

//!< デバイスの接続状態の監視を行うためのプラットフォームごとの実装クラス
class NwcpDriver final : public IEventTask
{
    NN_DISALLOW_MOVE(NwcpDriver);
    NN_DISALLOW_COPY(NwcpDriver);

private:
    //!< 接続を許可するデバイスタイプ
    NwcpDeviceType m_SupportedDeviceType;

    //!< 接続中のデバイスの種類
    NwcpDeviceType m_CurrentDeviceType;

    // Detect Gpio の状態変化があった際に通知されるイベント
    nn::os::LightEventType m_DetectGpioEvent;

    // Uart Cts Gpio の状態変化があった際に通知されるイベント
    nn::os::LightEventType m_UartCtsGpioEvent;

    // Uart の Inquiry (デバイスタイプの検出処理) が完了したら通知されるイベント
    nn::os::LightEventType m_InquiryCompleteEvent;

    // Cts の変化に対するタイムアウトイベント
    nn::os::TimerEventType m_UartCtsTimeoutEvent;

    // Uart の状態変化があった際に通知されるイベント
    nn::os::LightEventType m_UartEvent;

    // Devie の Detach 待ちイベント
    nn::os::TimerEventType m_UartDetachTimeoutEvent;

    // デバイスの接続状態に変化があった場合に通知するためのイベント
    nn::os::LightEventType* m_pDeviceUpdateEvent;

    //!< Pinmux 名
    nn::pinmux::AssignablePinGroupName m_PinmuxName;

    //!< Detact Gpio 監視
    GpioMonitorTask m_DetectGpioMonitor;

    //!< UART CTS Gpio 監視
    GpioMonitorTask m_UartCtsGpioMonitor;

    //!< Uart 制御
    NwcpUartDriver m_UartDriver;

    //!< 充電制御
    NwcpCharger m_Charger;

    //!< 電源制御
    PowerSupplyManager* m_pPowerSupplyManager;

    //!< Nwcp ドライバ内で利用するフラグの集合
    NwcpDriverFlagSet m_Flags;

    //!< 接続中デバイスのアドレス
    ::nn::bluetooth::Address m_Address;

    //!< サスペンド処理が完了した際に通知を行うイベント
    ::nn::os::LightEventType* m_pSuspendCompleteEvent;

    //!< AwakeTriggerReason の読み出しが完了した際に通知するイベント
    ::nn::os::LightEventType m_AwakeTriggerReasonReadCompleteEvent;

    //!< レールの接続状態に変化があった場合に通知するイベント
    ::nn::os::SystemEventType* m_pRailUpdateEvent;

    //!< レールの接続状態のイベントキューの深さ
    static const int RailUpdateEventQueueSize = 2;

    struct RailUpdateEventUnit
    {
        RailUpdateEventType eventType;
        ::nn::bluetooth::Address address;
    };

    //!< レールの接続状態のイベントの内容のキュー
    RailUpdateEventUnit m_RailUpdateEventQueue[RailUpdateEventQueueSize];

    enum SuspendState
    {
        SuspendState_Resumed,        //!< Resume されている
        SuspendState_Suspending,     //!< Suspend 処理中
        SuspendState_Suspended,      //!< Suspend されている
    };

    // Suspend/Resume の状態
    SuspendState m_SuspendState;

public:
    NwcpDriver(nn::pinmux::AssignablePinGroupName pinmuxName,
               nn::gpio::GpioPadName detectGpioPadName,
               nn::gpio::GpioPadName uartCtsGpioPadName,
               nn::uart::PortName uartPortName,
               nn::gpio::GpioPadName chargerGpioPadName,
               char* pUartRecivedBuffer,
               char* pUartSendBuffer,
               NwcpDeviceType deviceType) NN_NOEXCEPT;
    virtual ~NwcpDriver() NN_NOEXCEPT NN_OVERRIDE;

    //!< Eventが通知された際に呼ばれる関数
    virtual void EventFunction(const ::nn::os::MultiWaitHolderType* pMultiWaitHolder) NN_NOEXCEPT NN_OVERRIDE;

    //!< 定常的な処理を行う関数
    virtual void PeriodicEventFunction() NN_NOEXCEPT NN_OVERRIDE;

    //!< PowerSupplyManager をセットする
    void SetPowerSupplyManager(PowerSupplyManager* pManager) NN_NOEXCEPT;

    //!< デバイスの接続状態の監視を開始する
    void StartMonitoring(nn::os::LightEventType* pDeviceUpdateEvent) NN_NOEXCEPT;

    //!< デバイスの接続状態の監視を停止する
    void StopMonitoring() NN_NOEXCEPT;

    //!< サンプル受信時に解析する処理を登録する
    void SetSampleParserFunction(InputReportParserFunc func, void* pArg) NN_NOEXCEPT;

    //!< 接続されているデバイスの情報を取得する
    Result GetDeviceInfo(::nn::bluetooth::Address* pOutAddress, NwcpDeviceType* pOutDeviceType) NN_NOEXCEPT;

    //!< 最新の受信済み HidReport を取得する
    size_t GetInputReport(uint8_t* pBuffer, size_t size) NN_NOEXCEPT;

    //!< 送信する HidReport をセットする
    Result SetOutputReport(const uint8_t* pBuffer, size_t size) NN_NOEXCEPT;

    //!< デバイスが接続されているかどうかを取得する
    bool IsDeviceAttached() NN_NOEXCEPT;

    //!< レール部の Uart 通信の有効無効状態を変更する
    void SetNwcpEnabled(bool enabled) NN_NOEXCEPT;

    //!< ジョイコンのレール部の接続状態に変化があった場合に通知するイベントを登録します
    void SetRailUpdateEvent(nn::os::SystemEventType* pEvent) NN_NOEXCEPT;

    //!< ジョイコンのレール部の接続状態に変化について、変化の内容を取得します
    void GetRailUpdateEventType(RailUpdateEventType* pOutEventType, ::nn::bluetooth::Address* pOutAddressRailUpdateEventType) NN_NOEXCEPT;

    //!< NwcpDriver の処理を Suspend 処理を開始します。サスペンドは非同期で動作します。
    void TriggerSuspend(::nn::os::LightEventType* pEvent) NN_NOEXCEPT;

    //!< NwcpDriver の処理を Resume します。同期的に完了します
    void Resume() NN_NOEXCEPT;

    //!< NwcpDriver が Suspend 状態になったかどうかを確認します。
    bool IsSuspended() NN_NOEXCEPT;

    //!< スリープ復帰要因を取得します
    AwakeTriggerReason GetAwakeTriggerReason() NN_NOEXCEPT;

    //!< 切断時再起動を有効にする
    void SetRebootEnabled(bool enabled) NN_NOEXCEPT;

    //!< 通信周期を取得する
    ::nn::TimeSpan GetInterval() NN_NOEXCEPT;

    //!< SetReport を送信する
    void SetReport(uint8_t* pBuffer, size_t size, IHidListener* pListener) NN_NOEXCEPT;

    //!< GetReport を送信する
    void GetReport(uint8_t reportId, IHidListener* pListener) NN_NOEXCEPT;

    //!< Firmware 更新モード用に通信モードを変更する
    void SetFirmwareUpdateModeEnabled(bool enabled) NN_NOEXCEPT;

    //!< 高速通信モードを切り替える
    void SetFastModeEnabled(bool enabled) NN_NOEXCEPT;

    //!< Fifty の入力状態を取得します
    PadState GetFiftyPadState() NN_NOEXCEPT;

private:
    //!< GPIO 検出時に呼ばれる API
    void UpdateGpioState() NN_NOEXCEPT;

    //!< Uartの状態の変化が起きた際に呼ばれる API
    void UpdateUartState() NN_NOEXCEPT;

    //!< Uart の Cts 変化待ちのタイムアウトを処理します
    void HandleUartCtsTimeout() NN_NOEXCEPT;

    //!< Device の Detach 待ちタイムアウトを処理します
    void HandleUartDetachTimeout() NN_NOEXCEPT;

    //!< Uart の Cts 変化を処理します
    void HandleUartCtsUpdate() NN_NOEXCEPT;

    //!< Inquiry 完了した際の処理
    void HandleInquiryComplete() NN_NOEXCEPT;

    //!< Gpio の監視を有効にする
    void EnableGpioMonitoring() NN_NOEXCEPT;

    //!< デバイスとの通信を切断する
    void Detach() NN_NOEXCEPT;

    //!< スリープに入れるかどうかを確認して、可能であればスリープにいれる
    void TryIsCtsLow() NN_NOEXCEPT;

    //!< pinmux の切り替えを行う
    void SwitchPinmux(bool useUart) NN_NOEXCEPT;

    //!< Suspend の最終版に行う共通処理
    void CompleteSuspend() NN_NOEXCEPT;

    //!< レール接続部の変化をイベントキューに追加する
    void QueueNwcpAttachEvent(bool attach, ::nn::bluetooth::Address address) NN_NOEXCEPT;
};

}}} // namespace nn::xcd::detail
