﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Macro.h>
#include <nn/nn_Result.h>
#include <nn/nn_SdkAssert.h>
#include <nn/gpio/gpio.h>
#include <nn/os/os_LightEvent.h>
#include <nn/os/os_SystemEvent.h>
#include <nn/os/os_MultipleWait.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/settings/factory/settings_ConfigurationId.h>
#include <nn/util/util_ScopeExit.h>
#include <nn/util/util_StringView.h>
#include <nn/xcd/xcd_ResultForPrivate.h>
#include <nn/xcd/detail/xcd_Log.h>

#include "xcd_NwcpCharger-os.horizon.h"

namespace nn { namespace xcd { namespace detail {

NwcpCharger::NwcpCharger(nn::gpio::GpioPadName padName) NN_NOEXCEPT :
    m_PadName(padName),
    m_IsCharging(false),
    m_IsPowerSupplyEnabled(false),
    m_IsChargerManagementEnabled(false),
    m_IsSdev(false)
{
    // 何もしない
}

NwcpCharger::~NwcpCharger() NN_NOEXCEPT
{
    // 何もしない
}

void NwcpCharger::SetPowerSupplyManager(PowerSupplyManager* pManager) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pManager);
    m_pPowerSupplyManager = pManager;
}

void NwcpCharger::SetNwcpUartDriver(NwcpUartDriver* pDriver) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pDriver);
    m_pUartDriver = pDriver;
}

Result NwcpCharger::Activate() NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(m_pPowerSupplyManager);
    NN_SDK_REQUIRES_NOT_NULL(m_pUartDriver);
    NN_RESULT_THROW_UNLESS(!m_ActivationCount.IsMax(), ResultActivationUpperLimitOver());

    if (m_ActivationCount.IsZero())
    {
        // 対象ハードウェアが SDEV かどうかをチェック
        settings::factory::ConfigurationId1 configurationId1;
        settings::factory::GetConfigurationId1(&configurationId1);
        m_IsSdev = (nn::util::string_view(configurationId1.string).substr(0, 4) == "SDEV");

        m_pPowerSupplyManager->Activate();

        nn::os::InitializeLightEvent(&m_PowerSupplyEvent, false, nn::os::EventClearMode_ManualClear);
        m_pPowerSupplyManager->BindPowerStateChangeEvent(&m_PowerSupplyEvent);

        nn::os::InitializeLightEvent(&m_BatteryLevelEvent, false, nn::os::EventClearMode_ManualClear);
        m_pUartDriver->BindBatteryLevelChangeEvent(&m_BatteryLevelEvent);

        GetTaskManager().RegisterPeriodicTask(this);
    }
    ++m_ActivationCount;

    NN_RESULT_SUCCESS;
}

Result NwcpCharger::Deactivate() NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(m_pPowerSupplyManager);
    NN_RESULT_THROW_UNLESS(!m_ActivationCount.IsZero(), ResultActivationLowerLimitOver());

    --m_ActivationCount;

    if (m_ActivationCount.IsZero())
    {
        GetTaskManager().UnregisterPeriodicTask(this);

        m_pPowerSupplyManager->UnbindPowerStateChangeEvent(&m_PowerSupplyEvent);
        nn::os::FinalizeLightEvent(&m_PowerSupplyEvent);

        m_pUartDriver->UnbindBatteryLevelChangeEvent(&m_BatteryLevelEvent);
        nn::os::FinalizeLightEvent(&m_BatteryLevelEvent);

        m_pPowerSupplyManager->Deactivate();
    }

    NN_RESULT_SUCCESS;
}

void NwcpCharger::EventFunction(const ::nn::os::MultiWaitHolderType* pMultiWaitHolder) NN_NOEXCEPT
{
    NN_UNUSED(pMultiWaitHolder);
}

void NwcpCharger::PeriodicEventFunction() NN_NOEXCEPT
{
    if (nn::os::TryWaitLightEvent(&m_PowerSupplyEvent))
    {
        nn::os::ClearLightEvent(&m_PowerSupplyEvent);
        UpdateChargeState();
        return;
    }

    if (nn::os::TryWaitLightEvent(&m_BatteryLevelEvent))
    {
        nn::os::ClearLightEvent(&m_BatteryLevelEvent);
        UpdateChargeState();
        return;
    }
}

void NwcpCharger::StartPowerSupply() NN_NOEXCEPT
{
    // 強制的に給電・充電を開始する
    ControlPowerSupply(true);
    ControlPowerSwitch(true);
}

void NwcpCharger::StartChargeStateManagement() NN_NOEXCEPT
{
    m_IsChargerManagementEnabled = true;
    UpdateChargeState();
}

void NwcpCharger::StopPowerSupply() NN_NOEXCEPT
{
    // 強制的に給電・充電を終了する
    ControlPowerSupply(false);
    ControlPowerSwitch(false);
}

void NwcpCharger::StopChargeStateManagement() NN_NOEXCEPT
{
    m_IsChargerManagementEnabled = false;
}

void NwcpCharger::ResetChargeState() NN_NOEXCEPT
{
    // 充電している状態で初期化する
    // 本体非給電状態でジョイコン装着時に電池残量が BatteryLevel_Low ならば充電させるため
    m_IsCharging = true;
}

void NwcpCharger::UpdateChargeState() NN_NOEXCEPT
{
    if (m_IsChargerManagementEnabled)
    {
        // 給電状態の取得。SDEV の場合は常時給電状態とみなす
        auto isPowered = (m_IsSdev) ? true : m_pPowerSupplyManager->IsPowered();
        // 電池残量の取得
        auto batteryLevel = m_pUartDriver->GetBatteryLevel();

        bool enablePowerSupply = isPowered;
        enablePowerSupply = (enablePowerSupply || IsPowerSupplyRequiredOnNonPoweredState(batteryLevel));

        ControlChargeState(enablePowerSupply);
    }
}

void NwcpCharger::ControlPowerSupply(bool supplyEnabled) NN_NOEXCEPT
{
    NN_DETAIL_XCD_INFO("%s PowerSupply Port:%d\n", (supplyEnabled) ? "Enable" : "Disable", m_PadName);

    // PowerSupplyManager は Enable されたカウントを持っているため、Enable 済みの場合は Enable しない
    if (supplyEnabled)
    {
        if (!m_IsPowerSupplyEnabled)
        {
            m_IsPowerSupplyEnabled = true;
            m_pPowerSupplyManager->EnablePowerSupply();
        }
    }
    else
    {
        if (m_IsPowerSupplyEnabled)
        {
            m_IsPowerSupplyEnabled = false;
            m_pPowerSupplyManager->DisablePowerSupply();
        }
    }
}

void NwcpCharger::ControlChargeState(bool chargeEnabled) NN_NOEXCEPT
{
    NN_DETAIL_XCD_INFO("%s Charging Port:%d\n", (chargeEnabled) ? "Enable" : "Disable", m_PadName);
    ControlPowerSupply(chargeEnabled);
    ControlPowerSwitch(chargeEnabled);
    m_IsCharging = chargeEnabled;
}

void NwcpCharger::ControlPowerSwitch(bool supplyEnabled) NN_NOEXCEPT
{
    NN_DETAIL_XCD_INFO("%s PowerSwitch Port:%d\n", (supplyEnabled) ? "Enable" : "Disable", m_PadName);
    ::nn::gpio::GpioPadSession session;
    ::nn::gpio::Initialize();
    ::nn::gpio::OpenSession(&session, m_PadName);
    ::nn::gpio::SetDirection(&session, nn::gpio::Direction_Output);
    ::nn::gpio::SetValue(&session, (supplyEnabled) ? nn::gpio::GpioValue_High : nn::gpio::GpioValue_Low);
    ::nn::gpio::SetValueForSleepState(&session, (supplyEnabled) ? nn::gpio::GpioValue_High : nn::gpio::GpioValue_Low);
    ::nn::gpio::CloseSession(&session);
    ::nn::gpio::Finalize();
}

void NwcpCharger::Resume() NN_NOEXCEPT
{
}

void NwcpCharger::Suspend() NN_NOEXCEPT
{
    // この関数は接続されているときしか呼ばれない

    // 給電状態の取得。SDEV の場合は常時給電状態とみなす
    auto isPowered = (m_IsSdev) ? true : m_pPowerSupplyManager->IsPowered();
    // 電池残量の取得
    auto batteryLevel = m_pUartDriver->GetBatteryLevel();
    // 充電状態の取得
    auto isCharging = m_pUartDriver->IsCharging();

    bool enablePowerSupply = false;

    if (isPowered)
    {
        // 本体給電中 && 電池残量 Max && 充電なし は、満充電状態のため、電源を止める
        if (batteryLevel != BatteryLevel_High || isCharging == true)
        {
            enablePowerSupply = true;
        }
    }
    else

    {
        enablePowerSupply = IsPowerSupplyRequiredOnNonPoweredState(batteryLevel);
    }

    // Attach されている限りはポートはオープンにする
    ControlPowerSwitch(true);

    NN_DETAIL_XCD_INFO("%s Charging For Sleep -> Port:%d\n", (enablePowerSupply) ? "Enable" : "Disable", m_PadName);
    ControlPowerSupply(enablePowerSupply);
    if (enablePowerSupply)
    {
        m_pUartDriver->SetBatteryCharing(true, true);
    }
}

void NwcpCharger::SuspendLow() NN_NOEXCEPT
{
    if (m_IsPowerSupplyEnabled)
    {
        // gpio が Suspend するタイミングで給電を開始することにより
        // 満充電割り込みを取り逃さないようにする
        ::nn::gpio::GpioPadSession session;
        ::nn::gpio::Initialize();
        ::nn::gpio::OpenSession(&session, m_PadName);
        ::nn::gpio::SetDirection(&session, nn::gpio::Direction_Output);
        ::nn::gpio::SetValue(&session, nn::gpio::GpioValue_Low);
        ::nn::gpio::SetValueForSleepState(&session, nn::gpio::GpioValue_High);
        ::nn::gpio::CloseSession(&session);
        ::nn::gpio::Finalize();
    }
}

bool NwcpCharger::IsPowerSupplyRequiredOnNonPoweredState(BatteryLevel batteryLevel) NN_NOEXCEPT
{
    // 本体給電状態じゃない場合
    // CriticalLow 以下だったら必ず充電
    // 充電中かつ BatteryLevel_Low の場合は充電を継続
    if (batteryLevel <= BatteryLevel_CriticalLow ||
        (batteryLevel == BatteryLevel_Low && m_IsCharging))
    {
        return true;
    }
    return false;
}

}}} // namespace nn::xcd::detail
