﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <memory>
#include <tchar.h>
#include <nn/nn_Macro.h>
#include <nn/nn_Result.h>
#include <nn/nn_SdkAssert.h>
#include <nn/nn_Windows.h>
#include <nn/util/util_ScopeExit.h>
#include <nn/os/os_Event.h>
#include <nn/os/os_LightEvent.h>
#include <nn/os/os_TimerEvent.h>
#include <nn/os/os_MultipleWait.h>
#include <nn/os/os_Thread.h>
#include <nn/result/result_HandlingUtility.h>
#include "xcd_AardvarkAccessor-os.win32.h"
#include "xcd_LinkMonitorImpl-os.win32.h"
#include "../xcd_DeviceHandleGenerator.h"
#include "aardvark/aardvark-os.win32.h"

#pragma warning(push)
#pragma warning(disable: 4668)

extern "C" {
#include <hidsdi.h>
#include <SetupAPI.h>
#include <Dbt.h>
}

#pragma warning(pop)

namespace nn { namespace xcd { namespace detail {

namespace

{
    //!< Win32 のHidは未接続のデバイスも列挙してしまうためVID/PIDフィルタリングを行う
    const int JoyVid = 0x57E;
    const int JoyPids[] = {0x2006, 0x2007, 0x2009};
    bool CheckPid(int pid)
    {
        for(auto& xcdPid : JoyPids)
        {
            if(pid == xcdPid)
            {
                return true;
            }
        }
        return false;
    }

    //!< SetupDiGetDeviceInterfaceDetail に渡すバッファのサイズ。Winでは最大長は定義されていないため実績に基づく最大値
    const int DetailDataBufferSizeMax = 512;
    //!< SetupDiGetDeviceInterfaceDetail に渡すバッファ
    char g_DetailDataBuffer[DetailDataBufferSizeMax];
}

LinkMonitorImpl::LinkMonitorImpl() NN_NOEXCEPT
    : m_Activated(false),
    m_IsAardvarkDetected(false)
{
    // デバイス一覧を初期化
    for(auto& descryptor : m_Devices)
    {
        ZeroMemory(descryptor.devicePath, PathLengthMax);
    }
}

LinkMonitorImpl::~LinkMonitorImpl() NN_NOEXCEPT
{
    // 何もしない
}

void LinkMonitorImpl::StartMonitoring(nn::os::LightEventType* pUpdateEvent) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pUpdateEvent);
    NN_SDK_REQUIRES_EQUAL(m_Activated, false);

    m_pUpdatedEvent = pUpdateEvent;

    // Aardvark 開始
    aardvark::Initialize();

    HidD_GetHidGuid(&m_HidGuid);

    // 多重待ちを初期化
    nn::os::InitializeMultiWait(&m_MultiWait);

    // 終了用イベントの初期化
    nn::os::InitializeEvent(&m_TerminateEvent, false, nn::os::EventClearMode_ManualClear);
    nn::os::InitializeMultiWaitHolder(&m_TerminateHolder, &m_TerminateEvent);
    nn::os::LinkMultiWaitHolder(&m_MultiWait, &m_TerminateHolder);

    // 定期監視用イベントの初期化
    nn::os::InitializeTimerEvent(&m_TimerEvent, nn::os::EventClearMode_ManualClear);
    nn::os::InitializeMultiWaitHolder(&m_TimerHolder, &m_TimerEvent);
    nn::os::LinkMultiWaitHolder(&m_MultiWait, &m_TimerHolder);

    // デバイスの接続状態監視用のスレッドを生成
    nn::os::CreateThread(&m_Thread,
                         LinkMonitorImpl::ThreadFunction,
                         this,
                         m_ThreadStack,
                         LinkThreadStackSize,
                         nn::os::DefaultThreadPriority); // TODO
    nn::os::StartThread(&m_Thread);

    nn::os::StartPeriodicTimerEvent(&m_TimerEvent, 0, nn::TimeSpan::FromMilliSeconds(100));

    m_Activated = true;
}

void LinkMonitorImpl::StopMonitoring() NN_NOEXCEPT
{
    nn::os::SignalEvent(&m_TerminateEvent);
    nn::os::WaitThread(&m_Thread);
    nn::os::DestroyThread(&m_Thread);

    nn::os::UnlinkMultiWaitHolder(&m_TerminateHolder);
    nn::os::FinalizeEvent(&m_TerminateEvent);

    nn::os::UnlinkMultiWaitHolder(&m_TimerHolder);
    nn::os::FinalizeTimerEvent(&m_TimerEvent);

    nn::os::FinalizeMultiWait(&m_MultiWait);

    // Aardvark 終了
    aardvark::Finalize();

    m_Activated = false;
}

void LinkMonitorImpl::Suspend() NN_NOEXCEPT
{
    // 何もしない
}

void LinkMonitorImpl::Resume() NN_NOEXCEPT
{
    // 何もしない
}

void LinkMonitorImpl::ThreadFunction(void* arg) NN_NOEXCEPT
{
    reinterpret_cast<LinkMonitorImpl*>(arg)->ThreadFunctionImpl();
}

void LinkMonitorImpl::ThreadFunctionImpl() NN_NOEXCEPT
{
    while (NN_STATIC_CONDITION(true))
    {
        auto pHolder = nn::os::WaitAny(&m_MultiWait);

        if(pHolder == &m_TerminateHolder)
        {
            break;
        }
        else if(pHolder == &m_TimerHolder)
        {
            nn::os::ClearTimerEvent(&m_TimerEvent);
            ScanDevices();
        }
    }
}


void LinkMonitorImpl::ScanDevices() NN_NOEXCEPT
{
    HANDLE deviceInfoSet;
    uint8_t deviceCount = 0;
    TCHAR scannedDevices[HidDeviceCountMax * 2][PathLengthMax];
    bool stateChanged = false;

    deviceInfoSet = SetupDiGetClassDevs(&m_HidGuid,
                                        NULL,
                                        NULL,
                                        DIGCF_PRESENT | DIGCF_DEVICEINTERFACE);

    SP_DEVICE_INTERFACE_DATA deviceInterfaceData;
    deviceInterfaceData.cbSize = sizeof(deviceInterfaceData);

    for(DWORD deviceNumber = 0; ; ++deviceNumber)
    {
        BOOL result;

        result = SetupDiEnumDeviceInterfaces(deviceInfoSet,
                                             0,
                                             &m_HidGuid,
                                             deviceNumber,
                                             &deviceInterfaceData);

        //デバイスがこれ以上存在しない
        if(result == false)
        {
            break;
        }

        auto detailData = reinterpret_cast<PSP_DEVICE_INTERFACE_DETAIL_DATA>(g_DetailDataBuffer);

        detailData->cbSize = sizeof(SP_DEVICE_INTERFACE_DETAIL_DATA);

        result = SetupDiGetDeviceInterfaceDetail(deviceInfoSet,
                                                 &deviceInterfaceData,
                                                 detailData,
                                                 DetailDataBufferSizeMax,
                                                 NULL,
                                                 NULL);

        //デバイスの内容取得に失敗したら中止
        if(result == false)
        {
            continue;
        }

        _tcscpy_s(scannedDevices[deviceCount], PathLengthMax, detailData->DevicePath);
        deviceCount++;
    }

    SetupDiDestroyDeviceInfoList(deviceInfoSet);

    for(auto& descryptor : m_Devices)
    {
        if(descryptor.GetHidAccessor().IsActivated() == false)
        {
            continue;
        }

        bool registered = false;
        for(int i = 0; i < deviceCount; i++)
        {
            if(_tcscmp(scannedDevices[i], descryptor.devicePath) == 0)
            {
                registered = true;
                break;
            }
        }

        // Aardvark は切断判定をしない
        if (_tcscmp(AardvarkDevicePath, descryptor.devicePath) == 0)
        {
            registered = true;
        }

        // デバイスが切断された
        if(registered == false)
        {
            RemoveDevice(descryptor);
            stateChanged = true;
        }
    }

    for(int i = 0; i < deviceCount; i++)
    {
        bool registered = false;

        for(auto& descryptor : m_Devices)
        {
            if(_tcscmp(scannedDevices[i], descryptor.devicePath) == 0)
            {
                registered = true;
                break;
            }
        }

        if(registered == false)
        {
            // デバイスが接続された
            if(RegisterDevice(scannedDevices[i]) == true)
            {
                stateChanged = true;
            }
        }
    }

    // Aardvark の検出
    stateChanged |= ScanAardvark();

    if(stateChanged == true)
    {
        // 上位のデバイス情報の更新を通知
        nn::os::SignalLightEvent(m_pUpdatedEvent);
    }
}

bool LinkMonitorImpl::ScanAardvark() NN_NOEXCEPT
{
    if (!aardvark::IsInitialized() || m_IsAardvarkDetected)
    {
        return false;
    }

    auto aardvarkHandle = aardvark::Open(0);
    if (aardvarkHandle < 0)
    {
        // Aardvark 未接続
        return false;
    }

    for (auto& descryptor : m_Devices)
    {
        if (descryptor.GetHidAccessor().IsActivated())
        {
            continue;
        }

        // とりあえず VID をダミー値、PID を Aardvark のハンドルにしておく
        auto hid = HidDeviceInfo();
        hid.deviceHandle = DeviceHandleGenerator::Get().GetDeviceHandle();
        hid.vid = AardvarkVendorId;
        hid.pid = aardvarkHandle;
        hid.interfaceType = InterfaceType_Bluetooth;
        descryptor.deviceInfo = hid;

        // Aardvark として有効化
        descryptor.aardvarkAccessor.Activate(hid.deviceHandle, aardvarkHandle);
        _tcscpy_s(descryptor.devicePath, PathLengthMax, AardvarkDevicePath);
        m_IsAardvarkDetected = true;
        break;
    }

    return m_IsAardvarkDetected;
}

bool LinkMonitorImpl::RegisterDevice(LPCTSTR devicePath) NN_NOEXCEPT
{
    HANDLE hidHandle = CreateFile(devicePath,
                                  GENERIC_READ | GENERIC_WRITE,
                                  FILE_SHARE_READ | FILE_SHARE_WRITE,
                                  static_cast<LPSECURITY_ATTRIBUTES>(NULL),
                                  OPEN_EXISTING,
                                  0,
                                  NULL);

    HIDD_ATTRIBUTES attributes;
    attributes.Size = sizeof(attributes);
    bool result = false;

    // デバイスを登録できなかった場合はハンドルを閉じる
    NN_UTIL_SCOPE_EXIT
    {
        if(result == false)
        {
            CloseHandle(hidHandle);
        }
    };

    //attributes の取得に失敗していたら中止
    if(HidD_GetAttributes(hidHandle, &attributes) == false)
    {
        return result;
    }
    if(attributes.VendorID != JoyVid)
    {
        return result;
    }

    if(CheckPid(attributes.ProductID) == false)
    {
        return result;
    }

    // リストに追加
    for(auto& descryptor : m_Devices)
    {
        if(descryptor.GetHidAccessor().IsActivated() == false)
        {
            descryptor.deviceInfo.deviceHandle = DeviceHandleGenerator::Get().GetDeviceHandle();
            descryptor.deviceInfo.vid = attributes.VendorID;
            descryptor.deviceInfo.pid = attributes.ProductID;
            descryptor.deviceInfo.interfaceType = InterfaceType_Bluetooth;
            descryptor.GetHidAccessor().Activate(descryptor.deviceInfo.deviceHandle, hidHandle);
            _tcscpy_s(descryptor.devicePath, PathLengthMax, devicePath);
            result = true;
            return result;
        }
    }

    return result;
}

void LinkMonitorImpl::RemoveDevice(HidDeviceDescryptor& descryptor) NN_NOEXCEPT
{
    ZeroMemory(descryptor.devicePath, PathLengthMax);
    descryptor.GetHidAccessor().Deactivate();
}

size_t LinkMonitorImpl::GetDevices(HidDeviceInfo* pOutValue, size_t deviceCount) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutValue);

    size_t returnCount = 0;

    for(auto& descryptor : m_Devices)
    {
        if(descryptor.GetHidAccessor().IsActivated() == true)
        {
            pOutValue[returnCount] = descryptor.deviceInfo;
            returnCount++;

            if(returnCount == deviceCount)
            {
                break;
            }
        }
    }

    return returnCount;
}

HidAccessor* LinkMonitorImpl::GetHidAccessor(DeviceHandle deviceHandle) NN_NOEXCEPT
{
    for(int i = 0; i < HidDeviceCountMax; i++)
    {
        if(m_Devices[i].GetHidAccessor().GetDeviceHandle() == deviceHandle)
        {
            return &m_Devices[i].GetHidAccessor();
        }
    }

    // みつからなかった
    return nullptr;
}

int LinkMonitorImpl::GetMaxBluetoothLinks() NN_NOEXCEPT
{
    return 7;
}

}}} // namespace nn::xcd::detail
