﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <vector>
#include <nn/nn_Common.h>
#include <nn/nn_Macro.h>
#include <nn/os/os_LightEventTypes.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/xcd/xcd_Device.h>
#include <nn/xcd/xcd_Usb.h>
#include <nn/xcd/xcd_Rail.h>
#include <nn/xcd/xcd_Sleep.h>
#include <nn/xcd/xcd_KuinaFirmware.h>
#include "xcd_BleHidAccessor.h"
#include "xcd_LinkMonitorTypes.h"
#include "xcd_HidAccessor.h"
#include "../xcd_TaskManager.h"

namespace nn { namespace xcd { namespace detail{

//!< デバイスの接続状態の監視を行うためのプラットフォームごとの基底クラス
class ILinkMonitorImpl
{
    NN_DISALLOW_MOVE(ILinkMonitorImpl);
    NN_DISALLOW_COPY(ILinkMonitorImpl);

public:
    ILinkMonitorImpl() NN_NOEXCEPT { /* 何もしない */ };
    virtual ~ILinkMonitorImpl() NN_NOEXCEPT { /* 何もしない */ };

    //!< デバイスの接続状態の監視を開始する
    virtual void StartMonitoring(nn::os::LightEventType* pUpdateEvent) NN_NOEXCEPT
    {
        NN_UNUSED(pUpdateEvent);
        // 何もしない
    };

    //!< デバイスの接続状態の監視を停止する
    virtual void StopMonitoring() NN_NOEXCEPT
    {
        // 何もしない
    };

    //!< Suspend します。
    virtual void Suspend() NN_NOEXCEPT
    {
        // 何もしない
    };

    //!< Resume します。
    virtual void Resume() NN_NOEXCEPT
    {
        // 何もしない
    };

    //!< 接続済みデバイスのハンドルを取得する
    virtual size_t GetDevices(HidDeviceInfo *pOutValue, size_t deviceCount) NN_NOEXCEPT
    {
        NN_UNUSED(pOutValue);
        NN_UNUSED(deviceCount);
        // デフォルト環境では接続デバイスは常時 0
        return 0;
    };

    //!< 接続済みデバイスのハンドルを取得する
    virtual size_t GetBleDevices(uint32_t *pOutValue, size_t deviceCount) NN_NOEXCEPT
    {
        NN_UNUSED(pOutValue);
        NN_UNUSED(deviceCount);
        // デフォルト環境では接続デバイスは常時 0
        return 0;
    };

    //!< デバイスハンドルからHIDAccessorを取得する
    virtual HidAccessor* GetHidAccessor(DeviceHandle deviceHandle) NN_NOEXCEPT
    {
        NN_UNUSED(deviceHandle);
        // デフォルト環境ではハンドルの取得ができない
        return nullptr;
    };

    //!< ハンドルから BleHidAccessor を取得する
    virtual BleHidAccessor* GetBleHidAccessor(uint32_t connectionHandle) NN_NOEXCEPT
    {
        NN_UNUSED(connectionHandle);
        // デフォルト環境ではハンドルの取得ができない
        return nullptr;
    };

    //!< SlotSize の変更が起きた際に通知されるイベントを登録します
    virtual void SetSlotSizeUpdateEvent(DeviceHandle deviceHandle, ::nn::os::LightEventType* pEvent) NN_NOEXCEPT
    {
        NN_UNUSED(deviceHandle);
        NN_UNUSED(pEvent);
        // 何もしない
    };

    //!< Bluetooth で使用するスロットサイズを設定します。
    virtual Result SetSlotSize(DeviceHandle deviceHandle, int slotSize) NN_NOEXCEPT
    {
        NN_UNUSED(deviceHandle);
        NN_UNUSED(slotSize);
        // 何もしない
        NN_RESULT_SUCCESS;
    };

    //!< Bluetooth で使用されている現在のスロットサイズを取得します。スロットサイズ変更に対応しないプラットフォームでは 0 を返します
    virtual int GetSlotSize(DeviceHandle deviceHandle) NN_NOEXCEPT
    {
        NN_UNUSED(deviceHandle);
        // スロットサイズ変更に対応していない場合は常時0を返す
        return 0;
    };

    //!< Bluetooth で使用する変更先のスロットサイズの値を取得します。スロットサイズ変更に対応しないプラットフォームでは 0 を返します
    virtual int GetTargetSlotSize(DeviceHandle deviceHandle) NN_NOEXCEPT
    {
        NN_UNUSED(deviceHandle);
        // スロットサイズ変更に対応していない場合は常時0を返す
        return 0;
    };

    //!< Bluetooth で使用するスロットサイズを変更中かどうか返します
    virtual bool IsChangingSlotSize(DeviceHandle deviceHandle) NN_NOEXCEPT
    {
        NN_UNUSED(deviceHandle);
        return false;
    };

    //!< 右レールへのデバイスの接続状態を取得する
    virtual bool IsRightNwcpAttached() NN_NOEXCEPT
    {
        // TODO: ILinkMonitorImple の外の機能を出す
        // レールの存在していないプラットフォームは常時 False
        return false;
    };

    //!< 左レールへのデバイスの接続状態を取得する
    virtual bool IsLeftNwcpAttached() NN_NOEXCEPT
    {
        // TODO: ILinkMonitorImple の外の機能を出す
        // レールの存在していないプラットフォームは常時 False
        return false;
    };

    //!< Nwcp プロトコルを有効にする
    virtual void SetNwcpEnabled(bool enabled) NN_NOEXCEPT
    {
        NN_UNUSED(enabled);
        // 何もしない
    };

    //!< 本体の電源供給状態を取得する
    virtual bool IsConsolePowered() NN_NOEXCEPT
    {
        // TODO: ILinkMonitorImple の外の機能を出す
        // デフォルトのプラットフォームは本体の電源状態という概念が存在しないため false
        return false;
    };

    //!< ジョイコンのレール部の接続状態に変化があった場合に通知するイベントを登録します
    virtual void SetRailUpdateEvent(nn::os::SystemEventType* pEvent) NN_NOEXCEPT
    {
        NN_UNUSED(pEvent);
        // 何もしない
    };

    //!< ジョイコンのレール部の接続状態に変化について、変化の内容を取得します
    virtual void GetRailUpdateEventType(RailUpdateEventType* pOutEventType, ::nn::bluetooth::Address* pOutAddress) NN_NOEXCEPT
    {
        NN_UNUSED(pOutEventType);
        NN_UNUSED(pOutAddress);
    };

    //!< 左レール部のスリープ復帰要因を取得します
    virtual AwakeTriggerReason GetAwakeTriggerReasonForLeftRail() NN_NOEXCEPT
    {
        // デフォルトのプラットフォームではレールのスリープ復帰要因のイベントは取得できない
        return AwakeTriggerReason_None;
    };

    //!< 右レール部のスリープ復帰要因を取得します
    virtual AwakeTriggerReason GetAwakeTriggerReasonForRightRail() NN_NOEXCEPT
    {
        // デフォルトのプラットフォームではレールのスリープ復帰要因のイベントは取得できない
        return AwakeTriggerReason_None;
    };

    //!< 指定された Usb デバイスをサポートしているか評価します
    virtual bool IsUsbHidSupported(UsbHidDeviceInfo deviceInfo) NN_NOEXCEPT
    {
        NN_UNUSED(deviceInfo);
        // デフォルトのプラットフォームでは USB 接続を非サポート
        return false;
    };

    //!< USB デバイスを追加します
    virtual Result AddUsbHidDevice(int index, UsbHidDeviceInfo deviceInfo) NN_NOEXCEPT
    {
        NN_UNUSED(index);
        NN_UNUSED(deviceInfo);
        // 何もしない
        NN_RESULT_SUCCESS;
    };

    //!< USB デバイスを削除します
    virtual Result RemoveUsbHidDevice(int index) NN_NOEXCEPT
    {
        NN_UNUSED(index);
        // 何もしない
        NN_RESULT_SUCCESS;
    };

    //!< Input Report を処理します
    virtual Result SetUsbHidInputReport(int index, uint8_t *pBuffer, size_t length) NN_NOEXCEPT
    {
        NN_UNUSED(index);
        NN_UNUSED(pBuffer);
        NN_UNUSED(length);
        // 何もしない
        NN_RESULT_SUCCESS;
    };

    //!< Input Report を取得します
    virtual size_t GetUsbHidOutputReport(int index, uint8_t *pOutBuffer, size_t length) NN_NOEXCEPT
    {
        NN_UNUSED(index);
        NN_UNUSED(pOutBuffer);
        NN_UNUSED(length);
        // デフォルトのプラットフォームでは InputReport は取得できない
        return 0;
    };

    //!< 切断時再起動を有効にする
    virtual void SetRebootEnabled(bool enabled) NN_NOEXCEPT
    {
        NN_UNUSED(enabled);
        // 何もしない
    };

    //!< USB FullKeyController を有効にする
    virtual void SetFullKeyUsbEnabled(bool enabled) NN_NOEXCEPT
    {
        NN_UNUSED(enabled);
        // 何もしない
    };

    //!< デバイスが USB 接続されているか評価します
    virtual bool IsUsbConnected(DeviceHandle handle) NN_NOEXCEPT
    {
        NN_UNUSED(handle);
        // 何もしない
        return false;
    };

    //!< USB デバイスの接続状態を更新します
    virtual void UpdateUsbHidDeviceLists() NN_NOEXCEPT
    {
        // 何もしない
    };

    //!< Hid で待機状態のデバイスを再認識する
    virtual void ReInitializePendingUsbDevices() NN_NOEXCEPT
    {
        // 何もしない
    };

    //!< Bluetooth の接続可能最大数を取得します
    virtual int GetMaxBluetoothLinks() NN_NOEXCEPT
    {
        return 0;
    };

    //!< ファームウェア更新中のデバイスを指定します
    virtual Result SetFirmwareUpdatingDevice(DeviceHandle handle) NN_NOEXCEPT
    {
        NN_UNUSED(handle);
        // 何もしない
        NN_RESULT_SUCCESS;
    };

    //!< ファームウェア更新中のデバイス指定を解除します
    virtual void UnsetFirmwareUpdatingDevice() NN_NOEXCEPT
    {
        // 何もしない
    };

    //!< Ble を無効化します
    virtual void DisableBle() NN_NOEXCEPT
    {
        // 何もしない
    }

    //!< 全ての Palma のスキャンの有効無効を切り替える
    virtual void SetIsPalmaAllConnectable(bool connectable) NN_NOEXCEPT
    {
        NN_UNUSED(connectable);
    }

    //!< 登録済みの Palma のスキャンの有効無効を切り替える
    virtual void SetIsPalmaPairedConnectable(bool connectable) NN_NOEXCEPT
    {
        NN_UNUSED(connectable);
    }

    //!< 指定されたデバイスとペアリングする
    virtual Result PairPalma(const ::nn::bluetooth::Address& address) NN_NOEXCEPT
    {
        NN_UNUSED(address);
        NN_RESULT_THROW(ResultNotConnected());
    }

    //!< Fifty が接続されているかどうかを取得します
    virtual bool IsFiftyConnected() NN_NOEXCEPT
    {
        return false;
    };

    //!< Fifty の入力状態を取得します
    virtual PadState GetFiftyPadState() NN_NOEXCEPT
    {
        PadState empty = PadState();
        empty.buttons.Reset();
        return empty;
    };

    //!< Kuina の Fw バージョンを取得します
    virtual Result GetKuinaVersion(KuinaVersionData* pOutMcuVersionData, int index) NN_NOEXCEPT
    {
        NN_UNUSED(pOutMcuVersionData);
        NN_UNUSED(index);
        NN_RESULT_SUCCESS;
    }

    //!< Kuina の Fw バージョンを要求します
    virtual Result RequestKuinaVersion(int index) NN_NOEXCEPT
    {
        NN_UNUSED(index);
        NN_RESULT_SUCCESS;
    }

    //!< Kuina の Fw 更新モードに遷移させます
    virtual Result SetKuinaToFirmwareUpdateMode(int index) NN_NOEXCEPT
    {
        NN_UNUSED(index);
        NN_RESULT_SUCCESS;
    }

};


}}} // namespace nn::xcd::detail
