﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstring>
#include <algorithm>

#include <nn/nn_Common.h>
#include <nn/nn_Result.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/util/util_ScopeExit.h>
#include <nn/fs.h>
#include <nn/xcd/detail/xcd_Log.h>

#include "xcd_FirmwareUpdaterStream.h"

namespace nn { namespace xcd { namespace detail {

FirmwareUpdaterStream::FirmwareUpdaterStream() NN_NOEXCEPT
    : m_Image()
    , m_ReadBytes(0)
{
}

nn::Result FirmwareUpdaterStream::Setup(const FirmwareImage& image) NN_NOEXCEPT
{
    m_Image = image;
    if (m_Image.imageType == FirmwareImageType_File)
    {
        int64_t size;
        NN_RESULT_DO(nn::fs::GetFileSize(&size, m_Image.fileHandle));
        m_Image.imageSize = static_cast<size_t>(size);
    }

    Rewind();

    NN_RESULT_SUCCESS;
}

nn::Result FirmwareUpdaterStream::ReadImpl(uint8_t* pOutBuffer, size_t* pOutReadLength, size_t readLength, size_t offset) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutBuffer);
    NN_SDK_REQUIRES_NOT_NULL(pOutReadLength);

    size_t readableLength = std::min(GetRemainBytes(), readLength);

    switch (m_Image.imageType)
    {
    case FirmwareImageType_File:
        NN_RESULT_DO(
            nn::fs::ReadFile(m_Image.fileHandle, static_cast<int64_t>(offset), pOutBuffer, readableLength)
        );
        break;

    case FirmwareImageType_Memory:
        std::memcpy(pOutBuffer, m_Image.pFirmwareImage + offset, readableLength);
        break;

    default:
        NN_UNEXPECTED_DEFAULT;
    }

    m_ReadBytes += readableLength;
    *pOutReadLength = readableLength;

    NN_RESULT_SUCCESS;
}

}}}  // nn::xcd::detail
