﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Macro.h>
#include <nn/nn_Result.h>
#include <nn/nn_SdkAssert.h>
#include <nn/nn_SdkLog.h>
#include <nn/bluetooth/bluetooth_Api.h>
#include <nn/os/os_Event.h>
#include <nn/os/os_Thread.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/xcd/xcd_ResultForPrivate.h>

#include "xcd_BluetoothHidAccessor-hardware.nx.h"

namespace nn { namespace xcd { namespace detail {

namespace {

nn::bluetooth::BluetoothHhReportType ConvertReportType(ReportType type) NN_NOEXCEPT
{
    switch (type)
    {
    case ReportType_Input:
        return nn::bluetooth::BluetoothHhReportType::BTHH_INPUT_REPORT;
    case ReportType_Output:
        return nn::bluetooth::BluetoothHhReportType::BTHH_OUTPUT_REPORT;
    case ReportType_Feature:
        return nn::bluetooth::BluetoothHhReportType::BTHH_FEATURE_REPORT;
    default:
        NN_UNEXPECTED_DEFAULT;
    }
}

}

BluetoothHidAccessor::BluetoothHidAccessor() NN_NOEXCEPT
    : m_pInputReportParserFunc(nullptr)
    , m_pInputReportParserArg(nullptr)
    , m_IsInSniff(false)
    , m_Activated(false)
{
    // 何もしない
}

BluetoothHidAccessor::~BluetoothHidAccessor() NN_NOEXCEPT
{
    // 何もしない
}

void BluetoothHidAccessor::Activate(DeviceHandle deviceHandle, nn::bluetooth::BluetoothAddress address) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_EQUAL(m_Activated, false);

    m_DeviceHandle = deviceHandle;
    m_Address = address;
    m_IsInSniff = false;
    m_Activated = true;
}

void BluetoothHidAccessor::Deactivate() NN_NOEXCEPT
{
    // サンプリングを中止
    StopSampling();

    m_Activated = false;
}

void BluetoothHidAccessor::StartSampling(InputReportParserFunc func, void* pArg) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(func);
    NN_SDK_REQUIRES_NOT_NULL(pArg);
    NN_SDK_REQUIRES_EQUAL(m_Activated, true);

    m_pInputReportParserFunc = func;
    m_pInputReportParserArg  = pArg;
}

void BluetoothHidAccessor::StopSampling() NN_NOEXCEPT
{
    NN_SDK_REQUIRES_EQUAL(m_Activated, true);

    m_pInputReportParserFunc = nullptr;
    m_pInputReportParserArg  = nullptr;
}

void BluetoothHidAccessor::ParseInputReport(const uint8_t* pBuffer, size_t size) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_LESS_EQUAL(size, static_cast<size_t>(HidAccessorBufferSize));

    if (m_pInputReportParserFunc != nullptr)
    {
        m_pInputReportParserFunc(m_pInputReportParserArg, pBuffer, size);
    }
}

void BluetoothHidAccessor::SetGetReportData(uint8_t* pBuffer, size_t size) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pBuffer);
    NN_UNUSED(size);

    auto pListener = m_pGetReportListener;
    m_pGetReportListener = nullptr;
    pListener->GetReportComplete(pBuffer, size);
}

void BluetoothHidAccessor::NotifyGetReportFailed(nn::bluetooth::BluetoothHhStatus status) NN_NOEXCEPT
{
    auto pListener = m_pGetReportListener;
    m_pGetReportListener = nullptr;
    pListener->GetReportFailed(static_cast<int>(status));
}

void BluetoothHidAccessor::SetSetReportResult(Result result) NN_NOEXCEPT
{
    auto pListener = m_pSetReportListener;
    m_pSetReportListener = nullptr;
    pListener->SetReportComplete(result);
}

//!< コントローラに送るOutputReportをセットする
void BluetoothHidAccessor::SetOutputReport(const uint8_t* pBuffer, size_t size) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pBuffer);
    NN_SDK_REQUIRES_EQUAL(m_Activated, true);

    if (size > 0)
    {
        nn::bluetooth::HidData hidData;
        hidData.length = size;
        std::memcpy(hidData.data, pBuffer, size);

        nn::bluetooth::HidSendData(&m_Address, &hidData);
    }
}

void BluetoothHidAccessor::SetReport(ReportType reportType, uint8_t* pBuffer, size_t size, IHidListener* pListener) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pBuffer);
    NN_SDK_REQUIRES_NOT_NULL(pListener);
    NN_SDK_REQUIRES_EQUAL(m_Activated, true);
    NN_SDK_REQUIRES(size > 0);

    nn::bluetooth::HidData hidData;
    hidData.length = size;
    std::memcpy(hidData.data, pBuffer, size);

    m_pSetReportListener = pListener;

    auto hhReportType = ConvertReportType(reportType);

    if (nn::bluetooth::HidSetReport(&m_Address, hhReportType, &hidData).IsFailure())
    {
        NN_SDK_LOG("  -> Failed sending Set report\n");
    }
}

void BluetoothHidAccessor::GetReport(ReportType reportType, uint8_t reportId, IHidListener* pListener) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pListener);

    auto hhReportType = ConvertReportType(reportType);

    m_pGetReportListener = pListener;

    if (nn::bluetooth::HidGetReport(&m_Address, hhReportType, reportId).IsFailure())
    {
        NN_SDK_LOG("  -> Failed sending Get report\n");
    }
}

size_t BluetoothHidAccessor::GetInputReport(uint8_t* pOutValue, size_t size) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutValue);
    NN_SDK_REQUIRES_EQUAL(m_Activated, true);
    NN_SDK_REQUIRES(size > 0);

    NN_ABORT("Unused");

    return 0;
}

Result BluetoothHidAccessor::GetInterval(::nn::TimeSpan* pOutInterval) NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(m_IsInSniff, ResultHidNotPeriodical());

    *pOutInterval = m_Interval;
    NN_RESULT_SUCCESS;
}

void BluetoothHidAccessor::SetInterval(bool isPeriodical, ::nn::TimeSpan interval) NN_NOEXCEPT
{
    m_IsInSniff = isPeriodical;
    if (isPeriodical == true)
    {
        m_Interval = interval;
    }
}

void BluetoothHidAccessor::DetachDevice() NN_NOEXCEPT
{
    nn::bluetooth::HidDisconnect(&m_Address);
}

}}} // namespace nn::xcd::detail
