﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Macro.h>
#include <nn/nn_Result.h>
#include <nn/nn_SdkAssert.h>
#include <nn/nn_SdkLog.h>
#include <nn/bluetooth/bluetooth_Result.h>
#include <nn/btm/btm_Api.h>
#include <nn/bluetooth/bluetooth_ResultPrivate.h>
#include <nn/xcd/xcd_ResultForPrivate.h>
#include <nn/xcd/detail/xcd_Log.h>
#include <nn/result/result_HandlingUtility.h>
#include "xcd_BleNhogClient-hardware.nx.h"

namespace nn { namespace xcd { namespace detail {

namespace {
    const uint16_t DisableNotificationRawData = 0x0000;  //!< Notification と Indication を無効にする
    const uint16_t EnableNotificationRawData = 0x0001;
    const uint16_t EnableIndicationRawData = 0x0002;
}

BleNhogClient::BleNhogClient() NN_NOEXCEPT
    : m_Activated(false)
    , m_ConnectionHandle(nn::bluetooth::BleInvalidConnectionHandle)
    , m_ServiceHandle(0)
    , m_OperationCount(0)
    , m_OperationQueueHead(0)
    , m_OperationQueueTail(0)
    , m_InputReportFormatId(0)
    , m_OutputCommandFormatId(0)
    , m_CommandResponseFormatId(0)
{
    for (auto& param : m_CharacteristicParameterList)
    {
        param.isValid = false;
    }
    for (auto& param : m_DescriptorParameterList)
    {
        param.isValid = false;
    }
}

BleNhogClient::~BleNhogClient() NN_NOEXCEPT
{
    // 何もしない
}

bool BleNhogClient::GattOperationCompletedHandler(nn::bluetooth::InfoFromLeGattOperationCallback* pGattResult) NN_NOEXCEPT
{
    if (pGattResult->serviceUuid != NhogService.Uuid)
    {
        return false;
    }

    switch (pGattResult->operation)
    {
        case nn::bluetooth::GattOperationType_ReadCharacteristic:
        case nn::bluetooth::GattOperationType_ReadDescriptor:
        {
            return HandleReadCompletedEvent(pGattResult);
        }
        case nn::bluetooth::GattOperationType_WriteCharacteristic:
        case nn::bluetooth::GattOperationType_WriteDescriptor:
        {
            return HandleWriteCompletedEvent(pGattResult);
        }
        case nn::bluetooth::GattOperationType_Notify:
        case nn::bluetooth::GattOperationType_Indicate:
        {
            return HandleNotifyCompletedEvent(pGattResult);
        }
        // ハンドリングしない
        case nn::bluetooth::GattOperationType_Unknown:
            return false;
        default:
            NN_UNEXPECTED_DEFAULT;
    }

    return false;
}

void BleNhogClient::Activate(uint32_t connectionHandle, const nn::btm::user::GattService& service) NN_NOEXCEPT
{
    m_ConnectionHandle = connectionHandle;
    m_ServiceHandle = service.handle;

    nn::btm::user::GattCharacteristic characteristics[MaxAttributeListSize];
    uint8_t num = nn::btm::GetGattCharacteristics(characteristics, MaxAttributeListSize, connectionHandle, service.handle);

    for (int i = 0; i < num; ++i)
    {
        auto& characteristic = characteristics[i];
        // 対象の Characteristic があれば、アクセス用の情報をセットする
        if (characteristic.uuid == InputReportCharacteristic.Uuid)
        {
            SetGattClientAttributeParameterValue(true, CharacteristicIndex_InputReport, service, characteristic);
            // Descriptor のチェック
            nn::btm::user::GattDescriptor descriptors[MaxAttributeListSize];
            uint8_t descNum = nn::btm::GetGattDescriptors(descriptors, MaxAttributeListSize, connectionHandle, characteristic.handle);
            for (int j = 0; j < descNum; ++j)
            {
                auto& descriptor = descriptors[j];
                if (descriptor.uuid == ClientConfigDescriptor.Uuid)
                {
                    SetGattClientAttributeParameterValue(true, DescriptorIndex_InputReport_ClientConfig, service, characteristic, descriptor);
                }
                else if (descriptor.uuid == ReportReferenceDescriptor.Uuid)
                {
                    SetGattClientAttributeParameterValue(true, DescriptorIndex_InputReport_ReportReference, service, characteristic, descriptor);
                }
            }
        }
        else if (characteristic.uuid == OutputCommandCharacteristic.Uuid)
        {
            SetGattClientAttributeParameterValue(true, CharacteristicIndex_OutputCommand, service, characteristic);
            // Descriptor のチェック
            nn::btm::user::GattDescriptor descriptors[MaxAttributeListSize];
            uint8_t descNum = nn::btm::GetGattDescriptors(descriptors, MaxAttributeListSize, connectionHandle, characteristic.handle);
            for (int j = 0; j < descNum; ++j)
            {
                auto& descriptor = descriptors[j];
                if (descriptor.uuid == ReportReferenceDescriptor.Uuid)
                {
                    SetGattClientAttributeParameterValue(true, DescriptorIndex_OutputCommand_ReportReference, service, characteristic, descriptor);
                }
            }
        }
        else if (characteristic.uuid == CommandResponseCharacteristic.Uuid)
        {
            SetGattClientAttributeParameterValue(true, CharacteristicIndex_CommandResponse, service, characteristic);
            // Descriptor のチェック
            nn::btm::user::GattDescriptor descriptors[MaxAttributeListSize];
            uint8_t descNum = nn::btm::GetGattDescriptors(descriptors, MaxAttributeListSize, connectionHandle, characteristic.handle);
            for (int j = 0; j < descNum; ++j)
            {
                auto& descriptor = descriptors[j];
                if (descriptor.uuid == ClientConfigDescriptor.Uuid)
                {
                    SetGattClientAttributeParameterValue(true, DescriptorIndex_CommandResponse_ClientConfig, service, characteristic, descriptor);
                }
                else if (descriptor.uuid == ReportReferenceDescriptor.Uuid)
                {
                    SetGattClientAttributeParameterValue(true, DescriptorIndex_CommandResponse_ReportReference, service, characteristic, descriptor);
                }
            }
        }
    }

    m_Activated = true;
}

void BleNhogClient::Deactivate() NN_NOEXCEPT
{
    for (auto& param : m_CharacteristicParameterList)
    {
        param.isValid = false;
    }
    for (auto& param : m_DescriptorParameterList)
    {
        param.isValid = false;
    }

    m_OperationCount = 0;
    m_OperationQueueHead = 0;
    m_OperationQueueTail = 0;
    m_InputReportFormatId = 0;
    m_OutputCommandFormatId = 0;
    m_CommandResponseFormatId = 0;
    m_Activated = false;
}

uint16_t BleNhogClient::GetServiceHandle() NN_NOEXCEPT
{
    return m_ServiceHandle;
}

Result BleNhogClient::Proceed() NN_NOEXCEPT
{
    NN_SDK_REQUIRES_EQUAL(m_Activated, true);

    // キューに1つもない
    if (m_OperationCount == 0)
    {
        NN_RESULT_SUCCESS;
    }

    auto& command = m_OperationQueue[m_OperationQueueHead];

    // 未送信状態でない
    if (command.opStatus != GattOperationStatus_Queued)
    {
        NN_RESULT_SUCCESS;
    }

    switch (command.opType)
    {
        case GattOperationType_ReadCharacteristic:
        {
            NN_RESULT_TRY(ReadCharacteristic(m_ConnectionHandle, command.parameter, command.authType))
                NN_RESULT_CATCH(nn::bluetooth::ResultBleBusyError)

                {
                    // Busy の場合は次の Proceed でリトライする
                    NN_RESULT_RETHROW;
                }
                NN_RESULT_CATCH_ALL
                {
                    // そのほかのエラーはタスク終了とし、キューから取り除く
                    NN_DETAIL_XCD_WARN("Catch bluetooth Error\n");
                    RemoveHead();
                    NN_RESULT_RETHROW;
                }
            NN_RESULT_END_TRY
            break;
        }
        case GattOperationType_WriteCharacteristic:
        {
            // Nintendo Hid over Gatt はすべて Without Response とする
            NN_RESULT_TRY(WriteCharacteristic(m_ConnectionHandle, command.parameter, command.authType, false, command.payload))
                NN_RESULT_CATCH(nn::bluetooth::ResultBleBusyError)
                {
                    NN_RESULT_RETHROW;
                }
                NN_RESULT_CATCH_ALL
                {
                    NN_DETAIL_XCD_WARN("Catch bluetooth Error\n");
                    RemoveHead();
                    NN_RESULT_RETHROW;
                }
            NN_RESULT_END_TRY
            break;
        }
        case GattOperationType_ReadDescriptor:
        {
            NN_RESULT_TRY(ReadDescriptor(m_ConnectionHandle, command.parameter, command.authType))
                NN_RESULT_CATCH(nn::bluetooth::ResultBleBusyError)
                {
                    NN_RESULT_RETHROW;
                }
                NN_RESULT_CATCH_ALL
                {
                    NN_DETAIL_XCD_WARN("Catch bluetooth Error\n");
                    RemoveHead();
                    NN_RESULT_RETHROW;
                }
            NN_RESULT_END_TRY
            break;
        }
        case GattOperationType_WriteDescriptor:
        {
            NN_SDK_REQUIRES_NOT_NULL(command.payload.dataptr);

            NN_RESULT_TRY(WriteDescriptor(m_ConnectionHandle, command.parameter, command.authType, command.payload))
                NN_RESULT_CATCH(nn::bluetooth::ResultBleBusyError)
                {
                    NN_RESULT_RETHROW;
                }
                NN_RESULT_CATCH_ALL
                {
                    NN_DETAIL_XCD_WARN("Catch bluetooth Error\n");
                    RemoveHead();
                    NN_RESULT_RETHROW;
                }
            NN_RESULT_END_TRY
            break;
        }
        default:
            NN_UNEXPECTED_DEFAULT;
    }

    command.opStatus = GattOperationStatus_Sent;
    NN_RESULT_SUCCESS;
}

void BleNhogClient::WriteOutputCommandCharacteristicAsync(const GattOperationPayload& payload, IBleCommandListener* pListener) NN_NOEXCEPT
{
    if (m_CharacteristicParameterList[CharacteristicIndex_OutputCommand].isValid == true)
    {
        Enqueue(GattOperationType_WriteCharacteristic, 0, m_CharacteristicParameterList[CharacteristicIndex_OutputCommand], payload, pListener);
    }
}

void BleNhogClient::ReadInputReportReportReferenceAsync(IBleCommandListener* pListener) NN_NOEXCEPT
{
    if (m_DescriptorParameterList[DescriptorIndex_InputReport_ReportReference].isValid == true)
    {
        Enqueue(GattOperationType_ReadDescriptor, 0, m_DescriptorParameterList[DescriptorIndex_InputReport_ReportReference], pListener);
    }
}

void BleNhogClient::ReadOutputCommandReportReferenceAsync(IBleCommandListener* pListener) NN_NOEXCEPT
{
    if (m_DescriptorParameterList[DescriptorIndex_OutputCommand_ReportReference].isValid == true)
    {
        Enqueue(GattOperationType_ReadDescriptor, 0, m_DescriptorParameterList[DescriptorIndex_OutputCommand_ReportReference], pListener);
    }
}

void BleNhogClient::ReadCommandResponseReportReferenceAsync(IBleCommandListener* pListener) NN_NOEXCEPT
{
    if (m_DescriptorParameterList[DescriptorIndex_CommandResponse_ReportReference].isValid == true)
    {
        Enqueue(GattOperationType_ReadDescriptor, 0, m_DescriptorParameterList[DescriptorIndex_CommandResponse_ReportReference], pListener);
    }
}

void BleNhogClient::EnableInputReportNotification(IBleCommandListener* pListener, bool isEnabled) NN_NOEXCEPT
{
    if (m_DescriptorParameterList[DescriptorIndex_InputReport_ClientConfig].isValid == true)
    {
        if (isEnabled)
        {
            SetNotification(m_ConnectionHandle, m_DescriptorParameterList[DescriptorIndex_InputReport_ClientConfig], true);
            GattOperationPayload payload = { sizeof(uint16_t), &EnableNotificationRawData };
            Enqueue(GattOperationType_WriteDescriptor, 0, m_DescriptorParameterList[DescriptorIndex_InputReport_ClientConfig], payload, pListener);
        }
        else
        {
            SetNotification(m_ConnectionHandle, m_DescriptorParameterList[DescriptorIndex_InputReport_ClientConfig], false);
            GattOperationPayload payload = { sizeof(uint16_t), &DisableNotificationRawData };
            Enqueue(GattOperationType_WriteDescriptor, 0, m_DescriptorParameterList[DescriptorIndex_InputReport_ClientConfig], payload, pListener);
        }
    }
}

void BleNhogClient::EnableCommandResponseNotification(IBleCommandListener* pListener, bool isEnabled) NN_NOEXCEPT
{
    if (m_DescriptorParameterList[DescriptorIndex_CommandResponse_ClientConfig].isValid == true)
    {
        if (isEnabled)
        {
            SetNotification(m_ConnectionHandle, m_DescriptorParameterList[DescriptorIndex_CommandResponse_ClientConfig], true);
            GattOperationPayload payload = { sizeof(uint16_t), &EnableNotificationRawData };
            Enqueue(GattOperationType_WriteDescriptor, 0, m_DescriptorParameterList[DescriptorIndex_CommandResponse_ClientConfig], payload, pListener);
        }
        else
        {
            SetNotification(m_ConnectionHandle, m_DescriptorParameterList[DescriptorIndex_CommandResponse_ClientConfig], false);
            GattOperationPayload payload = { sizeof(uint16_t), &DisableNotificationRawData };
            Enqueue(GattOperationType_WriteDescriptor, 0, m_DescriptorParameterList[DescriptorIndex_CommandResponse_ClientConfig], payload, pListener);
        }
    }
}

void BleNhogClient::EnableCommandResponseIndication(IBleCommandListener* pListener, bool isEnabled) NN_NOEXCEPT
{
    if (m_DescriptorParameterList[DescriptorIndex_CommandResponse_ClientConfig].isValid == true)
    {
        if (isEnabled)
        {
            NN_ABORT_UNLESS_RESULT_SUCCESS(SetNotification(m_ConnectionHandle, m_DescriptorParameterList[DescriptorIndex_CommandResponse_ClientConfig], true));
            GattOperationPayload payload = { sizeof(uint16_t), &EnableIndicationRawData };
            Enqueue(GattOperationType_WriteDescriptor, 0, m_DescriptorParameterList[DescriptorIndex_CommandResponse_ClientConfig], payload, pListener);
        }
        else
        {
            SetNotification(m_ConnectionHandle, m_DescriptorParameterList[DescriptorIndex_CommandResponse_ClientConfig], false);
            GattOperationPayload payload = { sizeof(uint16_t), &DisableNotificationRawData };
            Enqueue(GattOperationType_WriteDescriptor, 0, m_DescriptorParameterList[DescriptorIndex_CommandResponse_ClientConfig], payload, pListener);
        }
    }
}

BleInputReportFormatVersion BleNhogClient::GetInputReportFormatVersion() NN_NOEXCEPT
{
    switch (m_InputReportFormatId)
    {
        case 0:
            return BleInputReportFormatVersion_Unknown;
        case 1:
            return BleInputReportFormatVersion_JoyRight;
        case 2:
            return BleInputReportFormatVersion_JoyLeft;
        case 3:
            return BleInputReportFormatVersion_Palma;
        default: // Unknown にしておく
            return BleInputReportFormatVersion_Unknown;
    }
}

BleOutputCommandFormatVersion BleNhogClient::GetOutputCommandFormatVersion() NN_NOEXCEPT
{
    switch (m_OutputCommandFormatId)
    {
        case 0:
            return BleOutputCommandFormatVersion_Unknown;
        case 1:
            return BleOutputCommandFormatVersion_Palma;
        default: // Unknown にしておく
            return BleOutputCommandFormatVersion_Unknown;
    }
}

BleCommandResponseFormatVersion BleNhogClient::GetCommandResponseFormatVersion() NN_NOEXCEPT
{
    switch (m_CommandResponseFormatId)
    {
        case 0:
            return BleCommandResponseFormatVersion_Unknown;
        case 1:
            return BleCommandResponseFormatVersion_Palma;
        default: // Unknown にしておく
            return BleCommandResponseFormatVersion_Unknown;
    }
}

void BleNhogClient::SetGattClientAttributeParameterValue(bool isValid, CharacteristicIndex index, const nn::btm::user::GattService& service, const nn::btm::user::GattCharacteristic& characteristic) NN_NOEXCEPT
{
    auto& param = m_CharacteristicParameterList[index];
    // isValid の値が変わらなければ何もしない
    if (param.isValid == isValid)
    {
        return;
    }

    param.isValid = isValid;
    param.type = characteristic.type;
    param.index = index;
    param.service.instanceId = service.instanceId;
    param.service.uuid = service.uuid;
    param.isPrimary = service.isPrimaryService;
    param.characteristic.instanceId = characteristic.instanceId;
    param.characteristic.uuid = characteristic.uuid;
}

void BleNhogClient::SetGattClientAttributeParameterValue(bool isValid, DescriptorIndex index, const nn::btm::user::GattService& service, const nn::btm::user::GattCharacteristic& characteristic, const nn::btm::user::GattDescriptor& descriptor) NN_NOEXCEPT
{
    auto& param = m_DescriptorParameterList[index];
    // isValid の値が変わらなければ何もしない
    if (param.isValid == isValid)
    {
        return;
    }

    param.isValid = isValid;
    param.type = descriptor.type;
    param.index = index;
    param.service.instanceId = service.instanceId;
    param.service.uuid = service.uuid;
    param.isPrimary = service.isPrimaryService;
    param.characteristic.instanceId = characteristic.instanceId;
    param.characteristic.uuid = characteristic.uuid;
    param.descriptor.instanceId = 0;
    param.descriptor.uuid = descriptor.uuid;
}

Result BleNhogClient::Enqueue(GattOperationType opType, uint8_t authType, const GattClientAttributeParameter& param, IBleCommandListener* pListener) NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(m_OperationCount < MaxOperationCount, ResultCommandQueueFull());

    auto& command = m_OperationQueue[m_OperationQueueTail];

    command.opStatus = GattOperationStatus_Queued;
    command.opType = opType;
    command.authType = authType;
    command.parameter = param;
    command.pListener = pListener;

    m_OperationQueueTail = (m_OperationQueueTail + 1) % MaxOperationCount;
    m_OperationCount++;

    NN_RESULT_SUCCESS;
}

Result BleNhogClient::Enqueue(GattOperationType opType, uint8_t authType, const GattClientAttributeParameter& param, const GattOperationPayload& payload, IBleCommandListener* pListener) NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(m_OperationCount < MaxOperationCount, ResultCommandQueueFull());

    auto& command = m_OperationQueue[m_OperationQueueTail];

    command.opStatus = GattOperationStatus_Queued;
    command.opType = opType;
    command.authType = authType;
    command.parameter = param;
    command.payload = payload;
    command.pListener = pListener;

    m_OperationQueueTail = (m_OperationQueueTail + 1) % MaxOperationCount;
    m_OperationCount++;

    NN_RESULT_SUCCESS;
}
void BleNhogClient::RemoveHead() NN_NOEXCEPT
{
    NN_SDK_REQUIRES_GREATER(m_OperationCount , 0);

    auto& command = m_OperationQueue[m_OperationQueueHead];

    command.opStatus = GattOperationStatus_Empty;

    m_OperationQueueHead = (m_OperationQueueHead + 1) % MaxOperationCount;
    m_OperationCount--;
}

bool BleNhogClient::HandleReadCompletedEvent(nn::bluetooth::InfoFromLeGattOperationCallback* pGattResult) NN_NOEXCEPT
{
    // キューに1つもない
    if (m_OperationCount == 0)
    {
        return false;
    }

    auto& command = m_OperationQueue[m_OperationQueueHead];

    // Callback を待っていない
    if (command.opStatus != GattOperationStatus_Sent)
    {
        return false;
    }

    // コールバック待ちの GATT 操作かどうか
    if (command.opType == GattOperationType_ReadCharacteristic)
    {
        if (pGattResult->charcteristicUuid != command.parameter.characteristic.uuid)
        {
            return false;
        }
    }
    else if (command.opType == GattOperationType_ReadDescriptor)
    {
        if (pGattResult->descriptorUuid != command.parameter.descriptor.uuid)
        {
            return false;
        }
    }

    bool returnValue = true;
    // コールバックの status がエラー
    if (pGattResult->status != nn::bluetooth::BT_OK)
    {
        NN_DETAIL_XCD_ERROR("Nintendo Hid over Gatt: GattOperation error char=%s status=%d operation=0x%02X\n",
            GetName(pGattResult->charcteristicUuid), pGattResult->status, pGattResult->operation);
        returnValue = false;
    }
    else
    {
        if (pGattResult->charcteristicUuid == InputReportCharacteristic.Uuid)
        {
            if (pGattResult->descriptorUuid == ReportReferenceDescriptor.Uuid)
            {
                m_InputReportFormatId = pGattResult->value[0];
            }
            else if (pGattResult->descriptorUuid == ClientConfigDescriptor.Uuid)
            {
                NN_DETAIL_XCD_INFO("InputReport ClientConfig=%04x\n", *reinterpret_cast<uint16_t*>(pGattResult->value));
            }
            returnValue = true;
        }
        else if (pGattResult->charcteristicUuid == OutputCommandCharacteristic.Uuid)
        {
            if (pGattResult->descriptorUuid == ReportReferenceDescriptor.Uuid)
            {
                m_OutputCommandFormatId = pGattResult->value[0];
            }
            returnValue = true;
        }
        else if (pGattResult->charcteristicUuid == CommandResponseCharacteristic.Uuid)
        {
            if (pGattResult->descriptorUuid == ReportReferenceDescriptor.Uuid)
            {
                m_CommandResponseFormatId = pGattResult->value[0];
            }
            else if (pGattResult->descriptorUuid == ClientConfigDescriptor.Uuid)
            {
                NN_DETAIL_XCD_INFO("CommandResponse ClientConfig=%04x\n", *reinterpret_cast<uint16_t*>(pGattResult->value));
            }
            returnValue = true;
        }
    }

    if (command.pListener != nullptr)
    {
        if (returnValue == true)
        {
            if (command.opType == GattOperationType_ReadDescriptor)
            {
                command.pListener->DescriptorReadComplete(pGattResult->charcteristicUuid, pGattResult->descriptorUuid);
            }
            else if (command.opType == GattOperationType_ReadCharacteristic)
            {
                command.pListener->CharacteristicReadComplete(pGattResult->charcteristicUuid);
            }
        }
        else
        {
            command.pListener->NotifyGattOperationError(pGattResult->status);
        }
    }

    RemoveHead();
    return returnValue;
}

bool BleNhogClient::HandleWriteCompletedEvent(nn::bluetooth::InfoFromLeGattOperationCallback* pGattResult) NN_NOEXCEPT
{
    // キューに1つもない
    if (m_OperationCount == 0)
    {
        return false;
    }

    auto& command = m_OperationQueue[m_OperationQueueHead];

    // Callback を待っていない
    if (command.opStatus != GattOperationStatus_Sent)
    {
        return false;
    }

    // コールバック待ちの GATT 操作かどうか
    if (command.opType == GattOperationType_WriteCharacteristic)
    {
        if (pGattResult->charcteristicUuid != command.parameter.characteristic.uuid)
        {
            return false;
        }
    }
    else if (command.opType == GattOperationType_WriteDescriptor)
    {
        if (pGattResult->descriptorUuid != command.parameter.descriptor.uuid)
        {
            return false;
        }
    }

    bool returnValue = true;
    // コールバックの status がエラー
    if (pGattResult->status != nn::bluetooth::BT_OK)
    {
        NN_DETAIL_XCD_ERROR("Nintendo Hid over Gatt: GattOperation error char=%s status=%d operation=0x%02X\n",
            GetName(pGattResult->charcteristicUuid), pGattResult->status, pGattResult->operation);
        returnValue = false;
    }
    else
    {
        returnValue = true;
    }

    if (command.pListener != nullptr)
    {
        if (returnValue == true)
        {
            if (command.opType == GattOperationType_WriteDescriptor)
            {
                command.pListener->DescriptorWriteComplete(pGattResult->charcteristicUuid, pGattResult->descriptorUuid);
            }
            else if (command.opType == GattOperationType_WriteCharacteristic)
            {
                command.pListener->CharacteristicWriteComplete(pGattResult->charcteristicUuid);
            }
        }
        else
        {
            command.pListener->NotifyGattOperationError(pGattResult->status);
        }
    }

    RemoveHead();
    return returnValue;
}

bool BleNhogClient::HandleNotifyCompletedEvent(nn::bluetooth::InfoFromLeGattOperationCallback* pGattResult) NN_NOEXCEPT
{
    bool returnValue = false;
    if (pGattResult->charcteristicUuid == InputReportCharacteristic.Uuid)
    {
        returnValue = true;
    }
    else if (pGattResult->charcteristicUuid == CommandResponseCharacteristic.Uuid)
    {
        returnValue = true;
    }

    return returnValue;
}

const char* BleNhogClient::GetName(const nn::bluetooth::GattAttributeUuid& uuid) NN_NOEXCEPT
{
    if (uuid == InputReportCharacteristic.Uuid)
    {
        return "InputReport";
    }
    else if (uuid == OutputCommandCharacteristic.Uuid)
    {
        return "OutputCommand";
    }
    else if (uuid == CommandResponseCharacteristic.Uuid)
    {
        return "CommandResponse";
    }
    else if (uuid == ClientConfigDescriptor.Uuid)
    {
        return "ClientConfig";
    }
    else if (uuid == ReportReferenceDescriptor.Uuid)
    {
        return "ReportReference";
    }
    else
    {
        return "Unknown";
    }
}

}}} // namespace nn::xcd::detail
