﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Macro.h>
#include <nn/nn_Result.h>
#include <nn/nn_SdkAssert.h>
#include <nn/nn_SdkLog.h>
#include <nn/bluetooth/bluetooth_Result.h>
#include <nn/btm/btm_Api.h>
#include <nn/bluetooth/bluetooth_ResultPrivate.h>
#include <nn/xcd/xcd_ResultForPrivate.h>
#include <nn/xcd/detail/xcd_Log.h>
#include <nn/result/result_HandlingUtility.h>
#include "xcd_BleNbatClient-hardware.nx.h"

namespace nn { namespace xcd { namespace detail {

namespace {
//    const uint16_t DisableNotificationRawData = 0x0000;
    const uint16_t EnableIndicationRawData = 0x0002;
}

BleNbatClient::BleNbatClient() NN_NOEXCEPT
    : m_Activated(false)
    , m_ConnectionHandle(nn::bluetooth::BleInvalidConnectionHandle)
    , m_ServiceHandle(0)
    , m_OperationCount(0)
    , m_OperationQueueHead(0)
    , m_OperationQueueTail(0)
{
    for (auto& param : m_CharacteristicParameterList)
    {
        param.isValid = false;
    }
    for (auto& param : m_DescriptorParameterList)
    {
        param.isValid = false;
    }
}

BleNbatClient::~BleNbatClient() NN_NOEXCEPT
{
    // 何もしない
}

bool BleNbatClient::GattOperationCompletedHandler(nn::bluetooth::InfoFromLeGattOperationCallback* pGattResult) NN_NOEXCEPT
{
    if (pGattResult->serviceUuid != NbatService.Uuid)
    {
        return false;
    }
    switch (pGattResult->operation)
    {
        case nn::bluetooth::GattOperationType_ReadCharacteristic:
        case nn::bluetooth::GattOperationType_ReadDescriptor:
        {
            return HandleReadCompletedEvent(pGattResult);
        }
        case nn::bluetooth::GattOperationType_WriteCharacteristic:
        case nn::bluetooth::GattOperationType_WriteDescriptor:
        {
            return HandleWriteCompletedEvent(pGattResult);
        }
        case nn::bluetooth::GattOperationType_Notify:
        case nn::bluetooth::GattOperationType_Indicate:
        {
            return HandleNotifyCompletedEvent(pGattResult);
        }
        // ハンドリングしない
        case nn::bluetooth::GattOperationType_Unknown:
            return false;
        default:
            NN_UNEXPECTED_DEFAULT;
    }

    return false;
}

void BleNbatClient::Activate(uint32_t connectionHandle, const nn::btm::user::GattService& service) NN_NOEXCEPT
{
    NN_DETAIL_XCD_INFO("Activate Nbat\n");
    m_ConnectionHandle = connectionHandle;
    m_ServiceHandle = service.handle;
    m_Flags.Reset();

    nn::btm::user::GattCharacteristic characteristics[MaxAttributeListSize];
    uint8_t num = nn::btm::GetGattCharacteristics(characteristics, MaxAttributeListSize, connectionHandle, service.handle);
    for (int i = 0; i < num; ++i)
    {
        auto& characteristic = characteristics[i];
        // 対象の Characteristic があれば、アクセス用の情報をセットする
        if (characteristic.uuid == BatteryLevelCharacteristic.Uuid)
        {
            SetGattClientAttributeParameterValue(true, CharacteristicIndex_BatteryLevel, service, characteristic);
            // Descriptor のチェック
            nn::btm::user::GattDescriptor descriptors[MaxAttributeListSize];
            uint8_t descNum = nn::btm::GetGattDescriptors(descriptors, MaxAttributeListSize, connectionHandle, characteristic.handle);
            for (int j = 0; j < descNum; ++j)
            {
                auto& descriptor = descriptors[j];
                if (descriptor.uuid == ClientConfigDescriptor.Uuid)
                {
                    SetGattClientAttributeParameterValue(true, DescriptorIndex_BatteryLevel_ClientConfig, service, characteristic, descriptor);
/* Activate 時に Idncation を勝手に有効にすると Busy に陥るので現状は DeviceHandler から制御
                    this->EnableBatteryLevelIndication();
*/
                }
            }
        }
        else if (characteristic.uuid == BatteryStateCharacteristic.Uuid)
        {
            SetGattClientAttributeParameterValue(true, CharacteristicIndex_BatteryState, service, characteristic);
            // Descriptor のチェック
            nn::btm::user::GattDescriptor descriptors[MaxAttributeListSize];
            uint8_t descNum = nn::btm::GetGattDescriptors(descriptors, MaxAttributeListSize, connectionHandle, characteristic.handle);
            for (int j = 0; j < descNum; ++j)
            {
                auto& descriptor = descriptors[j];
                if (descriptor.uuid == ClientConfigDescriptor.Uuid)
                {
                    SetGattClientAttributeParameterValue(true, DescriptorIndex_BatteryState_ClientConfig, service, characteristic, descriptor);
/* Activate 時に Idncation を勝手に有効にすると Busy に陥るので現状は DeviceHandler から制御
                    this->EnableBatteryStateIndication();
*/
                }
            }
        }
    }

    m_Activated = true;
}

void BleNbatClient::Deactivate() NN_NOEXCEPT
{
    for (auto& param : m_CharacteristicParameterList)
    {
        param.isValid = false;
    }
    for (auto& param : m_DescriptorParameterList)
    {
        param.isValid = false;
    }

    m_OperationCount = 0;
    m_OperationQueueHead = 0;
    m_OperationQueueTail = 0;
    m_Activated = false;
}

uint16_t BleNbatClient::GetServiceHandle() NN_NOEXCEPT
{
    return m_ServiceHandle;
}

void BleNbatClient::UpdateBatteryStatus() NN_NOEXCEPT
{
/*
    this->ReadBatteryLevelAsync();
    this->ReadBatteryStateAsync();
*/
    EnableBatteryLevelIndication();
    EnableBatteryStateIndication();
}

Result BleNbatClient::Proceed() NN_NOEXCEPT
{
    NN_SDK_REQUIRES_EQUAL(m_Activated, true);

    // キューに1つもない
    if (m_OperationCount == 0)
    {
        NN_RESULT_SUCCESS;
    }

    auto& command = m_OperationQueue[m_OperationQueueHead];

    // 未送信状態でない
    if (command.opStatus != GattOperationStatus_Queued)
    {
        NN_RESULT_SUCCESS;
    }

    switch (command.opType)
    {
        case GattOperationType_ReadCharacteristic:
        {
            NN_RESULT_TRY(ReadCharacteristic(m_ConnectionHandle, command.parameter, command.authType))
                NN_RESULT_CATCH(nn::bluetooth::ResultBleBusyError)
                {
                    // Busy の場合は次の Proceed でリトライする
                    NN_RESULT_RETHROW;
                }
                NN_RESULT_CATCH_ALL
                {
                    // そのほかのエラーはタスク終了とし、キューから取り除く
                    NN_DETAIL_XCD_WARN("Catch bluetooth Error\n");
                    RemoveHead();
                    NN_RESULT_RETHROW;
                }
            NN_RESULT_END_TRY
            break;
        }
        case GattOperationType_WriteCharacteristic:
        {
            // Nintendo Battery Service はすべて Without Response とする
            NN_RESULT_TRY(WriteCharacteristic(m_ConnectionHandle, command.parameter, command.authType, false, command.payload))
                NN_RESULT_CATCH(nn::bluetooth::ResultBleBusyError)
                {
                    NN_RESULT_RETHROW;
                }
                NN_RESULT_CATCH_ALL
                {
                    NN_DETAIL_XCD_WARN("Catch bluetooth Error\n");
                    RemoveHead();
                    NN_RESULT_RETHROW;
                }
            NN_RESULT_END_TRY
            break;
        }
        case GattOperationType_ReadDescriptor:
        {
            NN_RESULT_TRY(ReadDescriptor(m_ConnectionHandle, command.parameter, command.authType))
                NN_RESULT_CATCH(nn::bluetooth::ResultBleBusyError)
                {
                    NN_RESULT_RETHROW;
                }
                NN_RESULT_CATCH_ALL
                {
                    NN_DETAIL_XCD_WARN("Catch bluetooth Error\n");
                    RemoveHead();
                    NN_RESULT_RETHROW;
                }
            NN_RESULT_END_TRY
            break;
        }
        case GattOperationType_WriteDescriptor:
        {
            NN_SDK_REQUIRES_NOT_NULL(command.payload.dataptr);

            NN_RESULT_TRY(WriteDescriptor(m_ConnectionHandle, command.parameter, command.authType, command.payload))
                NN_RESULT_CATCH(nn::bluetooth::ResultBleBusyError)
                {
                    NN_RESULT_RETHROW;
                }
                NN_RESULT_CATCH_ALL
                {
                    NN_DETAIL_XCD_WARN("Catch bluetooth Error\n");
                    RemoveHead();
                    NN_RESULT_RETHROW;
                }
            NN_RESULT_END_TRY
            break;
        }
        default:
            NN_UNEXPECTED_DEFAULT;
    }

    command.opStatus = GattOperationStatus_Sent;
    NN_RESULT_SUCCESS;
}


void BleNbatClient::SetGattClientAttributeParameterValue(bool isValid, CharacteristicIndex index, const nn::btm::user::GattService& service, const nn::btm::user::GattCharacteristic& characteristic) NN_NOEXCEPT
{
    auto& param = m_CharacteristicParameterList[index];
    // isValid の値が変わらなければ何もしない
    if (param.isValid == isValid)
    {
        return;
    }

    param.isValid = isValid;
    param.type = characteristic.type;
    param.index = index;
    param.service.instanceId = service.instanceId;
    param.service.uuid = service.uuid;
    param.isPrimary = service.isPrimaryService;
    param.characteristic.instanceId = characteristic.instanceId;
    param.characteristic.uuid = characteristic.uuid;
}

void BleNbatClient::SetGattClientAttributeParameterValue(bool isValid, DescriptorIndex index, const nn::btm::user::GattService& service, const nn::btm::user::GattCharacteristic& characteristic, const nn::btm::user::GattDescriptor& descriptor) NN_NOEXCEPT
{
    auto& param = m_DescriptorParameterList[index];
    // isValid の値が変わらなければ何もしない
    if (param.isValid == isValid)
    {
        return;
    }

    param.isValid = isValid;
    param.type = descriptor.type;
    param.index = index;
    param.service.instanceId = service.instanceId;
    param.service.uuid = service.uuid;
    param.isPrimary = service.isPrimaryService;
    param.characteristic.instanceId = characteristic.instanceId;
    param.characteristic.uuid = characteristic.uuid;
    param.descriptor.instanceId = 0;
    param.descriptor.uuid = descriptor.uuid;
}

Result BleNbatClient::Enqueue(GattOperationType opType, uint8_t authType, const GattClientAttributeParameter& param, IBleCommandListener* pListener) NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(m_OperationCount < MaxOperationCount, ResultCommandQueueFull());

    auto& command = m_OperationQueue[m_OperationQueueTail];

    command.opStatus = GattOperationStatus_Queued;
    command.opType = opType;
    command.authType = authType;
    command.parameter = param;
    command.pListener = pListener;

    m_OperationQueueTail = (m_OperationQueueTail + 1) % MaxOperationCount;
    m_OperationCount++;

    NN_RESULT_SUCCESS;
}

Result BleNbatClient::Enqueue(GattOperationType opType, uint8_t authType, const GattClientAttributeParameter& param, const GattOperationPayload& payload, IBleCommandListener* pListener) NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(m_OperationCount < MaxOperationCount, ResultCommandQueueFull());

    auto& command = m_OperationQueue[m_OperationQueueTail];

    command.opStatus = GattOperationStatus_Queued;
    command.opType = opType;
    command.authType = authType;
    command.parameter = param;
    command.payload = payload;
    command.pListener = pListener;

    m_OperationQueueTail = (m_OperationQueueTail + 1) % MaxOperationCount;
    m_OperationCount++;

    NN_RESULT_SUCCESS;
}
void BleNbatClient::RemoveHead() NN_NOEXCEPT
{
    NN_SDK_REQUIRES_GREATER(m_OperationCount , 0);

    auto& command = m_OperationQueue[m_OperationQueueHead];

    command.opStatus = GattOperationStatus_Empty;

    m_OperationQueueHead = (m_OperationQueueHead + 1) % MaxOperationCount;
    m_OperationCount--;
}

bool BleNbatClient::HandleReadCompletedEvent(nn::bluetooth::InfoFromLeGattOperationCallback* pGattResult) NN_NOEXCEPT
{
    // キューに1つもない
    if (m_OperationCount == 0)
    {
        return false;
    }

    auto& command = m_OperationQueue[m_OperationQueueHead];

    // Callback を待っていない
    if (command.opStatus != GattOperationStatus_Sent)
    {
        return false;
    }

    // コールバック待ちの GATT 操作かどうか
    if (command.opType == GattOperationType_ReadCharacteristic)
    {
        if (pGattResult->charcteristicUuid != command.parameter.characteristic.uuid)
        {
            return false;
        }
    }
    else if (command.opType == GattOperationType_ReadDescriptor)
    {
        if (pGattResult->descriptorUuid != command.parameter.descriptor.uuid)
        {
            return false;
        }
    }

    bool returnValue = true;
    // コールバックの status がエラー
    if (pGattResult->status != nn::bluetooth::BT_OK)
    {
        NN_DETAIL_XCD_ERROR("Nintendo Battery Service: GattOperation error char=%s status=%d operation=0x%02X\n",
            GetName(pGattResult->charcteristicUuid), pGattResult->status, pGattResult->operation);
        returnValue = false;
    }
    else
    {
        if (pGattResult->charcteristicUuid == BatteryLevelCharacteristic.Uuid)
        {
            returnValue = HandleReadBatteryLevel(pGattResult->value, pGattResult->length);
        }
        else if (pGattResult->charcteristicUuid == BatteryStateCharacteristic.Uuid)
        {
            returnValue = HandleReadBatteryState(pGattResult->value, pGattResult->length);
        }
    }

    if (command.pListener != nullptr)
    {
        if (returnValue == true)
        {
            if (command.opType == GattOperationType_ReadDescriptor)
            {
                command.pListener->DescriptorReadComplete(pGattResult->charcteristicUuid, pGattResult->descriptorUuid);
            }
            else if (command.opType == GattOperationType_ReadCharacteristic)
            {
                command.pListener->CharacteristicReadComplete(pGattResult->charcteristicUuid);
            }
        }
        else
        {
            command.pListener->NotifyGattOperationError(pGattResult->status);
        }
    }

    RemoveHead();
    return returnValue;
}

bool BleNbatClient::HandleWriteCompletedEvent(nn::bluetooth::InfoFromLeGattOperationCallback* pGattResult) NN_NOEXCEPT
{
    // キューに1つもない
    if (m_OperationCount == 0)
    {
        return false;
    }

    auto& command = m_OperationQueue[m_OperationQueueHead];

    // Callback を待っていない
    if (command.opStatus != GattOperationStatus_Sent)
    {
        return false;
    }

    // コールバック待ちの GATT 操作かどうか
    if (command.opType == GattOperationType_WriteCharacteristic)
    {
        if (pGattResult->charcteristicUuid != command.parameter.characteristic.uuid)
        {
            return false;
        }
    }
    else if (command.opType == GattOperationType_WriteDescriptor)
    {
        if (pGattResult->descriptorUuid != command.parameter.descriptor.uuid)
        {
            return false;
        }
    }

    bool returnValue = true;
    // コールバックの status がエラー
    if (pGattResult->status != nn::bluetooth::BT_OK)
    {
        NN_DETAIL_XCD_ERROR("Nintendo Battery Service: GattOperation error char=%s status=%d operation=0x%02X\n",
            GetName(pGattResult->charcteristicUuid), pGattResult->status, pGattResult->operation);
        returnValue = false;
    }
    else
    {
        if (command.opType == GattOperationType_WriteDescriptor)
        {
            if (pGattResult->charcteristicUuid == BatteryLevelCharacteristic.Uuid &&
                pGattResult->descriptorUuid == ClientConfigDescriptor.Uuid)
            {
                NN_DETAIL_XCD_INFO("NBAT : Battery Level Indication Enabled\n");
                this->ReadBatteryLevelAsync();
            }
            if (pGattResult->charcteristicUuid == BatteryStateCharacteristic.Uuid &&
                pGattResult->descriptorUuid == ClientConfigDescriptor.Uuid)
            {
                NN_DETAIL_XCD_INFO("NBAT : Battery State Indication Enabled\n");
                this->ReadBatteryStateAsync();
            }
        }
        returnValue = true;
    }

    if (command.pListener != nullptr)
    {
        if (returnValue == true)
        {
            if (command.opType == GattOperationType_WriteDescriptor)
            {
                command.pListener->DescriptorWriteComplete(pGattResult->charcteristicUuid, pGattResult->descriptorUuid);
            }
            else if (command.opType == GattOperationType_WriteCharacteristic)
            {
                command.pListener->CharacteristicWriteComplete(pGattResult->charcteristicUuid);
            }
        }
        else
        {
            command.pListener->NotifyGattOperationError(pGattResult->status);
        }
    }

    RemoveHead();
    return returnValue;
}

bool BleNbatClient::HandleNotifyCompletedEvent(nn::bluetooth::InfoFromLeGattOperationCallback* pGattResult) NN_NOEXCEPT
{
    bool returnValue = false;
    if (pGattResult->charcteristicUuid == BatteryLevelCharacteristic.Uuid)
    {
        returnValue = this->HandleReadBatteryLevel(pGattResult->value, pGattResult->length);
    }
    if (pGattResult->charcteristicUuid == BatteryStateCharacteristic.Uuid)
    {
        returnValue = this->HandleReadBatteryState(pGattResult->value, pGattResult->length);
    }

    return returnValue;
}

const char* BleNbatClient::GetName(const nn::bluetooth::GattAttributeUuid& uuid) NN_NOEXCEPT
{
    if (uuid == BatteryLevelCharacteristic.Uuid)
    {
        return "BatteryLevel";
    }
    else if (uuid == BatteryStateCharacteristic.Uuid)
    {
        return "BatteryState";
    }
    else if (uuid == ClientConfigDescriptor.Uuid)
    {
        return "ClientConfig";
    }
    else
    {
        return "Unknown";
    }
}

bool BleNbatClient::HandleReadBatteryLevel(uint8_t* pValue, int length) NN_NOEXCEPT
{
    if (length > 0)
    {
        switch (pValue[0])
        {
        case 0:
            m_BatteryLevel = BatteryLevel_None;
            break;
        case 1:
            m_BatteryLevel = BatteryLevel_CriticalLow;
            break;
        case 2:
            m_BatteryLevel = BatteryLevel_Low;
            break;
        case 3:
            m_BatteryLevel = BatteryLevel_Midium;
            break;
        case 4:
            m_BatteryLevel = BatteryLevel_High;
            break;
        default:
            m_BatteryLevel = BatteryLevel_None;
        }
    }
    else
    {
        m_BatteryLevel = BatteryLevel_None;
    }
    NN_DETAIL_XCD_INFO("NBAT: Battery Level Updated %d\n", m_BatteryLevel);
    m_Flags.Set<BleNbatFlag::IsBatteryLevelReadDone>();
    return true;
}

bool BleNbatClient::HandleReadBatteryState(uint8_t* pValue, int length) NN_NOEXCEPT
{
    if (length > 0)
    {
        m_Flags.Set<BleNbatFlag::IsCharging>(pValue[0] & 0x01);
        m_Flags.Set<BleNbatFlag::IsPowered>(pValue[0] & 0x02);
    }
    else
    {
        m_Flags.Set<BleNbatFlag::IsCharging>(false);
        m_Flags.Set<BleNbatFlag::IsPowered>(false);
    }
    NN_DETAIL_XCD_INFO("NBAT: Battery State Updated %d\n", pValue[0]);
    m_Flags.Set<BleNbatFlag::IsBatteryStateReadDone>();
    return true;
}

void BleNbatClient::ReadBatteryLevelAsync() NN_NOEXCEPT
{
    if (m_CharacteristicParameterList[CharacteristicIndex_BatteryLevel].isValid == true)
    {
        Enqueue(GattOperationType_ReadCharacteristic, 0, m_CharacteristicParameterList[CharacteristicIndex_BatteryLevel], nullptr);
    }
}

void BleNbatClient::ReadBatteryStateAsync() NN_NOEXCEPT
{
    if (m_CharacteristicParameterList[CharacteristicIndex_BatteryState].isValid == true)
    {
        Enqueue(GattOperationType_ReadCharacteristic, 0, m_CharacteristicParameterList[CharacteristicIndex_BatteryState], nullptr);
    }
}

void BleNbatClient::EnableBatteryLevelIndication() NN_NOEXCEPT
{
    if (m_DescriptorParameterList[DescriptorIndex_BatteryLevel_ClientConfig].isValid == true)
    {
        SetNotification(m_ConnectionHandle, m_DescriptorParameterList[DescriptorIndex_BatteryLevel_ClientConfig], true);
        GattOperationPayload payload = { sizeof(uint16_t), &EnableIndicationRawData };
        Enqueue(GattOperationType_WriteDescriptor, 0, m_DescriptorParameterList[DescriptorIndex_BatteryLevel_ClientConfig], payload, nullptr);
    }
    else
    {
        // 後方互換対策
        m_BatteryLevel = BatteryLevel_None;
        m_Flags.Set<BleNbatFlag::IsBatteryLevelReadDone>();
    }
}

void BleNbatClient::EnableBatteryStateIndication() NN_NOEXCEPT
{
    if (m_DescriptorParameterList[DescriptorIndex_BatteryState_ClientConfig].isValid == true)
    {
        SetNotification(m_ConnectionHandle, m_DescriptorParameterList[DescriptorIndex_BatteryState_ClientConfig], true);
        GattOperationPayload payload = { sizeof(uint16_t), &EnableIndicationRawData };
        Enqueue(GattOperationType_WriteDescriptor, 0, m_DescriptorParameterList[DescriptorIndex_BatteryState_ClientConfig], payload, nullptr);
    }
    else
    {
        // 後方互換対策
        m_Flags.Set<BleNbatFlag::IsBatteryStateReadDone>();
    }
}

}}} // namespace nn::xcd::detail
