﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Macro.h>
#include <nn/nn_Result.h>
#include <nn/nn_SdkAssert.h>
#include <nn/bluetooth/bluetooth_Api.h>
#include <nn/bluetooth/bluetooth_BleScanParameterIdPalma.h>
#include <nn/btm/btm_Api.h>
#include <nn/btm/btm_Result.h>
#include <nn/btm/user/btm_Result.user.h>
#include <nn/os/os_Event.h>
#include <nn/os/os_SystemEvent.h>
#include <nn/os/os_MultipleWait.h>
#include <nn/os/os_Thread.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/util/util_ScopeExit.h>
#include <nn/xcd/xcd_Result.h>
#include <nn/xcd/xcd_ResultForPrivate.h>
#include <nn/xcd/detail/xcd_Log.h>

#include "xcd_BleCentralTask-hardware.nx.h"
#include "xcd_BleGattUuids.h"
#include "../xcd_DeviceHandleGenerator.h"
#include "../xcd_TaskManager.h"

namespace nn { namespace xcd { namespace detail {

const nn::bluetooth::GattAttributeUuid BleCentralTask::GattServiceUuidList[] =
{
    NhogService.Uuid,
    NbatService.Uuid
};


void BleCentralTask::EventFunction(const ::nn::os::MultiWaitHolderType* pMultiWaitHolder) NN_NOEXCEPT
{
    if (pMultiWaitHolder == m_GattOperationEvent.GetWaitId())
    {
        HandleGattOperationEvent();
    }
}

void BleCentralTask::PeriodicEventFunction() NN_NOEXCEPT
{
    if (nn::os::TryWaitSystemEvent(&m_ScanEvent))
    {
        nn::os::ClearSystemEvent(&m_ScanEvent);
        this->HandleScanEvent();
    }
    else if (nn::os::TryWaitSystemEvent(&m_ConnectionEvent))
    {
        nn::os::ClearSystemEvent(&m_ConnectionEvent);
        HandleConnectionEvent();
    }
    else if (nn::os::TryWaitSystemEvent(&m_MtuConfigEvent))
    {
        nn::os::ClearSystemEvent(&m_MtuConfigEvent);
        HandleMtuConfigEvent();
    }
    else if (nn::os::TryWaitSystemEvent(&m_ServiceDiscoveryEvent))
    {
        nn::os::ClearSystemEvent(&m_ServiceDiscoveryEvent);
        HandleServiceDiscoveryEvent();
    }
    else if (nn::os::TryWaitTimerEvent(&m_BtmRetryEvent))
    {
        HandleBtmRetryEvent();
    }

    PeriodicGattClientOperation();
}

void BleCentralTask::HandleScanEvent() NN_NOEXCEPT
{
    if (m_AllowPalmaScanAll == false || m_DeviceCount >= BleDeviceCountMax)
    {
        return;
    }
    ::nn::btm::user::ScanResult scanResults[::nn::bluetooth::BleScanResultCountMax / 2];
    auto count = ::nn::btm::GetBleScanResultsForGeneral(scanResults, NN_ARRAY_SIZE(scanResults));
    for (int deviceIndex = 0; deviceIndex < count; ++deviceIndex)
    {
        auto& device = scanResults[deviceIndex];
        for (int advIndex = 0; advIndex < device.adStructureNum; ++advIndex)
        {
            if (device.adStructures[advIndex].adType == 0xff)
            {
                if (std::memcmp(m_AdvertiseParameter.manufacturerId, &device.adStructures[advIndex].data[0], nn::bluetooth::BleNnAdvertiseManufacturerIdSize) == 0 &&
                    std::memcmp(m_AdvertiseParameter.clientId, &device.adStructures[advIndex].data[3], nn::bluetooth::BleNnAdvertiseManufactureClientIdSize) == 0 &&
                    std::memcmp(m_AdvertiseParameter.serverId, &device.adStructures[advIndex].data[6], nn::bluetooth::BleNnAdvertiseManufactureServerIdSize) == 0)
                {
                    nn::btm::BleConnect(device.address);
                    continue;
                }
            }
        }
    }
}

void BleCentralTask::HandleConnectionEvent() NN_NOEXCEPT
{
    ConnectionStateList stateList;
    uint8_t connectedNum = nn::btm::BleGetConnectionState(stateList.state, NN_ARRAY_SIZE(stateList.state));
    stateList.deviceCount = connectedNum;

    HandleDetachedDevices(&stateList);
    HandleAttachedDevices(&stateList);
}

void BleCentralTask::HandleMtuConfigEvent() NN_NOEXCEPT
{
    for (auto& device : m_Devices)
    {
        if (device.accessor.IsActivated() == true)
        {
            device.accessor.HandleMtuConfigCallback();
        }
    }
}

void BleCentralTask::HandleServiceDiscoveryEvent() NN_NOEXCEPT
{
    for (auto& device : m_Devices)
    {
        if (device.accessor.IsActivated() == true)
        {
            bool wasDeactivated = (device.accessor.GetBleNhogClient() == nullptr);
            wasDeactivated &= (device.accessor.GetBleNbatClient() == nullptr);

            // GATT クライアントを有効にする
            if (device.accessor.ActivateGattClient())
            {
                if (wasDeactivated)
                {
                    ++m_DeviceCount;
                    UpdateScanStateAll();
                    if (m_DeviceCount > BleDeviceCountMax)
                    {
                        device.accessor.DetachDevice();
                    }
                }
            }

            // 上位のデバイス情報の更新を通知
            nn::os::SignalLightEvent(m_pUpdatedEvent);
        }
    }
}

void BleCentralTask::HandleGattOperationEvent() NN_NOEXCEPT
{
    nn::bluetooth::BleEventType eventType;
    uint8_t buffer[nn::bluetooth::BUFFER_SIZE_OF_BLE_OUT];

    nn::bluetooth::GetLeHidEventInfo(&eventType, buffer, nn::bluetooth::BUFFER_SIZE_OF_BLE_OUT);

    switch (eventType)
    {
        case nn::bluetooth::EventFromLeClientGattOpCallback:
        {
            auto pResult = reinterpret_cast<nn::bluetooth::InfoFromLeGattOperationCallback*>(buffer);

            for (auto& device : m_Devices)
            {
                // GattOperationCallback の対象の接続ハンドルを持つデバイスが見つかったら処理をする
                if (pResult->connId == device.accessor.GetConnectionHandle())
                {
                    device.accessor.HandleGattOperationCallback(pResult);
                }
            }
            break;
        }
        case nn::bluetooth::EventFromLeClientServiceDiscoveryStateChangedCallback:
        case nn::bluetooth::EventFromLeClientStateChangedCallback:
        case nn::bluetooth::EventFromLeServerStateChangedCallback:
        case nn::bluetooth::EventFromLeConnParamUpdateCallback:
        case nn::bluetooth::EventFromLeClientConnStateChangedCallback:
        case nn::bluetooth::EventFromLeServerConnStateChangedCallback:
        case nn::bluetooth::EventFromLeScanStateChangedCallback:
        case nn::bluetooth::EventFromLeScanFilterStateChangedCallback:
        case nn::bluetooth::EventFromLeClientServiceDiscoveryCallback:
        case nn::bluetooth::EventFromLeClientConfigureMtuCallback:
        case nn::bluetooth::EventFromLeServerProfileChangedCallback:
        case nn::bluetooth::EventFromLeServerGattReqCallback:
            break;
        default:
            NN_UNEXPECTED_DEFAULT;
    }
}

void BleCentralTask::HandleDetachedDevices(const ConnectionStateList* pList) NN_NOEXCEPT
{
    for (auto& device : m_Devices)
    {
        if (device.accessor.IsActivated() == true)
        {
            bool isDetached = true;

            for (int i = 0; i < pList->deviceCount; ++i)
            {
                if (device.accessor.GetConnectionHandle() == pList->state[i].connectionHandle)
                {
                    isDetached = false;
                    break;
                }
            }

            if (isDetached == true)
            {
                if (RemoveDevice(device.accessor.GetConnectionHandle()) == true)
                {
                    // 上位のデバイス情報の更新を通知
                    nn::os::SignalLightEvent(m_pUpdatedEvent);
                }
            }
        }
    }
}

void BleCentralTask::HandleAttachedDevices(const ConnectionStateList* pList) NN_NOEXCEPT
{
    for (int i = 0; i < pList->deviceCount; ++i)
    {
        bool isAttached = true;

        for (auto& device : m_Devices)
        {
            if (device.accessor.IsActivated() == true &&
                device.accessor.GetConnectionHandle() == pList->state[i].connectionHandle)
            {
                isAttached = false;
                break;
            }
        }

        if (isAttached == true)
        {
            if (RegisterDevice(pList->state[i].address, pList->state[i].connectionHandle) == false)
            {
                // 割り当てができない場合は切断する
                ::nn::btm::BleDisconnect(pList->state[i].connectionHandle);
            }
        }
    }
}

BleCentralTask::BleCentralTask() NN_NOEXCEPT :
    m_pUpdatedEvent(nullptr),
    m_IsSuspended(false),
    m_DeviceCount(0),
    m_AllowPalmaScanAll(false),
    m_AllowPalmaScanRegistered(false)
{
    m_RetryFlags.Reset();
}

BleCentralTask::~BleCentralTask() NN_NOEXCEPT
{
    // 何もしない
}

nn::Result BleCentralTask::Activate(nn::os::LightEventType* pUpdatedEvent) NN_NOEXCEPT
{
    m_pUpdatedEvent = pUpdatedEvent;

    // BTM に Scan イベントをセット
    nn::btm::AcquireBleScanEvent(&m_ScanEvent);

    // BTM に BLE 接続イベントをセット
    nn::btm::AcquireBleConnectionEvent(&m_ConnectionEvent);

    // BTM に MTU 設定完了イベントをセット
    nn::btm::AcquireBleMtuConfigEvent(&m_MtuConfigEvent);

    // BTM にサービス発見イベントをセット
    nn::btm::AcquireBleServiceDiscoveryEvent(&m_ServiceDiscoveryEvent);

    // HID 向けの GATT オペレーション完了イベントをセット
    nn::bluetooth::RegisterBleHidEvent(m_GattOperationEvent.GetBase());

    GetTaskManager().RegisterPeriodicTask(this);
    GetTaskManager().RegisterEvent(&m_GattOperationEvent, this);

    // BTM に GATT データパスを設定する
    for (auto& uuid : GattServiceUuidList)
    {
        nn::btm::user::BleDataPath path =
        {
            nn::btm::user::BLE_DATA_PATH_HID,
            uuid
        };
        nn::btm::RegisterBleGattDataPath(path);
    }

    ::nn::os::InitializeTimerEvent(&m_BtmRetryEvent, ::nn::os::EventClearMode_AutoClear);
    nn::btm::GetBleScanFilterParameter(&m_AdvertiseParameter, nn::bluetooth::BleScanParameterId_Palma);

    m_IsSuspended = false;
    m_DeviceCount = 0;
    m_AllowPalmaScanAll = false;
    m_AllowPalmaScanRegistered = false;

    m_RetryFlags.Reset();

    NN_RESULT_SUCCESS;
}

void BleCentralTask::Deactivate() NN_NOEXCEPT
{
    GetTaskManager().UnregisterEvent(&m_GattOperationEvent);
    GetTaskManager().UnregisterPeriodicTask(this);
    ::nn::os::FinalizeTimerEvent(&m_BtmRetryEvent);
}

void BleCentralTask::Suspend() NN_NOEXCEPT
{
    if (m_IsSuspended == true)
    {
        return;
    }

    GetTaskManager().UnregisterPeriodicTask(this);
    GetTaskManager().UnregisterEvent(&m_GattOperationEvent);

    m_IsSuspended = true;
}

void BleCentralTask::Resume() NN_NOEXCEPT
{
    if (m_IsSuspended == false)
    {
        return;
    }

    // Ble のデバイスリストを削除
    for (auto& device : m_Devices)
    {
        if (device.accessor.IsActivated() == true)
        {
            device.accessor.Deactivate();
        }
    }
    // デバイスリストの更新を通知
    nn::os::SignalLightEvent(m_pUpdatedEvent);

    // 各種イベントを link して駆動を再開する
    GetTaskManager().RegisterEvent(&m_GattOperationEvent, this);
    GetTaskManager().RegisterPeriodicTask(this);

    m_IsSuspended = false;
}

size_t BleCentralTask::GetConnectionHandleList(uint32_t* pOutList, size_t deviceCount) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutList);

    size_t returnCount = 0;
    for (auto& device : m_Devices)
    {
        if (device.accessor.IsActivated() == true && device.accessor.IsGattClientActivated() == true)
        {
            pOutList[returnCount] = device.accessor.GetConnectionHandle();
            returnCount++;

            if (returnCount == deviceCount || returnCount >= m_DeviceCount)
            {
                break;
            }
        }
    }

    return returnCount;
}

BleHidAccessor* BleCentralTask::GetBleHidAccessor(uint32_t connectionHandle) NN_NOEXCEPT
{
    for (int i = 0; i < NN_ARRAY_SIZE(m_Devices); i++)
    {
        if (m_Devices[i].accessor.GetConnectionHandle() == connectionHandle)
        {
            return &m_Devices[i].accessor;
        }
    }

    // みつからなかった
    return nullptr;
}

void BleCentralTask::SetIsPalmaAllConnectable(bool connectable) NN_NOEXCEPT
{
    if (m_AllowPalmaScanAll != connectable)
    {
        m_AllowPalmaScanAll = connectable;
        this->UpdateGeneralScanState();
    }
}

void BleCentralTask::SetIsPalmaPairedConnectable(bool connectable) NN_NOEXCEPT
{
    if (m_AllowPalmaScanRegistered != connectable)
    {
        m_AllowPalmaScanRegistered = connectable;
        this->UpdateAutoConnectState();
    }
}

Result BleCentralTask::PairPalma(const ::nn::bluetooth::Address& address) NN_NOEXCEPT
{
    for (auto& device : m_Devices)
    {
        if (device.accessor.IsActivated())
        {
            if (device.address == address)
            {
                device.pendingPairing = true;
                TryPairPalma();
                NN_RESULT_SUCCESS;
            }
        }
    }

    NN_RESULT_THROW(ResultNotConnected());
}

bool BleCentralTask::RegisterDevice(const nn::bluetooth::Address& address, uint32_t connectionHandle) NN_NOEXCEPT
{
    for (auto& device : m_Devices)
    {
        if (device.accessor.IsActivated() == false)
        {
            device.address = address;
            device.accessor.Activate(connectionHandle);
            device.accessor.SetDeviceAddress(address);

            return true;
        }
    }

    return false;
}

bool BleCentralTask::RemoveDevice(uint32_t connectionHandle) NN_NOEXCEPT
{
    auto device = GetBleDeviceFromConnectionHandle(connectionHandle);
    if (device == nullptr)
    {
        return false;
    }

    if (device->accessor.GetBleNhogClient() != nullptr ||
        device->accessor.GetBleNbatClient() != nullptr)
    {
        --m_DeviceCount;
        UpdateScanStateAll();
    }

    device->accessor.Deactivate();

    return true;
}

void BleCentralTask::PeriodicGattClientOperation() NN_NOEXCEPT
{
    // アクティブなデバイスの GATT クライアントの処理をポーリングする
    for (auto& device : m_Devices)
    {
        if (device.accessor.IsActivated() == true)
        {
            device.accessor.PeriodicOperation();
        }
    }
}

BleCentralTask::BleDevice* BleCentralTask::GetBleDeviceFromConnectionHandle(uint32_t connectionHandle) NN_NOEXCEPT
{
    for (auto& device : m_Devices)
    {
        if (device.accessor.IsActivated() == true)
        {
            if (device.accessor.GetConnectionHandle() == connectionHandle)
            {
                return &device;
            }
        }
    }
    return nullptr;
}

void BleCentralTask::UpdateScanStateAll() NN_NOEXCEPT
{
    UpdateGeneralScanState();
    UpdateAutoConnectState();
}

void BleCentralTask::UpdateGeneralScanState() NN_NOEXCEPT
{
    if (m_AllowPalmaScanAll == true && m_DeviceCount < BleDeviceCountMax)
    {
        SetRetryTimerIfFail(nn::btm::StartBleScanForGeneral(m_AdvertiseParameter), &m_BtmRetryEvent, m_RetryFlags, BtmRetryFlag::GeneralScan::Index);
    }
    else
    {
        SetRetryTimerIfFail(nn::btm::StopBleScanForGeneral(), &m_BtmRetryEvent, m_RetryFlags, BtmRetryFlag::GeneralScan::Index);
    }
}

void BleCentralTask::UpdateAutoConnectState() NN_NOEXCEPT
{
    if (m_AllowPalmaScanRegistered == true && m_DeviceCount < BleDeviceCountMax)
    {
        SetRetryTimerIfFail(nn::btm::StartBleScanForPaired(m_AdvertiseParameter), &m_BtmRetryEvent, m_RetryFlags, BtmRetryFlag::AutoConnect::Index);
    }
    else
    {
        SetRetryTimerIfFail(nn::btm::StopBleScanForPaired(), &m_BtmRetryEvent, m_RetryFlags, BtmRetryFlag::AutoConnect::Index);
    }
}

void BleCentralTask::TryPairPalma() NN_NOEXCEPT
{
    bool retryRequired = false;
    for (auto& device : m_Devices)
    {
        if (device.pendingPairing == true)
        {
            if (nn::btm::BlePairDevice(device.accessor.GetConnectionHandle(), m_AdvertiseParameter).IsSuccess())
            {
                device.pendingPairing = false;
            }
            else
            {
                retryRequired = true;
            }
        }
    }

    if (retryRequired == true)
    {
        SetRetryTimer(&m_BtmRetryEvent, m_RetryFlags, BtmRetryFlag::PairGattServer::Index);
    }
}

void BleCentralTask::HandleBtmRetryEvent() NN_NOEXCEPT
{
    m_RetryFlags.Reset<BtmRetryFlag::Retrying>();

    if (m_RetryFlags.Test<BtmRetryFlag::GeneralScan>())
    {
        m_RetryFlags.Reset<BtmRetryFlag::GeneralScan>();
        UpdateGeneralScanState();
    }
    if (m_RetryFlags.Test<BtmRetryFlag::AutoConnect>())
    {
        m_RetryFlags.Reset<BtmRetryFlag::AutoConnect>();
        UpdateAutoConnectState();
    }
    if (m_RetryFlags.Test<BtmRetryFlag::PairGattServer>())
    {
        m_RetryFlags.Reset<BtmRetryFlag::PairGattServer>();
        TryPairPalma();
    }
}

}}} // namespace nn::xcd::detail
