﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Abort.h>
#include <nn/nn_SdkAssert.h>
#include <nn/nn_SdkLog.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/wlan/wlan_Result.h>
#include <nn/wlan/wlan_LocalApi.h>
#include <nn/wlan/detail/wlan_InternalTypes.h>
#include <nn/sf/sf_Types.h>
#include "wlan_CreateWlanManagers.h"
#include "../../Processes/wlan/wlan_SignalStrength.h"

namespace nn {
namespace wlan {

namespace {

nn::sf::SharedPointer<detail::ILocalManager> g_LocalManager;
nn::sf::HipcSimpleClientSessionManager g_Manager;
nn::sf::SharedPointer<detail::ILocalGetFrame> g_LocalGetFrameManager;
nn::sf::HipcSimpleClientSessionManager g_GetFrameManager;
nn::sf::SharedPointer<detail::ILocalGetActionFrame> g_LocalGetActionFrameManager;
nn::sf::HipcSimpleClientSessionManager g_GetActionFrameManager;

// Initialize の参照カウント
int g_LocalInitializeCount = 0;
}

nn::Result InitializeLocalManager() NN_NOEXCEPT
{
    NN_SDK_REQUIRES(!g_LocalManager);
    NN_SDK_REQUIRES(!g_LocalGetFrameManager);
    NN_SDK_REQUIRES(!g_LocalGetActionFrameManager);
    if( g_LocalInitializeCount == 0 )
    {
        g_LocalManager = nn::wlan::CreateLocalManagerByHipc(&g_Manager);
        g_LocalGetFrameManager = nn::wlan::CreateLocalGetFrameManagerByHipc(&g_GetFrameManager);
        g_LocalGetActionFrameManager = nn::wlan::CreateLocalGetActionFrameManagerByHipc(&g_GetActionFrameManager);
        g_LocalInitializeCount++;
    }
    NN_RESULT_SUCCESS;
}

nn::Result FinalizeLocalManager() NN_NOEXCEPT
{
    NN_SDK_REQUIRES(g_LocalManager);
    NN_SDK_REQUIRES(g_LocalGetFrameManager);
    NN_SDK_REQUIRES(g_LocalGetActionFrameManager);
    if( g_LocalInitializeCount > 0 )
    {
        g_LocalManager = nullptr;
        g_Manager.Finalize();
        g_LocalGetFrameManager = nullptr;
        g_GetFrameManager.Finalize();
        g_LocalGetActionFrameManager = nullptr;
        g_GetActionFrameManager.Finalize();
        g_LocalInitializeCount--;
    }
    NN_RESULT_SUCCESS;
}

namespace Local {

nn::Result OpenMasterMode() NN_NOEXCEPT
{
    NN_SDK_REQUIRES(g_LocalManager);
    auto result = g_LocalManager->OpenMasterMode();
    return result;
}

nn::Result CloseMasterMode() NN_NOEXCEPT
{
    NN_SDK_REQUIRES(g_LocalManager);
    auto result = g_LocalManager->CloseMasterMode();
    return result;
}

nn::Result OpenClientMode() NN_NOEXCEPT
{
    NN_SDK_REQUIRES(g_LocalManager);
    auto result = g_LocalManager->OpenClientMode();
    return result;
}

nn::Result CloseClientMode() NN_NOEXCEPT
{
    NN_SDK_REQUIRES(g_LocalManager);
    auto result = g_LocalManager->CloseClientMode();
    return result;
}

nn::Result OpenSpectatorMode() NN_NOEXCEPT
{
    NN_SDK_REQUIRES(g_LocalManager);
    auto result = g_LocalManager->OpenSpectatorMode();
    return result;
}

nn::Result CloseSpectatorMode() NN_NOEXCEPT
{
    NN_SDK_REQUIRES(g_LocalManager);
    auto result = g_LocalManager->CloseSpectatorMode();
    return result;
}

nn::Result OpenLcsMasterMode() NN_NOEXCEPT
{
    NN_SDK_REQUIRES(g_LocalManager);
    auto result = g_LocalManager->OpenLcsMasterMode();
    return result;
}

nn::Result CloseLcsMasterMode() NN_NOEXCEPT
{
    NN_SDK_REQUIRES(g_LocalManager);
    auto result = g_LocalManager->CloseLcsMasterMode();
    return result;
}

nn::Result OpenLcsClientMode() NN_NOEXCEPT
{
    NN_SDK_REQUIRES(g_LocalManager);
    auto result = g_LocalManager->OpenLcsClientMode();
    return result;
}

nn::Result CloseLcsClientMode() NN_NOEXCEPT
{
    NN_SDK_REQUIRES(g_LocalManager);
    auto result = g_LocalManager->CloseLcsClientMode();
    return result;
}

nn::Result CreateBss(const MasterBssParameters& bssParameters) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(g_LocalManager);
    // パラメータチェック
    if( !(bssParameters.channel == -1 ||
        (bssParameters.channel >= WirelessChannel_1ch && bssParameters.channel <= WirelessChannel_11ch) ||
        bssParameters.channel == WirelessChannel_36ch ||
        bssParameters.channel == WirelessChannel_40ch ||
        bssParameters.channel == WirelessChannel_44ch ||
        bssParameters.channel == WirelessChannel_48ch)
    )
    {
        NN_SDK_REQUIRES(false, "%s: channel is invalid\n", __FUNCTION__);
        return nn::wlan::ResultInvalidArgument();
    }
    if( bssParameters.inactivePeriod > 180 )
    {
        NN_SDK_REQUIRES(false, "%s: inactivePeriod must be less than 180\n", __FUNCTION__);
        return nn::wlan::ResultInvalidArgument();
    }
    if( !(bssParameters.security.privacyMode == SecurityMode_Open || bssParameters.security.privacyMode == SecurityMode_StaticAes) )
    {
        NN_SDK_REQUIRES(false, "%s: privacyMode must be Open or StaticAES\n", __FUNCTION__);
        return nn::wlan::ResultInvalidArgument();
    }
    if( bssParameters.ssid.GetLength() == 0 )
    {
        NN_SDK_REQUIRES(false, "%s: SSID must not be empty\n", __FUNCTION__);
        return nn::wlan::ResultInvalidArgument();
    }

    nn::wlan::detail::SfdlMasterBssParameters masterParams;
    masterParams.channel = bssParameters.channel;
    masterParams.hiddenSsid = bssParameters.hiddenSsid;
    masterParams.inactivePeriod = bssParameters.inactivePeriod;
    masterParams.autoKeepAlive = bssParameters.autoKeepAlive;
    masterParams.supportedRates = bssParameters.supportedRates;
    masterParams.basicRates = bssParameters.basicRates;
    masterParams.security.privacyMode = bssParameters.security.privacyMode;
    masterParams.security.groupPrivacyMode = bssParameters.security.groupPrivacyMode;
    masterParams.security.keyIdx = 0;
    std::memcpy(masterParams.security.key, bssParameters.security.key, sizeof(bssParameters.security.key));
    masterParams.ssid.length = bssParameters.ssid.GetLength();
    std::memcpy(masterParams.ssid.data, bssParameters.ssid.GetSsidData(), bssParameters.ssid.GetLength());
    masterParams.beaconInterval = bssParameters.beaconInterval;

    auto result = g_LocalManager->CreateBss(masterParams);

    return result;
}

nn::Result DestroyBss() NN_NOEXCEPT
{
    NN_SDK_REQUIRES(g_LocalManager);
    auto result = g_LocalManager->DestroyBss();
    return result;
}

nn::Result StartScan(void* pOutScanBuffer, size_t size, const ScanParameters& param, const ScanIeMatchInfo* pIeInfo) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(g_LocalManager);
    // パラメータチェック
    if( pOutScanBuffer == NULL )
    {
        NN_SDK_REQUIRES(false, "%s: pOutScanBuffer must not be NULL\n", __FUNCTION__);
        return nn::wlan::ResultInvalidArgument();
    }
    if( size <= sizeof(nn::wlan::ScanResultHeader) )
    {
        NN_SDK_REQUIRES(false, "%s: scan buffer size must be larger than %d\n", __FUNCTION__, sizeof(nn::wlan::ScanResultHeader));
        return nn::wlan::ResultInvalidArgument();
    }
    if( !(param.scanType == nn::wlan::ScanType_Active || param.scanType == nn::wlan::ScanType_Passive) )
    {
        NN_SDK_REQUIRES(false, "%s: scanType is invalid\n", __FUNCTION__);
        return nn::wlan::ResultInvalidArgument();
    }
    if( !(param.channelScanTime == -1 || (param.channelScanTime >= 10 && param.channelScanTime <= 3000)) )
    {
        NN_SDK_REQUIRES(false, "%s: channelScanTime is invalid\n", __FUNCTION__);
        return nn::wlan::ResultInvalidArgument();
    }
    if( !(param.homeChannelTime == -1 || (param.homeChannelTime >= 0 && param.homeChannelTime <= 500)) )
    {
        NN_SDK_REQUIRES(false, "%s: homeChannelTime is invalid\n", __FUNCTION__);
        return nn::wlan::ResultInvalidArgument();
    }
    if( param.bssid == nn::wlan::MacAddress::CreateZeroMacAddress() )
    {
        NN_SDK_REQUIRES(false, "%s: bssid must not be ZeroMacAddress\n", __FUNCTION__);
        return nn::wlan::ResultInvalidArgument();
    }

    nn::Result result;

    // IPC用構造体へ値をコピー
    nn::wlan::detail::SfdlScanParameters scanParams;
    scanParams.scanType = param.scanType;
    scanParams.channelCount = param.channelCount;
    for(int i = 0; i < param.channelCount; i++)
    {
        // チャンネルチェック
        if( param.channelList[i] < WirelessChannel_1ch || param.channelList[i] > WirelessChannel_165ch )
        {
            NN_SDK_REQUIRES(false, "channel is out of range");
            return nn::wlan::ResultInvalidArgument();
        }
        scanParams.channelList.channel[i] = param.channelList[i];
    }
    scanParams.channelScanTime = param.channelScanTime;
    scanParams.homeChannelTime = param.homeChannelTime;
    scanParams.ssidCount = param.ssidCount;
    // scan対象SSIDリストのコピー判断
    if( param.ssidCount > 0 && param.ssidList != NULL )
    {
        if( param.ssidCount > nn::wlan::ScanningSsidCountMax )
        {
            NN_SDK_LOG("[WLAN][WARN]Scanning SSID count should not be over %d.\n", nn::wlan::ScanningSsidCountMax);
            scanParams.ssidCount = nn::wlan::ScanningSsidCountMax;
        }
        for(int i = 0; i < scanParams.ssidCount; i++)
        {
            if( param.ssidList[i].GetLength() == 0 )
            {
                NN_SDK_LOG("[WLAN][WARN]The length of a filtering SSID must be more than 0.\n");
                NN_SDK_ASSERT(false, "[WLAN][WARN]The length of a filtering SSID must be more than 0.");
                return nn::wlan::ResultInvalidArgument();
            }
            scanParams.ssidList[i].length = param.ssidList[i].GetLength();
            std::memcpy(&scanParams.ssidList[i].data[0], param.ssidList[i].GetSsidData(), param.ssidList[i].GetLength());
        }
    }
    else
    {
        scanParams.ssidCount = 0;  // scanParams.ssidListはNULLではないので、ssidCountを0にしておくことでIPCサーバー側でSSIDフィルタ無しと判断する。
    }

    for(uint8_t i = 0; i < nn::wlan::MacAddress::MacAddressSize; i++)
    {
        scanParams.bssid.data[i] = param.bssid.GetMacAddressData()[i];
    }

    // Scan完了通知を受け取るためのSystemEventを設定
    nn::sf::NativeHandle sfHandle;  // 出力用変数
    result = g_LocalManager->GetSystemEvent(&sfHandle, static_cast<uint32_t>(detail::SystemEventType_ScanComplete));
    NN_SDK_ASSERT( result.IsSuccess() );

    nn::os::SystemEvent scanCompEvent;
    scanCompEvent.AttachReadableHandle(sfHandle.GetOsHandle(), sfHandle.IsManaged(), nn::os::EventClearMode_AutoClear);
    sfHandle.Detach();

    result = g_LocalManager->StartScan(scanParams);
    if( result.IsFailure() )
    {
        return result;
    }

    // Scan完了待機
    scanCompEvent.Wait();

    // Scanが完了したので結果を取りに行く

    // 出力用バッファの作成
    nn::sf::OutBuffer pOutBuffer(reinterpret_cast<char*>(pOutScanBuffer), size);
    if( pIeInfo == NULL )
    {
        // ゼロ埋めしたものを渡しておく
        ScanIeMatchInfo info;
        info.matchLength = 0;
        std::memset(&info.matchData[0], 0x00, 255);
        result = g_LocalManager->GetScanResult(pOutBuffer, info);
    }
    else
    {
        result = g_LocalManager->GetScanResult(pOutBuffer, static_cast<const ScanIeMatchInfo&>(*pIeInfo));
    }

    return result;
} //NOLINT(impl/function_size)


nn::Result StopScan() NN_NOEXCEPT
{
     NN_SDK_REQUIRES(g_LocalManager);
     auto result = g_LocalManager->StopScan();
     return result;
}


nn::Result Connect(const Ssid& ssid, const MacAddress& bssid, int16_t channel, const Security& security, bool autoKeepAlive,
        BeaconIndication indication, int beaconLostTimeout) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(g_LocalManager);
    // パラメータチェック
    if( bssid == nn::wlan::MacAddress::CreateZeroMacAddress() )
    {
        NN_SDK_REQUIRES(false, "%s: bssid must not be ZeroMacAddress\n", __FUNCTION__);
        return nn::wlan::ResultInvalidArgument();
    }
    else if( bssid == nn::wlan::MacAddress::CreateBroadcastMacAddress() )
    {
        // SSIDのチェック
        // BSSIDがBroadcast指定の場合、空SSIDは無効
        if( ssid.GetLength() == 0 )
        {
            NN_SDK_REQUIRES(false, "%s: SSID must not be empty when BSSID is broadcast address\n", __FUNCTION__);
            return nn::wlan::ResultInvalidArgument();
        }
    }
    if( !(channel == -1 || (channel >= nn::wlan::WirelessChannel_1ch && channel <= nn::wlan::WirelessChannel_165ch)) )
    {
        NN_SDK_REQUIRES(false, "%s: channel is invalid\n", __FUNCTION__);
        return nn::wlan::ResultInvalidArgument();
    }
    if( !(security.privacyMode == SecurityMode_Open || security.privacyMode == SecurityMode_StaticAes) )
    {
        NN_SDK_REQUIRES(false, "%s: privacyMode is invalid\n", __FUNCTION__);
        return nn::wlan::ResultInvalidArgument();
    }
    if( !(beaconLostTimeout >= 1 && beaconLostTimeout <= 30) )
    {
        NN_SDK_REQUIRES(false, "%s: beaconLostTimeout is invalid\n", __FUNCTION__);
        return nn::wlan::ResultInvalidArgument();
    }

    nn::Result result;

    // Connect完了通知を受け取るイベントの設定
    nn::sf::NativeHandle sfHandle;  // 出力用変数
    result = g_LocalManager->GetSystemEvent(&sfHandle, static_cast<uint32_t>(detail::SystemEventType_ConnectionComplete));
    if( result.IsFailure() )
    {
        return result;
    }

    nn::os::SystemEvent connectCompEvent;
    connectCompEvent.AttachReadableHandle(sfHandle.GetOsHandle(), sfHandle.IsManaged(), nn::os::EventClearMode_AutoClear);
    sfHandle.Detach();


    detail::SfdlSsid sfdlSsid;
    detail::SfdlMacAddress sfdlBssid;
    detail::SfdlSecurity sfdlSecurity;

    std::memcpy(&sfdlSsid.data[0], ssid.GetSsidData(), ssid.GetLength());
    sfdlSsid.length = static_cast<uint8_t>(ssid.GetLength());

    std::memcpy(&sfdlBssid.data[0], bssid.GetMacAddressData(), MacAddress::MacAddressSize);

    sfdlSecurity.privacyMode = static_cast<uint32_t>(security.privacyMode);
    sfdlSecurity.groupPrivacyMode = static_cast<uint32_t>(security.groupPrivacyMode);
    std::memcpy(&sfdlSecurity.key[0], &security.key[0], sizeof(security.key));

    result = g_LocalManager->Connect(sfdlSsid, sfdlBssid, channel, sfdlSecurity, autoKeepAlive, static_cast<uint32_t>(indication), beaconLostTimeout);
    if( result.IsFailure() )
    {
        return result;
    }

    // Connect完了待機
    connectCompEvent.Wait();

    return result;
}

nn::Result CancelConnect() NN_NOEXCEPT
{
    NN_SDK_REQUIRES(g_LocalManager);
    auto result = g_LocalManager->CancelConnect();
    return result;
}


nn::Result Join(const Ssid& ssid, const MacAddress& bssid, int16_t channel, const Security& security,
        BeaconIndication indication, int beaconLostTimeout) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(g_LocalManager);
    // パラメータチェック
    if( ssid.GetLength() == 0 )
    {
        NN_SDK_REQUIRES(false, "%s: SSID must not be empty\n", __FUNCTION__);
        return nn::wlan::ResultInvalidArgument();
    }
    if( bssid == nn::wlan::MacAddress::CreateZeroMacAddress() )
    {
        NN_SDK_REQUIRES(false, "%s: bssid must not be ZeroMacAddress\n", __FUNCTION__);
        return nn::wlan::ResultInvalidArgument();
    }
    if( !(channel == -1 || (channel >= nn::wlan::WirelessChannel_1ch && channel <= nn::wlan::WirelessChannel_165ch)) )
    {
        NN_SDK_REQUIRES(false, "%s: channel is invalid\n", __FUNCTION__);
        return nn::wlan::ResultInvalidArgument();
    }
    if( !(security.privacyMode == SecurityMode_Open || security.privacyMode == SecurityMode_StaticAes) )
    {
        NN_SDK_REQUIRES(false, "%s: privacyMode is invalid\n", __FUNCTION__);
        return nn::wlan::ResultInvalidArgument();
    }
    if( !(beaconLostTimeout >= 1 && beaconLostTimeout <= 30) )
    {
        NN_SDK_REQUIRES(false, "%s: beaconLostTimeout is invalid\n", __FUNCTION__);
        return nn::wlan::ResultInvalidArgument();
    }

    nn::Result result;

    // Connect完了通知を受け取るイベントの設定
    nn::sf::NativeHandle sfHandle;  // 出力用変数
    result = g_LocalManager->GetSystemEvent(&sfHandle, static_cast<uint32_t>(detail::SystemEventType_ConnectionComplete));
    if( result.IsFailure() )
    {
        return result;
    }

    nn::os::SystemEvent connectCompEvent;
    connectCompEvent.AttachReadableHandle(sfHandle.GetOsHandle(), sfHandle.IsManaged(), nn::os::EventClearMode_AutoClear);
    sfHandle.Detach();


    detail::SfdlSsid sfdlSsid;
    detail::SfdlMacAddress sfdlBssid;
    detail::SfdlSecurity sfdlSecurity;

    std::memcpy(&sfdlSsid.data[0], ssid.GetSsidData(), ssid.GetLength());
    sfdlSsid.length = static_cast<uint8_t>(ssid.GetLength());

    std::memcpy(&sfdlBssid.data[0], bssid.GetMacAddressData(), MacAddress::MacAddressSize);

    sfdlSecurity.privacyMode = static_cast<uint32_t>(security.privacyMode);
    std::memcpy(&sfdlSecurity.key[0], &security.key[0], sizeof(security.key));

    result = g_LocalManager->Join(sfdlSsid, sfdlBssid, channel, sfdlSecurity, indication, beaconLostTimeout);
    if( result.IsFailure() )
    {
        return result;
    }

    // Connect完了待機
    connectCompEvent.Wait();

    return result;
}


nn::Result CancelJoin() NN_NOEXCEPT
{
    NN_SDK_REQUIRES(g_LocalManager);
    auto result = g_LocalManager->CancelJoin();
    return result;
}


nn::Result Disconnect(LocalCommunicationMode mode, const DisconnectClient* pInClient) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(g_LocalManager);

    nn::wlan::detail::SfdlDisconnectClient client;

    if( mode == LocalCommunicationMode_Master )
    {
        NN_SDK_REQUIRES_NOT_NULL(pInClient);

        std::memcpy(&client.clientMacAddress.data[0], pInClient->clientMacAddress.GetMacAddressData(), MacAddress::MacAddressSize);

        // reason codeが設定されているかチェックする
        NN_SDK_REQUIRES_RANGE(pInClient->reasonCode, Dot11ReasonCode_Reserved, Dot11ReasonCode_InvalidPairwiseCipher + 1);
        client.reasonCode = pInClient->reasonCode;
    }

    auto result = g_LocalManager->Disconnect(static_cast<uint32_t>(mode), client);
    return result;
}

nn::Result GetState(WlanState* pOutState) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(g_LocalManager);
    if( pOutState == NULL )
    {
        NN_SDK_REQUIRES(false, "%s: pOutState must not be null\n", __FUNCTION__);
        return nn::wlan::ResultInvalidArgument();
    }

    uint32_t state;  // 出力用変数
    auto result = g_LocalManager->GetState(&state);
    if( result.IsFailure() )
    {
        return result;
    }

    *pOutState = static_cast<WlanState>(state);

    NN_RESULT_SUCCESS;
}

nn::Result GetAllowedChannels(int16_t (&outChannels)[WirelessChannelsCountMax], uint32_t* pOutCount) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(g_LocalManager);
    if( pOutCount == NULL )
    {
        NN_SDK_REQUIRES(false, "pOutCount must not be null");
        return nn::wlan::ResultInvalidArgument();
    }

    return g_LocalManager->GetAllowedChannels(nn::sf::OutArray<int16_t>(outChannels, WirelessChannelsCountMax), pOutCount);
}

nn::Result GetConnectionEvent(nn::os::SystemEventType* pSystemEvent) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(g_LocalManager);
    if( pSystemEvent == NULL )
    {
        NN_SDK_REQUIRES(false, "%s: pSystemEvent must not be null\n", __FUNCTION__);
        return nn::wlan::ResultInvalidArgument();
    }

    nn::sf::NativeHandle sfHandle;  // 出力用変数

    nn::Result result = g_LocalManager->GetSystemEvent(&sfHandle, static_cast<uint32_t>(detail::SystemEventType_ConnectionEvent));
    if( result.IsFailure() )
    {
        return result;
    }

    nn::os::AttachReadableHandleToSystemEvent(pSystemEvent,
                                              sfHandle.GetOsHandle(),
                                              sfHandle.IsManaged(),
                                              nn::os::EventClearMode_AutoClear);
    sfHandle.Detach();

    return result;
}

nn::Result GetConnectionStatus(ConnectionStatus* pOutStatus) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(g_LocalManager);
    if( pOutStatus == NULL )
    {
        NN_SDK_REQUIRES(false, "%s: pOutStatus must not be null\n", __FUNCTION__);
        return nn::wlan::ResultInvalidArgument();
    }

    nn::wlan::detail::SfdlConnectionStatus sfConnectionStatus;  // 出力用変数

    auto result = g_LocalManager->GetConnectionStatus(&sfConnectionStatus);
    if( result.IsFailure() )
    {
        return result;
    }

    // 結果のコピー
    pOutStatus->aid = sfConnectionStatus.aid;
    pOutStatus->beaconInterval = sfConnectionStatus.beaconInterval;
    pOutStatus->bssid.Set( sfConnectionStatus.bssid.data );
    pOutStatus->capabilityInfo = sfConnectionStatus.capabilityInfo;
    pOutStatus->cause = static_cast<CauseOfInfo>(sfConnectionStatus.cause);
    pOutStatus->channel = sfConnectionStatus.channel;
    pOutStatus->ssid.Set( sfConnectionStatus.ssid.data, static_cast<size_t>(sfConnectionStatus.ssid.length) );
    pOutStatus->state = static_cast<ConnectionState>(sfConnectionStatus.state);
    pOutStatus->statusReasonCode = sfConnectionStatus.statusReasonCode;

    return result;
}

nn::Result GetClientStatus(ClientStatus pOutStatusArray[ConnectableClientsCountMax], Bit32* pOutUpdatedIndexBitMap) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(g_LocalManager);
    if( pOutStatusArray == NULL )
    {
        NN_SDK_REQUIRES(false, "%s: pOutStatusArray must not be null\n", __FUNCTION__);
        return nn::wlan::ResultInvalidArgument();
    }
    if( pOutUpdatedIndexBitMap == NULL )
    {
        NN_SDK_REQUIRES(false, "%s: pOutUpdatedIndexBitMap must not be null\n", __FUNCTION__);
        return nn::wlan::ResultInvalidArgument();
    }

    nn::wlan::detail::SfdlClientStatusList sfStatusList;  // 出力用変数

    auto result = g_LocalManager->GetClientStatus(&sfStatusList);
    if( result.IsFailure() )
    {
        return result;
    }

    // 結果のコピー
    *pOutUpdatedIndexBitMap = sfStatusList.updatedIndexBitMap;
    for(uint8_t i = 0; i < ConnectableClientsCountMax; i++)
    {
        pOutStatusArray[i].state = static_cast<ConnectionState>(sfStatusList.statusList[i].state);
        pOutStatusArray[i].cause = static_cast<CauseOfInfo>(sfStatusList.statusList[i].cause);
        pOutStatusArray[i].clientMacAddress.Set(sfStatusList.statusList[i].clientMacAddress.data);
        pOutStatusArray[i].statusReasonCode = sfStatusList.statusList[i].statusReasonCode;
        pOutStatusArray[i].capabilityInfo = sfStatusList.statusList[i].capabilityInfo;
        pOutStatusArray[i].rssi = sfStatusList.statusList[i].rssi;
        pOutStatusArray[i].updateTick = nn::os::Tick(sfStatusList.statusList[i].updateTick);
    }

    return result;
}

nn::Result AddIe(uint32_t* pOutIeIndex, ManagementFrameType type, const uint8_t* ieBody, size_t ieBodySize) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(g_LocalManager);
    if( pOutIeIndex == NULL )
    {
        NN_SDK_REQUIRES(false, "%s: pOutIeIndex must not be NULL\n", __FUNCTION__);
        return nn::wlan::ResultInvalidArgument();
    }
    if( (type & (nn::wlan::ManagementFrameType_Beacon |
            nn::wlan::ManagementFrameType_ProbeRes |
            nn::wlan::ManagementFrameType_AssocRes |
            nn::wlan::ManagementFrameType_AuthRes |
            nn::wlan::ManagementFrameType_ProbeReq |
            nn::wlan::ManagementFrameType_AssocReq)) == 0 )
    {
        NN_SDK_REQUIRES(false, "%s: ManagementFrameType(0x%2X) is out of range\n", __FUNCTION__, type);
        return nn::wlan::ResultInvalidArgument();
    }
    if( ieBody == NULL )
    {
        NN_SDK_REQUIRES(false, "%s: ieBody must not be NULL\n", __FUNCTION__);
        return nn::wlan::ResultInvalidArgument();
    }
    if( ieBodySize < 1 || ieBodySize > 255  )
    {
        NN_SDK_REQUIRES(false, "%s: ieBodySize(%d) is invalid\n", __FUNCTION__, ieBodySize);
        return nn::wlan::ResultInvalidArgument();
    }

    uint32_t IeIndex;  // 出力用変数

    nn::sf::InBuffer pIeBody(reinterpret_cast<const char*>(ieBody), ieBodySize);
    auto result = g_LocalManager->AddIe(&IeIndex, static_cast<uint32_t>(type), pIeBody);
    if( result.IsFailure() )
    {
        return result;
    }

    *pOutIeIndex = IeIndex;

    return result;
}

nn::Result DeleteIe(uint32_t IeIndex) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(g_LocalManager);
    auto result = g_LocalManager->DeleteIe(IeIndex);
    return result;
}

nn::Result GetMacAddress(MacAddress* pOutMac) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(g_LocalManager);
    if( pOutMac == NULL )
    {
        NN_SDK_REQUIRES(false, "%s: pOutMac must not be NULL\n", __FUNCTION__);
        return nn::wlan::ResultInvalidArgument();
    }

    nn::wlan::detail::SfdlMacAddress pMac;
    auto result = g_LocalManager->GetMacAddress(nn::sf::Out<nn::wlan::detail::SfdlMacAddress>(&pMac));
    if( result.IsFailure() )
    {
        return result;
    }

    pOutMac->Set(pMac.data);

    return result;
}

nn::Result GetRssi(int32_t* pRssi) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(g_LocalManager);
    if( pRssi == NULL )
    {
        NN_SDK_REQUIRES(false, "%s: pRssi must not be null\n", __FUNCTION__);
        return nn::wlan::ResultInvalidArgument();
    }

    int32_t rssi;  // 出力用変数
    auto result = g_LocalManager->GetRssi(&rssi);
    *pRssi = rssi;

    return result;
}

nn::Result GetLinkLevel(LinkLevel* pLinkLevel) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(g_LocalManager);
    if( pLinkLevel == NULL )
    {
        NN_SDK_REQUIRES(false, "%s: pLinkLevel must not be null\n", __FUNCTION__);
        return nn::wlan::ResultInvalidArgument();
    }

    int32_t rssi;
    auto result = GetRssi(&rssi);
    if( result.IsSuccess() )
    {
        *pLinkLevel = SignalStrength(rssi).GetLinkLevelLocal();
    }
    else
    {
        *pLinkLevel = LinkLevel_0;
    }

    return result;
}

LinkLevel ConvertRssiToLinkLevel(int32_t rssi) NN_NOEXCEPT
{
    return SignalStrength(rssi).GetLinkLevelLocal();
}


nn::Result PutFrameRaw(const uint8_t pInput[], size_t size ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(g_LocalManager);
    if( pInput == NULL )
    {
        NN_SDK_REQUIRES(false, "%s: pInput must not be null\n", __FUNCTION__);
        return nn::wlan::ResultInvalidArgument();
    }
    if( size >= 2048 )
    {
        NN_SDK_REQUIRES(false, "%s: PutFrame size(%d) must be less than 2048 bytes\n", __FUNCTION__, size);
        return nn::wlan::ResultInvalidArgument();
    }

    nn::sf::InBuffer pInBuffer(reinterpret_cast<const char*>(pInput), size);
    auto result = g_LocalManager->PutFrameRaw(pInBuffer);

    return result;
}


nn::Result GetFrameRaw(uint8_t pOutBuf[], size_t size, size_t* pOutSize, uint32_t rxId) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(g_LocalGetFrameManager);
    if( pOutBuf == NULL )
    {
        NN_SDK_REQUIRES(false, "%s: pOutBuf must not be null\n", __FUNCTION__);
        return nn::wlan::ResultInvalidArgument();
    }
    if( pOutSize == NULL )
    {
        NN_SDK_REQUIRES(false, "%s: pOutSize must not be null\n", __FUNCTION__);
        return nn::wlan::ResultInvalidArgument();
    }

    nn::sf::OutBuffer pOutBuffer(reinterpret_cast<char*>(pOutBuf), size);
    uint32_t outSize = 0;  // 受信データサイズが32bit以上になることはないので、uint32_tで十分
    auto result = g_LocalGetFrameManager->GetFrameRaw(pOutBuffer, &outSize, rxId);
    *pOutSize = static_cast<size_t>(outSize);

    return result;
}

nn::Result CancelGetFrame(uint32_t rxId) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(g_LocalManager);
    auto result = g_LocalManager->CancelGetFrame(rxId);
    return result;
}

nn::Result CreateRxEntry(uint32_t* pOutRxId, const uint16_t pEthertypes[], uint32_t count, uint32_t capacity) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(g_LocalManager);
    if( pOutRxId == NULL )
    {
        NN_SDK_REQUIRES(false, "%s: pOutRxId must not be null\n", __FUNCTION__);
        return nn::wlan::ResultInvalidArgument();
    }
    if( pEthertypes == NULL )
    {
        NN_SDK_REQUIRES(false, "%s: pEthertypes must not be null\n", __FUNCTION__);
        return nn::wlan::ResultInvalidArgument();
    }
    if( count > EthertypeCountMax )
    {
        NN_SDK_REQUIRES(false, "%s: The number of ethertypes must be less than %d\n", __FUNCTION__, EthertypeCountMax + 1);
        return nn::wlan::ResultInvalidArgument();
    }
    if( capacity > DataFrameRxCapacityMax )
    {
        NN_SDK_REQUIRES(false, "%s: Capacity of RxEntry must be less than %d\n", __FUNCTION__, DataFrameRxCapacityMax + 1);
        return nn::wlan::ResultInvalidArgument();
    }

    nn::sf::InArray<uint16_t> InArray(pEthertypes, count);
    uint32_t rxId = 0;
    auto result = g_LocalManager->CreateRxEntry(&rxId, InArray, capacity);
    if( result.IsSuccess() )
    {
        *pOutRxId = rxId;
    }

    return result;
}

nn::Result DeleteRxEntry(uint32_t rxId) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(g_LocalManager);

    auto result = g_LocalManager->DeleteRxEntry(rxId);

    return result;
}


nn::Result AddEthertypeToRxEntry(uint32_t rxId, uint16_t ethertype) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(g_LocalManager);

    auto result = g_LocalManager->AddEthertypeToRxEntry(rxId, ethertype);

    return result;
}


nn::Result DeleteEthertypeFromRxEntry(uint32_t* pOutRxId, uint16_t ethertype) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(g_LocalManager);
    if( pOutRxId == NULL )
    {
        NN_SDK_REQUIRES(false, "%s: pOutRxId must not be NULL\n", __FUNCTION__);
        return nn::wlan::ResultInvalidArgument();
    }

    uint32_t rxId = 0;
    auto result = g_LocalManager->DeleteEthertypeFromRxEntry(&rxId, ethertype);
    if( result.IsSuccess() )
    {
        *pOutRxId = rxId;
    }

    return result;
}


nn::Result AddMatchingDataToRxEntry(uint32_t rxId, const ReceivedDataMatchInfo& pMatchInfo) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(g_LocalManager);
    if( pMatchInfo.matchLength > ReceivedDataMatchLengthMax )
    {
        NN_SDK_REQUIRES(false, "%s: ReceiveDataMatch length must be less than %d\n", __FUNCTION__, ReceivedDataMatchLengthMax + 1);
        return nn::wlan::ResultInvalidArgument();
    }

    auto result = g_LocalManager->AddMatchingDataToRxEntry(rxId, pMatchInfo);

    return result;
}


nn::Result RemoveMatchingDataFromRxEntry(uint32_t rxId, const ReceivedDataMatchInfo& pMatchInfo) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(g_LocalManager);
    if( pMatchInfo.matchLength > ReceivedDataMatchLengthMax )
    {
        NN_SDK_REQUIRES(false, "%s: ReceiveDataMatch length must be less than %d\n", __FUNCTION__, ReceivedDataMatchLengthMax + 1);
        return nn::wlan::ResultInvalidArgument();
    }

    auto result = g_LocalManager->RemoveMatchingDataFromRxEntry(rxId, pMatchInfo);

    return result;
}

nn::Result PutActionFrameOneShot(const MacAddress& dstMac, const uint8_t* pData, size_t size, int16_t channel, uint32_t dwellTime) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(g_LocalManager);
    if( dstMac == MacAddress::CreateZeroMacAddress() )
    {
        NN_SDK_REQUIRES(false, "%s: dstMac must not be Zero mac address\n", __FUNCTION__);
        return nn::wlan::ResultInvalidArgument();
    }
    if( pData == NULL )
    {
        NN_SDK_REQUIRES(false, "%s: pData must not be null\n", __FUNCTION__);
        return nn::wlan::ResultInvalidArgument();
    }
    if( size < 5 || size > OneShotActionFrameSizeMax )
    {
        NN_SDK_REQUIRES(false, "%s: size is out of range\n", __FUNCTION__);
        return nn::wlan::ResultInvalidArgument();
    }
    if( channel < WirelessChannel_1ch || channel > WirelessChannel_165ch )
    {
        NN_SDK_REQUIRES(false, "%s: channel is out of range\n", __FUNCTION__);
        return nn::wlan::ResultInvalidArgument();
    }
    if( dwellTime > 300 )
    {
        NN_SDK_REQUIRES(false, "%s: dwellTime must be less than 301\n", __FUNCTION__);
        return nn::wlan::ResultInvalidArgument();
    }

    // データの正当性評価
    if( pData[0] != 0x7F )
    {
        NN_SDK_REQUIRES(false, "%s: pData[0] must be 0x7F\n", __FUNCTION__);
        return nn::wlan::ResultInvalidArgument();
    }
    if( pData[4] < ActionFrameType_Beacon || pData[4] >= ActionFrameType_End )
    {
        NN_SDK_REQUIRES(false, "%s: pData[4] is out of range\n", __FUNCTION__);
        return nn::wlan::ResultInvalidArgument();
    }

    nn::sf::InBuffer pInBuffer(reinterpret_cast<const char*>(pData), size);
    nn::wlan::detail::SfdlMacAddress mac;
    std::memcpy(&mac.data[0], dstMac.GetMacAddressData(), MacAddress::MacAddressSize);

    auto result = g_LocalManager->PutActionFrameOneShot(mac, pInBuffer, static_cast<uint32_t>(channel), dwellTime);
    return result;
}

nn::Result SetActionFrameWithBeacon(const uint8_t* pData, size_t size) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(g_LocalManager);
    if( pData == NULL )
    {
        NN_SDK_REQUIRES(false, "%s: pData must not be null\n", __FUNCTION__);
        return nn::wlan::ResultInvalidArgument();
    }
    if( size < 5 || size > BeaconActionFrameSizeMax )
    {
        NN_SDK_REQUIRES(false, "%s: size is out of range\n", __FUNCTION__);
        return nn::wlan::ResultInvalidArgument();
    }

    // データの正当性評価
    if( pData[0] != 0x7F )
    {
        NN_SDK_REQUIRES(false, "%s: pData[0] must be 0x7F\n", __FUNCTION__);
        return nn::wlan::ResultInvalidArgument();
    }
    if( pData[4] != ActionFrameType_Beacon )
    {
        NN_SDK_REQUIRES(false, "%s: pData[4] must be ActionFrameType_Beacon\n", __FUNCTION__);
        return nn::wlan::ResultInvalidArgument();
    }

    nn::sf::InBuffer pInBuffer(reinterpret_cast<const char*>(pData), size);

    auto result = g_LocalManager->SetActionFrameWithBeacon(pInBuffer);
    return result;
}

nn::Result CancelActionFrameWithBeacon() NN_NOEXCEPT
{
    NN_SDK_REQUIRES(g_LocalManager);
    auto result = g_LocalManager->CancelActionFrameWithBeacon();
    return result;
}

nn::Result GetActionFrameCore(MacAddress* pOutSrcMac, uint8_t pOutBuf[], size_t size,
        size_t* pOutSize, uint32_t rxId, uint16_t* pOutChannel, int16_t* pOutRssi, bool isEx) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(g_LocalGetActionFrameManager);
    if( pOutSrcMac == NULL )
    {
        NN_SDK_REQUIRES(false, "%s: pOutSrcMac must not be null\n", __FUNCTION__);
        return nn::wlan::ResultInvalidArgument();
    }
    if( pOutBuf == NULL )
    {
        NN_SDK_REQUIRES(false, "%s: pOutBuf must not be null\n", __FUNCTION__);
        return nn::wlan::ResultInvalidArgument();
    }
    if( pOutSize == NULL )
    {
        NN_SDK_REQUIRES(false, "%s: pOutSize must not be null\n", __FUNCTION__);
        return nn::wlan::ResultInvalidArgument();
    }
    if( isEx == true )
    {
        if( pOutChannel == NULL )
        {
            NN_SDK_REQUIRES(false, "%s: pOutChannel must not be null\n", __FUNCTION__);
            return nn::wlan::ResultInvalidArgument();
        }
        if( pOutRssi == NULL )
        {
            NN_SDK_REQUIRES(false, "%s: pOutRssi must not be null\n", __FUNCTION__);
            return nn::wlan::ResultInvalidArgument();
        }
    }

    nn::sf::OutBuffer pOutBuffer(reinterpret_cast<char*>(pOutBuf), size);
    uint32_t outSize = 0;  // 受信データサイズが32bit以上になることはないので、uint32_tで十分
    nn::wlan::detail::SfdlMacAddress mac;  // 出力用
    uint16_t outChannel;
    int16_t outRssi;
    auto result = g_LocalGetActionFrameManager->GetActionFrameCore(&mac, pOutBuffer, &outSize, rxId, &outChannel, &outRssi, isEx);

    pOutSrcMac->Set(mac.data);
    *pOutSize = static_cast<size_t>(outSize);
    if( isEx == true )
    {
        *pOutChannel = outChannel;
        *pOutRssi = outRssi;
    }

    return result;
}

nn::Result GetActionFrame(MacAddress* pOutSrcMac, uint8_t pOutBuf[], size_t size, size_t* pOutSize, uint32_t rxId) NN_NOEXCEPT
{
    return GetActionFrameCore(pOutSrcMac, pOutBuf, size, pOutSize, rxId, NULL, NULL, false);
}

nn::Result GetActionFrameEx(MacAddress* pOutSrcMac, uint8_t pOutBuf[], size_t size,
        size_t* pOutSize, uint32_t rxId, uint16_t* pOutChannel, int16_t* pOutRssi) NN_NOEXCEPT
{
    return GetActionFrameCore(pOutSrcMac, pOutBuf, size, pOutSize, rxId, pOutChannel, pOutRssi, true);
}

nn::Result CancelGetActionFrame(uint32_t rxId) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(g_LocalManager);
    auto result = g_LocalManager->CancelGetActionFrame(rxId);
    return result;
}

nn::Result CreateRxEntryForActionFrame(uint32_t* pOutRxId, const uint16_t pAfTypes[], uint32_t count, uint32_t capacity) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(g_LocalManager);
    if( pOutRxId == NULL )
    {
        NN_SDK_REQUIRES(false, "%s: pOutRxId must not be null\n", __FUNCTION__);
        return nn::wlan::ResultInvalidArgument();
    }
    if( pAfTypes == NULL )
    {
        NN_SDK_REQUIRES(false, "%s: pAfTypes must not be null\n", __FUNCTION__);
        return nn::wlan::ResultInvalidArgument();
    }
    if( count > ActionFrameTypeCountMax )
    {
        NN_SDK_REQUIRES(false, "%s: The number of ActionFrameType must be less than %d\n", __FUNCTION__, ActionFrameTypeCountMax + 1);
        return nn::wlan::ResultInvalidArgument();
    }
    if( capacity > ActionFrameRxCapacityMax )
    {
        NN_SDK_REQUIRES(false, "%s: Capacity of RxEntry must be less than %d\n", __FUNCTION__, ActionFrameRxCapacityMax + 1);
        return nn::wlan::ResultInvalidArgument();
    }
    for( int i = 0; i < count; i++ )
    {
        if( pAfTypes[i] < ActionFrameType_Beacon || pAfTypes[i] >= ActionFrameType_End )
        {
            NN_SDK_REQUIRES(false, "%s: Action frame type is out of range\n", __FUNCTION__);
            return nn::wlan::ResultInvalidArgument();
        }
    }

    nn::sf::InArray<uint16_t> InArray(pAfTypes, count);
    uint32_t rxId = 0;
    auto result = g_LocalManager->CreateRxEntryForActionFrame(&rxId, InArray, capacity);
    if( result.IsSuccess() )
    {
        *pOutRxId = rxId;
    }

    return result;
}

nn::Result DeleteRxEntryForActionFrame(uint32_t rxId) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(g_LocalManager);

    auto result = g_LocalManager->DeleteRxEntryForActionFrame(rxId);

    return result;
}

nn::Result AddSubtypeToRxEntryForActionFrame(uint32_t rxId, ActionFrameType type) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(g_LocalManager);
    if( type < ActionFrameType_Beacon || type >= ActionFrameType_End )
    {
        NN_SDK_REQUIRES(false, "%s: Action frame type is out of range\n", __FUNCTION__);
        return nn::wlan::ResultInvalidArgument();
    }

    auto result = g_LocalManager->AddSubtypeToRxEntryForActionFrame(rxId, static_cast<uint32_t>(type));

    return result;
}

nn::Result DeleteSubtypeFromRxEntryForActionFrame(uint32_t* pOutRxId, ActionFrameType type) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(g_LocalManager);
    if( pOutRxId == NULL )
    {
        NN_SDK_REQUIRES(false, "%s: pOutRxId must not be null\n", __FUNCTION__);
        return nn::wlan::ResultInvalidArgument();
    }
    if( type < ActionFrameType_Beacon || type >= ActionFrameType_End )
    {
        NN_SDK_REQUIRES(false, "%s: Action frame type is out of range\n", __FUNCTION__);
        return nn::wlan::ResultInvalidArgument();
    }

    uint32_t rxId = 0;
    auto result = g_LocalManager->DeleteSubtypeFromRxEntryForActionFrame(&rxId, static_cast<uint32_t>(type));
    if( result.IsSuccess() )
    {
        *pOutRxId = rxId;
    }

    return result;
}

nn::Result SetMaxAssociationNumber(uint32_t maxNum) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(g_LocalManager);
    if( maxNum < 1 || maxNum > ConnectableClientsCountMax )
    {
        NN_SDK_REQUIRES(false, "%s: Max association number must be in 1 ~ %d", __FUNCTION__, ConnectableClientsCountMax);
        return ResultInvalidArgument();
    }
    auto result = g_LocalManager->SetMaxAssociationNumber(maxNum);
    return result;
}

nn::Result GetChannelStats(ChannelStats (&outStats)[WirelessChannelsCountMax + 1], uint32_t* pOutCount, const ScanParameters& param) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(g_LocalManager);
    // パラメータチェック
    if( !(param.scanType == nn::wlan::ScanType_Active || param.scanType == nn::wlan::ScanType_Passive) )
    {
        NN_SDK_REQUIRES(false, "scanType is invalid");
        return nn::wlan::ResultInvalidArgument();
    }
    if( !(param.channelScanTime == -1 || (param.channelScanTime >= 10 && param.channelScanTime <= 3000)) )
    {
        NN_SDK_REQUIRES(false, "channelScanTime is invalid");
        return nn::wlan::ResultInvalidArgument();
    }
    if( !(param.homeChannelTime == -1 || (param.homeChannelTime >= 0 && param.homeChannelTime <= 500)) )
    {
        NN_SDK_REQUIRES(false, "homeChannelTime is invalid");
        return nn::wlan::ResultInvalidArgument();
    }
    if( param.bssid == nn::wlan::MacAddress::CreateZeroMacAddress() )
    {
        NN_SDK_REQUIRES(false, "bssid must not be ZeroMacAddress");
        return nn::wlan::ResultInvalidArgument();
    }
    if( pOutCount == NULL )
    {
        NN_SDK_REQUIRES(false, "pOutCount must not be null");
        return nn::wlan::ResultInvalidArgument();
    }

    nn::Result result;

    // IPC用構造体へ値をコピー
    nn::wlan::detail::SfdlScanParameters scanParams;
    scanParams.scanType = param.scanType;
    scanParams.channelCount = param.channelCount;
    for(int i = 0; i < param.channelCount; i++)
    {
        // チャンネルチェック
        if( param.channelList[i] < WirelessChannel_1ch || param.channelList[i] > WirelessChannel_165ch )
        {
            NN_SDK_REQUIRES(false, "channel is out of range");
            return nn::wlan::ResultInvalidArgument();
        }
        scanParams.channelList.channel[i] = param.channelList[i];
    }
    scanParams.channelScanTime = param.channelScanTime;
    scanParams.homeChannelTime = param.homeChannelTime;
    scanParams.ssidCount = param.ssidCount;
    // scan対象SSIDリストのコピー判断
    if( param.ssidCount > 0 && param.ssidList != NULL )
    {
        if( param.ssidCount > nn::wlan::ScanningSsidCountMax )
        {
            NN_SDK_LOG("[WLAN][WARN]Scanning SSID count should not be over %d.\n", nn::wlan::ScanningSsidCountMax);
            scanParams.ssidCount = nn::wlan::ScanningSsidCountMax;
        }
        for(int i = 0; i < scanParams.ssidCount; i++)
        {
            if( param.ssidList[i].GetLength() == 0 )
            {
                NN_SDK_LOG("[WLAN][WARN]The length of a filtering SSID must not be 0.\n");
                NN_SDK_ASSERT(false);
                return nn::wlan::ResultInvalidArgument();
            }
            scanParams.ssidList[i].length = param.ssidList[i].GetLength();
            std::memcpy(&scanParams.ssidList[i].data[0], param.ssidList[i].GetSsidData(), param.ssidList[i].GetLength());
        }
    }
    else
    {
        scanParams.ssidCount = 0;  // scanParams.ssidListはNULLではないので、ssidCountを0にしておくことでIPCサーバー側でSSIDフィルタ無しと判断する。
    }

    for(uint8_t i = 0; i < nn::wlan::MacAddress::MacAddressSize; i++)
    {
        scanParams.bssid.data[i] = param.bssid.GetMacAddressData()[i];
    }

    // Scan完了通知を受け取るためのSystemEventを設定
    nn::sf::NativeHandle sfHandle;  // 出力用変数
    result = g_LocalManager->GetSystemEvent(&sfHandle, static_cast<uint32_t>(detail::SystemEventType_ScanComplete));
    if( result.IsFailure() )
    {
        return result;
    }

    nn::os::SystemEvent scanCompEvent;
    scanCompEvent.AttachReadableHandle(sfHandle.GetOsHandle(), sfHandle.IsManaged(), nn::os::EventClearMode_AutoClear);
    sfHandle.Detach();

    result = g_LocalManager->StartScan(scanParams);
    if( result.IsFailure() )
    {
        return result;
    }

    // Scan完了待機
    scanCompEvent.Wait();

    // 測定結果を取得
    *pOutCount = 0;
    return g_LocalManager->GetChannelStats(nn::sf::OutArray<ChannelStats>(outStats, WirelessChannelsCountMax + 1), pOutCount);

} //NOLINT(impl/function_size)

}
}
}

