﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "visrv_VicOperation.h"

#include <nn/result/result_HandlingUtility.h>
#include <nn/vi/vi_Result.h>

namespace nn{ namespace visrv{ namespace vic{

    nn::Result VicOperation::CopySingle(
        NvRmFence* pOutFence,
        VicCopySingleDestinationInfo& dst,
        VicModule* pModule,
        const VicCopySingleSourceInfo& src,
        const VicCopyOption& option
    ) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_NOT_NULL(pOutFence);
        NN_SDK_REQUIRES_NOT_NULL(pModule);
        NN_SDK_REQUIRES(pModule->IsInitialized());

        const NvU32 SrcPlaneCount = 1;
        const NvU32 DstPlaneCount = 1;
        const NvU32 InputSlotIndex = 0;

        NvDdkVicConfigParameters config = {};
        {
            auto bgc = option.backgroundColor;
            memset(&config, 0, sizeof(config));
            config.BackgroundColor = NvDdkVicFloatColorRec{bgc.GetR(), bgc.GetG(), bgc.GetB(), bgc.GetA()};
            config.BackgroundColorSpace = NvDdkVicColorSpace_sRGB;
            switch(option.alphaMode)
            {
            case VicCopyAlphaMode_Source:
                config.AlphaFillMode = NvDdkVicAlphaFillMode_SourceAlpha;
                break;
            default:
                config.AlphaFillMode = NvDdkVicAlphaFillMode_Opaque;
                break;
            }
        }

        NvRectF32 sRect = {};
        sRect.left   = src.region.x;
        sRect.top    = src.region.y;
        sRect.right  = src.region.x + src.region.width;
        sRect.bottom = src.region.y + src.region.height;

        NvRect dRect = {};
        dRect.left   = dst.region.x;
        dRect.top    = dst.region.y;
        dRect.right  = dst.region.x + dst.region.width;
        dRect.bottom = dst.region.y + dst.region.height;

        // setup orientation
        if((src.transform & nn::vi::ImageTransform_YFlip) != (dst.transform & nn::vi::ImageTransform_YFlip))
        {
            config.OutputFlipY = true;
        }

        // setup up/downsample method
        switch(option.filter)
        {
        case VicCopyFilter_Nearest:
            {
                config.Slot[InputSlotIndex].FilterX = NvDdkVicFilter_Nearest;
                config.Slot[InputSlotIndex].FilterY = NvDdkVicFilter_Nearest;
                break;
            }
        case VicCopyFilter_Bilinear:
            {
                config.Slot[InputSlotIndex].FilterX = NvDdkVicFilter_Bilinear;
                config.Slot[InputSlotIndex].FilterY = NvDdkVicFilter_Bilinear;
                break;
            }
        case VicCopyFilter_Nicest:
            {
                config.Slot[InputSlotIndex].FilterX = NvDdkVicFilter_Nicest;
                config.Slot[InputSlotIndex].FilterY = NvDdkVicFilter_Nicest;
                break;
            }
        default:
            ;
        }

        // setup blend parameter
        {
            auto& blend = config.Slot[InputSlotIndex].Blend;
            blend.K1 = 1;
            blend.SrcFactC = NvDdkVicBlendSrcFactC_K1;
            blend.DstFactC = NvDdkVicBlendDstFactC_Zero;
            blend.SrcFactA = NvDdkVicBlendSrcFactA_K1;
            blend.DstFactA = NvDdkVicBlendDstFactA_Zero;
        }

        NvDdkVicConfigureBlending(
            pModule->GetSession(),
            &config,
            InputSlotIndex,
            NvDdkVicPerPixelAlpha_Premultiplied,
            1.0f
        );

        NN_RESULT_THROW_UNLESS(
            NvDdkVicConfigureTargetSurface(
                pModule->GetSession(),
                &config,
                dst.pSurface,
                DstPlaneCount,
                &dRect
            ) == NvSuccess,
            nn::vi::ResultOperationFailed()
        );

        NN_RESULT_THROW_UNLESS(
            NvDdkVicConfigureSourceSurface(
                pModule->GetSession(),
                &config,
                InputSlotIndex,
                const_cast<NvRmSurface*>(src.pSurface),
                SrcPlaneCount,
                &sRect,
                &dRect
            ) == NvSuccess,
            nn::vi::ResultOperationFailed()
        );

        NN_RESULT_THROW_UNLESS(
            NvDdkVicConfigure(pModule->GetSession(), &config) == NvSuccess,
            nn::vi::ResultOperationFailed()
        );

        NvDdkVicExecParameters params;
        {
            memset(&params, 0, sizeof(params));
            params.OutputSurface = dst.pSurface;
            params.InputSurfaces[InputSlotIndex][0] = const_cast<NvRmSurface*>(src.pSurface);
        }

        NvRmFence fence;
        NN_RESULT_THROW_UNLESS(
            NvDdkVicExecute(pModule->GetSession(), &params, nullptr, 0, &fence) == NvSuccess,
            nn::vi::ResultOperationFailed()
        );

        *pOutFence = fence;
        NN_RESULT_SUCCESS;
    }// NOLINT(impl/function_size)

}}}
