﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "visrv_TransferMemoryBinder.h"

#include <nn/result/result_HandlingUtility.h>
#include <nn/util/util_ScopeExit.h>
#include <nn/os/os_TransferMemoryApi.h>
#include <nn/sf/sf_NativeHandle.h>
#include "../visrv_Macro.h"

namespace nn{ namespace visrv{ namespace util{

    TransferMemoryBinder::TransferMemoryBinder() NN_NOEXCEPT
        : m_TransferMemory()
        , m_pBoundMemory()
        , m_BoundMemorySize()
    {
    }

    bool TransferMemoryBinder::IsBound() const NN_NOEXCEPT
    {
        return m_pBoundMemory != nullptr;
    }

    void* TransferMemoryBinder::GetBoundMemory() NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(IsBound());
        return m_pBoundMemory;
    }

    const void* TransferMemoryBinder::GetBoundMemory() const NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(IsBound());
        return m_pBoundMemory;
    }

    size_t TransferMemoryBinder::GetBoundMemorySize() const NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(IsBound());
        return m_BoundMemorySize;
    }

    nn::Result TransferMemoryBinder::Bind(nn::sf::NativeHandle& handle, size_t size, nn::os::MemoryPermission ownerPermission) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(!IsBound());

        NN_VISRV_PROCESS_START();

        nn::os::AttachTransferMemory(&m_TransferMemory, size, handle.GetOsHandle(), handle.IsManaged());
        handle.Detach();
        NN_VISRV_PROCESS_ROLLBACK(nn::os::DestroyTransferMemory(&m_TransferMemory));

        NN_RESULT_DO(nn::os::MapTransferMemory(&m_pBoundMemory, &m_TransferMemory, ownerPermission));

        NN_VISRV_PROCESS_SUCCESS();
        m_BoundMemorySize = size;
        NN_SDK_ASSERT(IsBound());
        NN_RESULT_SUCCESS;
    }

    void TransferMemoryBinder::Unbind() NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(IsBound());

        nn::os::DestroyTransferMemory(&m_TransferMemory);
        m_pBoundMemory = nullptr;
        m_BoundMemorySize = 0;
        NN_SDK_ASSERT(!IsBound());
    }

}}}

