﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/nn_Result.h>
#include <nn/nn_SdkAssert.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/util/util_ScopeExit.h>
#include <nn/os.h>

#include <nn/vi/vi_Result.h>

namespace nn{ namespace visrv{ namespace util{

    template<typename T, int TSize>
    class TResourcePool
    {
    public:
        static const int Size = TSize;

        enum State
        {
            State_Free,
            State_Used,
        };

        typedef T value_type;

    public:
        TResourcePool() NN_NOEXCEPT
            : m_Mutex()
        {
            std::memset(m_StateList, 0, sizeof(m_StateList));
        }

        void Initialize() NN_NOEXCEPT
        {
            nn::os::InitializeMutex(&m_Mutex, false, 0);
            std::memset(m_StateList, 0, sizeof(m_StateList));
        }

        void Finalize() NN_NOEXCEPT
        {
            nn::os::FinalizeMutex(&m_Mutex);
        }

        nn::Result Allocate(T** pOutValue) NN_NOEXCEPT
        {
            nn::os::LockMutex(&m_Mutex);
            NN_UTIL_SCOPE_EXIT{ nn::os::UnlockMutex(&m_Mutex); };

            for(int i = 0; i < Size; i++)
            {
                if(m_StateList[i] == State_Free)
                {
                    m_StateList[i] = State_Used;
                    *pOutValue = &m_List[i];
                    NN_RESULT_SUCCESS;
                }
            }
            NN_RESULT_THROW(nn::vi::ResultResourceLimit());
        }

        void Free(T* p) NN_NOEXCEPT
        {
            ptrdiff_t index = GetIndex(p);
            NN_ABORT_UNLESS_RANGE(index, 0, static_cast<int>(Size));
            NN_ABORT_UNLESS_EQUAL(p, &m_List[index]);

            nn::os::LockMutex(&m_Mutex);
            NN_UTIL_SCOPE_EXIT{ nn::os::UnlockMutex(&m_Mutex); };

            NN_SDK_ASSERT_EQUAL(m_StateList[index], State_Used);
            m_StateList[index] = State_Free;
        }

        int GetIndex(T* p) const NN_NOEXCEPT
        {
            ptrdiff_t index = p - &m_List[0];
            NN_SDK_REQUIRES_RANGE(index, static_cast<ptrdiff_t>(0), static_cast<ptrdiff_t>(Size));
            NN_SDK_REQUIRES_EQUAL(p, &m_List[index]);
            return static_cast<int>(index);
        }

        // 指定したインデックスのオブジェクトのポインタを返す。
        // 未初期化の場合 nullptr を返す
        T* GetAt(int index) NN_NOEXCEPT
        {
            NN_SDK_REQUIRES_RANGE(index, 0, static_cast<int>(Size));
            if(m_StateList[index] == State_Free)
            {
                return nullptr;
            }
            return &m_List[index];
        }

        T* GetHead() NN_NOEXCEPT
        {
            nn::os::LockMutex(&m_Mutex);
            NN_UTIL_SCOPE_EXIT{ nn::os::UnlockMutex(&m_Mutex); };

            for(int i = 0; i < Size; i++)
            {
                if(m_StateList[i] != State_Free)
                {
                    return &m_List[i];
                }
            }
            return nullptr;
        }

        int GetCount() const NN_NOEXCEPT
        {
            nn::os::LockMutex(&m_Mutex);
            NN_UTIL_SCOPE_EXIT{ nn::os::UnlockMutex(&m_Mutex); };

            int count = 0;
            for(int i = 0; i < Size; i++)
            {
                if(m_StateList[i] != State_Free)
                {
                    count++;
                }
            }
            return count;
        }

    protected:
        // bool (*F)(const T& elem)
        template<typename F>
        nn::Result FindImpl(T** pOutValue, F cond) NN_NOEXCEPT
        {
            nn::os::LockMutex(&m_Mutex);
            NN_UTIL_SCOPE_EXIT{ nn::os::UnlockMutex(&m_Mutex); };

            for(int i = 0; i < Size; i++)
            {
                if(m_StateList[i] == State_Used)
                {
                    const T& e = m_List[i];
                    if(cond(e))
                    {
                        *pOutValue = &m_List[i];
                        NN_RESULT_SUCCESS;
                    }
                }
            }
            NN_RESULT_THROW(nn::vi::ResultNotFound());
        }

    private:
        nn::os::MutexType m_Mutex;
        State m_StateList[Size];
        T m_List[Size];
    };

}}}
