﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <new>
#include <cstdio>
#include <nn/nn_Abort.h>
#include <nn/util/util_TypedStorage.h>
#include "../visrv_Log.h"
#include "visrv_Settings.h"
#include "visrv_PlatformConfig.h"
#include "detail/visrv_Read.h"
#include "visrv_Strategy.h"

namespace
{
    nn::util::TypedStorage<nn::visrv::settings::PlatformConfig,
                           sizeof(nn::visrv::settings::PlatformConfig),
                           alignof(nn::visrv::settings::PlatformConfig)> g_PlatformConfig;
}

namespace nn { namespace visrv { namespace settings {

    void Initialize() NN_NOEXCEPT
    {
        PlatformConfig* config = new (&Get(g_PlatformConfig)) PlatformConfig();

        if( detail::ReadItem(&config->physicalDisplayCount, "physical_display_count") != sizeof(config->physicalDisplayCount) )
        {
            NN_VISRV_LOG_WARN("Missing key 'physical_display_count'.  Only Default and Null displays will be available.\n");
            config->physicalDisplayCount = 0;
        }

        {
            int writeIndex = 0;

            for( int i = 0; i < config->physicalDisplayCount && i < sizeof(config->displays) / sizeof(config->displays[0]); ++i )
            {
                if( config->displays[i].Read(i) )
                {
                    ++writeIndex;
                }
                else
                {
                    // found incomplete physical display listing
                    // ignore entry
                    NN_VISRV_LOG_WARN("Found incomplete physical display entry (ID = %d).  The entry will be ignored.\n", i);
                }
            }

            config->physicalDisplayCount = writeIndex;
        }

        std::underlying_type<Strategy>::type strategy;
        if( detail::ReadItem(&strategy, "default_display_dc_strategy") != sizeof(strategy) )
        {
            NN_VISRV_LOG_WARN("Missing key 'default_display_dc_strategy'.  Only the highest priority DC will be used.\n");
            strategy = static_cast<decltype(strategy)>(nn::visrv::settings::Strategy::HighestPriority);
        }

        config->defaultDisplayStrategy = ConvertStrategy(strategy);

        for( int i = 0; i < sizeof(config->dcList) / sizeof(config->dcList[0]); ++i )
        {
            config->dcList[i].Read(i);
        }
    }

    void Finalize() NN_NOEXCEPT
    {
        Get(g_PlatformConfig).~PlatformConfig();
    }

    const PlatformConfig* GetPlatformConfig() NN_NOEXCEPT
    {
        return &Get(g_PlatformConfig);
    }

}}}
