﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstdio>
#include <cstring>
#include "visrv_DisplayController.h"
#include "visrv_InterfaceType.h"
#include "detail/visrv_Read.h"
#include "../visrv_Log.h"

namespace
{
    const int InvalidPriority = -1;

    bool IsModeValid(const nn::vi::DisplayModeInfoInternal& mode) NN_NOEXCEPT
    {
        return mode.clientInfo.width > 0 &&
               mode.clientInfo.height > 0 &&
               mode.clientInfo.refreshRate > 0.f &&
               mode.clientInfo.mode >= nn::vi::StereoMode_None &&
               mode.clientInfo.mode <= nn::vi::StereoMode_TopAndBottom &&
               mode.ratio >= nn::vi::AspectRatio_4_3 &&
               mode.ratio <= nn::vi::AspectRatio_256_135;
    }
}

namespace nn { namespace visrv { namespace settings {

    DisplayController::DisplayController() NN_NOEXCEPT
        : priority(InvalidPriority)
        , interface(InterfaceType::Null)
        , modeCount(0)
    {
        std::memset(this->modes, 0, sizeof(this->modes));
    }

    void DisplayController::Read(int id) NN_NOEXCEPT
    {
        char keyBuffer[128];
        std::snprintf(keyBuffer, sizeof(keyBuffer), "default_display_dc_%d_priority", id);
        if( detail::ReadItem(&this->priority, keyBuffer) != sizeof(this->priority) )
        {
            NN_VISRV_LOG_WARN("No priority assigned for DC (id = %d).  This DC will not be used with the default display.\n", id);
            this->priority = InvalidPriority;
        }

        {
            std::underlying_type<InterfaceType>::type temp;
            std::snprintf(keyBuffer, sizeof(keyBuffer), "dc_%d_interface", id);
            if( detail::ReadItem(&temp, keyBuffer) != sizeof(temp) )
            {
                NN_VISRV_LOG_WARN("No interface set for DC (id = %d).  Interface set to null!\n", id);
                temp = static_cast<std::underlying_type<InterfaceType>::type>(InterfaceType::Null);
            }

            this->interface = ConvertInterface(temp);
        }

        std::snprintf(keyBuffer, sizeof(keyBuffer), "dc_%d_mode_count", id);
        if( detail::ReadItem(&this->modeCount, keyBuffer) != sizeof(this->modeCount) )
        {
            NN_VISRV_LOG_WARN("No modes set for DC (id = %d).  Clients will be unable to change modes.\n", id);
            this->modeCount = 0;
        }

        if( modeCount > sizeof(this->modes) / sizeof(this->modes[0]) )
        {
            NN_VISRV_LOG_WARN("Settings request %d modes, but only the first %u will be read.", this->modeCount, sizeof(this->modes) / sizeof(this->modes[0]));
        }

        int writeIndex = 0;
        for( int i = 0; i < modeCount && writeIndex < sizeof(this->modes) / sizeof(this->modes[0]); ++i )
        {
            bool isValid = true;

            nn::vi::DisplayModeInfoInternal& mode = this->modes[writeIndex];
            std::snprintf(keyBuffer, sizeof(keyBuffer), "dc_%d_mode_%d_width", id, i);
            if( detail::ReadItem(&mode.clientInfo.width, keyBuffer) != sizeof(mode.clientInfo.width) )
            {
                NN_VISRV_LOG_WARN("Missing width for mode (id = %d) for DC (id = %d).  Skipping mode entry.\n", i, id);
                isValid = false;
            }

            std::snprintf(keyBuffer, sizeof(keyBuffer), "dc_%d_mode_%d_height", id, i);
            if( detail::ReadItem(&mode.clientInfo.height, keyBuffer) != sizeof(mode.clientInfo.height) )
            {
                NN_VISRV_LOG_WARN("Missing height for mode (id = %d) for DC (id = %d).  Skipping mode entry.\n", i, id);
                isValid = false;
            }

            std::snprintf(keyBuffer, sizeof(keyBuffer), "dc_%d_mode_%d_refresh_rate", id, i);
            int mHz;

            if( detail::ReadItem(&mHz, keyBuffer) != sizeof(mHz) )
            {
                NN_VISRV_LOG_WARN("Missing refresh rate for mode (id = %d) for DC (id = %d).  Skipping mode entry.\n", i, id);
                isValid = false;
            }

            mode.clientInfo.refreshRate = mHz / 1000.f;

            std::snprintf(keyBuffer, sizeof(keyBuffer), "dc_%d_mode_%d_stereo_mode", id, i);
            if( detail::ReadItem(&mode.clientInfo.mode, keyBuffer) != sizeof(mode.clientInfo.mode) )
            {
                NN_VISRV_LOG_WARN("Missing stereo mode for mode (id = %d) for DC (id = %d).  Defaulting to no stereoscopic 3D.\n", i, id);
                mode.clientInfo.mode = nn::vi::StereoMode_None;
            }

            std::snprintf(keyBuffer, sizeof(keyBuffer), "dc_%d_mode_%d_aspect_ratio", id, i);
            if( detail::ReadItem(&mode.ratio, keyBuffer) != sizeof(mode.ratio) )
            {
                NN_VISRV_LOG_WARN("Missing aspect ratio for mode (id = %d) for DC (id = %d).  Defaulting to 16:9.\n", i, id);
                mode.ratio = nn::vi::AspectRatio_16_9;
            }

            if( isValid && IsModeValid(mode) )
            {
                ++writeIndex;
            }
            else
            {
                NN_VISRV_LOG_WARN("Mode (id = %d) for DC (id = %d) has invalid settings.\n", i, id);
                mode.clientInfo.width = 0;
                mode.clientInfo.height = 0;
                mode.clientInfo.refreshRate = 0.f;
                mode.clientInfo.mode = nn::vi::StereoMode_None;
                mode.ratio = nn::vi::AspectRatio_Unknown;
            }
        }

        this->modeCount = writeIndex;
    }

}}}
