﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "visrv_ManagerDisplayServiceImpl.h"

#include <nn/nn_SdkAssert.h>
#include <nn/os.h>
#include <nn/util/util_ScopeExit.h>
#include <nn/vi/vi_Result.h>
#include <nn/result/result_HandlingUtility.h>
#include "../visrv_Config.h"
#include "../visrv_Log.h"
#include "../master/visrv_Lib.h"

namespace nn{ namespace visrv{ namespace service{

    ManagerDisplayServiceImpl::ManagerDisplayServiceImpl(client::ClientObjectSmartHolder* pClientHolder) NN_NOEXCEPT
        : m_ClientHolder(pClientHolder)
    {
        NN_VISRV_LOG_IPC("m:ctor\n");
        NN_VISRV_LOG_LIFETIME("ManagerDisplayServiceImpl ctor\n");
    }

    ManagerDisplayServiceImpl::~ManagerDisplayServiceImpl() NN_NOEXCEPT
    {
        NN_VISRV_LOG_IPC("m:dtor\n");
        NN_VISRV_LOG_LIFETIME("ManagerDisplayServiceImpl dtor\n");
    }

    nn::Result ManagerDisplayServiceImpl::AllocateProcessHeapBlock(nn::sf::Out<std::uint64_t> outBlockId, std::uint64_t size) NN_NOEXCEPT
    {
        NN_VISRV_TRACE_IPC_M(AllocateProcessHeapBlock);
        ResourceId blockId = 0;
        NN_RESULT_DO(m_ClientHolder->AllocateProcessHeapBlock(&blockId, size));
        outBlockId.Set(blockId);
        NN_RESULT_SUCCESS;
    }

    nn::Result ManagerDisplayServiceImpl::FreeProcessHeapBlock(std::uint64_t blockId) NN_NOEXCEPT
    {
        NN_VISRV_TRACE_IPC_M(FreeProcessHeapBlock);
        return m_ClientHolder->FreeProcessHeapBlock(blockId);
    }

    nn::Result ManagerDisplayServiceImpl::GetDisplayResolution(nn::sf::Out<std::int64_t> pOutWidth, nn::sf::Out<std::int64_t> pOutHeight, nn::vi::DisplayId displayId) NN_NOEXCEPT
    {
        NN_VISRV_TRACE_IPC_M(GetDisplayResolution);
        return m_ClientHolder->GetDisplayResolution(pOutWidth.GetPointer(), pOutHeight.GetPointer(), displayId);
    }

    Result ManagerDisplayServiceImpl::AddToLayerStack(nn::vi::LayerId layerId, nn::vi::LayerStackType stackId) NN_NOEXCEPT
    {
        NN_VISRV_TRACE_IPC_M(AddToLayerStack);
        return m_ClientHolder->AddToLayerStack(layerId, stackId);
    }

    nn::Result ManagerDisplayServiceImpl::CreateManagedLayer(
        nn::sf::Out<nn::vi::LayerId> outLayerId,
        nn::vi::DisplayId displayId,
        nn::vi::LayerSettingsType settings,
        nn::applet::AppletResourceUserId userAruid
    ) NN_NOEXCEPT
    {
        NN_VISRV_TRACE_IPC_M(CreateManagedLayer);
        return m_ClientHolder->CreateManagedLayer(
            outLayerId.GetPointer(),
            displayId,
            settings,
            userAruid
        );
    }

    nn::Result ManagerDisplayServiceImpl::DestroyManagedLayer(nn::vi::LayerId layerId) NN_NOEXCEPT
    {
        NN_VISRV_TRACE_IPC_M(DestroyManagedLayer);
        return m_ClientHolder->DestroyManagedLayer(
            layerId
        );
    }

    nn::Result ManagerDisplayServiceImpl::CreateIndirectLayer(nn::sf::Out<nn::vi::IndirectLayerHandleType> outLayerHandle) NN_NOEXCEPT
    {
        NN_VISRV_TRACE_IPC_M(CreateIndirectLayer);
        return m_ClientHolder->CreateIndirectLayer(outLayerHandle.GetPointer());
    }

    nn::Result ManagerDisplayServiceImpl::DestroyIndirectLayer(nn::vi::IndirectLayerHandleType layerHandle) NN_NOEXCEPT
    {
        NN_VISRV_TRACE_IPC_M(DestroyIndirectLayer);
        return m_ClientHolder->DestroyIndirectLayer(layerHandle);
    }

    nn::Result ManagerDisplayServiceImpl::CreateIndirectProducerEndPoint(
        nn::sf::Out<nn::vi::IndirectProducerHandleType> outProducerEndPointHandle,
        nn::vi::IndirectLayerHandleType layerHandle,
        nn::applet::AppletResourceUserId aruid
    ) NN_NOEXCEPT
    {
        NN_VISRV_TRACE_IPC_M(CreateIndirectProducerEndPoint);
        return m_ClientHolder->CreateIndirectProducerEndPoint(
            outProducerEndPointHandle.GetPointer(),
            layerHandle,
            aruid
        );
    }

    nn::Result ManagerDisplayServiceImpl::DestroyIndirectProducerEndPoint(nn::vi::IndirectLayerHandleType layerHandle) NN_NOEXCEPT
    {
        NN_VISRV_TRACE_IPC_M(DestroyIndirectProducerEndPoint);
        return m_ClientHolder->DestroyIndirectProducerEndPoint(layerHandle);
    }

    nn::Result ManagerDisplayServiceImpl::CreateIndirectConsumerEndPoint(
        nn::sf::Out<nn::vi::IndirectConsumerHandleType> outConsumerEndPointHandle,
        nn::vi::IndirectLayerHandleType layerHandle,
        nn::applet::AppletResourceUserId aruid
    ) NN_NOEXCEPT
    {
        NN_VISRV_TRACE_IPC_M(CreateIndirectConsumerEndPoint);
        return m_ClientHolder->CreateIndirectConsumerEndPoint(
            outConsumerEndPointHandle.GetPointer(),
            layerHandle,
            aruid
        );
    }

    nn::Result ManagerDisplayServiceImpl::DestroyIndirectConsumerEndPoint(nn::vi::IndirectLayerHandleType layerHandle) NN_NOEXCEPT
    {
        NN_VISRV_TRACE_IPC_M(DestroyIndirectConsumerEndPoint);
        return m_ClientHolder->DestroyIndirectConsumerEndPoint(layerHandle);
    }

    nn::Result ManagerDisplayServiceImpl::CreateSharedBufferStaticStorage(nn::sf::Out<nn::vi::fbshare::SharedBufferHandle> outHandle, std::uint64_t storageKey, const nn::vi::fbshare::SharedMemoryPoolLayout& layout) NN_NOEXCEPT
    {
        NN_VISRV_TRACE_IPC_M(CreateSharedBufferStaticStorage);
        nn::vi::fbshare::SharedBufferHandle h = {};
        NN_RESULT_DO(m_ClientHolder->CreateSharedBufferStaticStorage(
            &h,
            storageKey,
            layout
        ));
        outHandle.Set(h);
        NN_RESULT_SUCCESS;
    }

    nn::Result ManagerDisplayServiceImpl::CreateSharedBufferTransferMemory(nn::sf::Out<nn::vi::fbshare::SharedBufferHandle> outHandle, nn::sf::NativeHandle&& transferMemoryHandle, std::uint64_t transferMemorySize, const nn::vi::fbshare::SharedMemoryPoolLayout& layout) NN_NOEXCEPT
    {
        NN_VISRV_TRACE_IPC_M(CreateSharedBufferTransferMemory);
        nn::vi::fbshare::SharedBufferHandle h = {};
        NN_RESULT_DO(m_ClientHolder->CreateSharedBufferTransferMemory(
            &h,
            transferMemoryHandle,
            static_cast<size_t>(transferMemorySize),
            layout
        ));
        outHandle.Set(h);
        NN_RESULT_SUCCESS;
    }

    nn::Result ManagerDisplayServiceImpl::CreateSharedBufferProcessHeap(nn::sf::Out<nn::vi::fbshare::SharedBufferHandle> outHandle, std::uint64_t blockId, const nn::vi::fbshare::SharedMemoryPoolLayout& layout) NN_NOEXCEPT
    {
        NN_VISRV_TRACE_IPC_M(CreateSharedBufferProcessHeap);
        nn::vi::fbshare::SharedBufferHandle h = {};
        NN_RESULT_DO(m_ClientHolder->CreateSharedBufferProcessHeap(
            &h,
            blockId,
            layout
        ));
        outHandle.Set(h);
        NN_RESULT_SUCCESS;
    }

    nn::Result ManagerDisplayServiceImpl::DestroySharedBuffer(nn::vi::fbshare::SharedBufferHandle handle) NN_NOEXCEPT
    {
        NN_VISRV_TRACE_IPC_M(DestroySharedBuffer);
        return m_ClientHolder->DestroySharedBuffer(handle);
    }

    nn::Result ManagerDisplayServiceImpl::BindSharedLowLevelLayerToManagedLayer(nn::vi::LayerId layerId, const nn::vi::DisplayName& displayName, nn::applet::AppletResourceUserId aruid) NN_NOEXCEPT
    {
        NN_VISRV_TRACE_IPC_M(BindSharedLowLevelLayerToManagedLayer);
        NN_RESULT_DO(m_ClientHolder->BindSharedLowLevelLayerToManagedLayer(layerId, displayName.value, aruid));
        NN_RESULT_SUCCESS;
    }

    nn::Result ManagerDisplayServiceImpl::BindSharedLowLevelLayerToIndirectLayer(nn::vi::IndirectProducerHandleType handle, nn::applet::AppletResourceUserId aruid) NN_NOEXCEPT
    {
        NN_VISRV_TRACE_IPC_M(BindSharedLowLevelLayerToIndirectLayer);
        NN_RESULT_DO(m_ClientHolder->BindSharedLowLevelLayerToIndirectLayer(handle, aruid));
        NN_RESULT_SUCCESS;
    }

    nn::Result ManagerDisplayServiceImpl::UnbindSharedLowLevelLayer(nn::vi::LayerId layerId) NN_NOEXCEPT
    {
        NN_VISRV_TRACE_IPC_M(UnbindSharedLowLevelLayer);
        NN_RESULT_DO(m_ClientHolder->UnbindSharedLowLevelLayer(layerId));
        NN_RESULT_SUCCESS;
    }

    nn::Result ManagerDisplayServiceImpl::ConnectSharedLowLevelLayerToSharedBuffer(nn::vi::LayerId layerId, nn::vi::fbshare::SharedBufferHandle hSharedBuffer) NN_NOEXCEPT
    {
        NN_VISRV_TRACE_IPC_M(ConnectSharedLowLevelLayerToSharedBuffer);
        NN_RESULT_DO(m_ClientHolder->ConnectSharedLowLevelLayerToSharedBuffer(layerId, hSharedBuffer));
        NN_RESULT_SUCCESS;
    }

    nn::Result ManagerDisplayServiceImpl::DisconnectSharedLowLevelLayerFromSharedBuffer(nn::vi::LayerId layerId) NN_NOEXCEPT
    {
        NN_VISRV_TRACE_IPC_M(DisconnectSharedLowLevelLayerFromSharedBuffer);
        NN_RESULT_DO(m_ClientHolder->DisconnectSharedLowLevelLayerFromSharedBuffer(layerId));
        NN_RESULT_SUCCESS;
    }

    nn::Result ManagerDisplayServiceImpl::CreateSharedLayer(nn::sf::Out<nn::vi::fbshare::SharedLayerHandle> outLayerHandle, nn::applet::AppletResourceUserId userAruid) NN_NOEXCEPT
    {
        NN_VISRV_TRACE_IPC_M(CreateSharedLayer);
        nn::vi::fbshare::SharedLayerHandle h = {};
        NN_RESULT_DO(m_ClientHolder->CreateSharedClientLayer(&h, userAruid));
        outLayerHandle.Set(h);
        NN_RESULT_SUCCESS;
    }

    nn::Result ManagerDisplayServiceImpl::DestroySharedLayer(nn::vi::fbshare::SharedLayerHandle hLayer) NN_NOEXCEPT
    {
        NN_VISRV_TRACE_IPC_M(DestroySharedLayer);
        NN_RESULT_DO(m_ClientHolder->DestroySharedClientLayer(hLayer));
        NN_RESULT_SUCCESS;
    }

    nn::Result ManagerDisplayServiceImpl::AttachSharedLayerToLowLevelLayer(
        nn::vi::fbshare::SharedLayerHandle hSharedLayer,
        nn::vi::LayerId layerId,
        const nn::vi::fbshare::SharedLayerTextureIndexList& frameBufferIndexList
    ) NN_NOEXCEPT
    {
        NN_VISRV_TRACE_IPC_M(AttachSharedLayerToLowLevelLayer);
        NN_RESULT_DO(m_ClientHolder->AttachSharedLayerToLowLevelLayer(hSharedLayer, layerId, frameBufferIndexList));
        NN_RESULT_SUCCESS;
    }

    nn::Result ManagerDisplayServiceImpl::StartDetachSharedLayerFromLowLevelLayer(nn::vi::fbshare::SharedLayerHandle hSharedLayer) NN_NOEXCEPT
    {
        NN_VISRV_TRACE_IPC_M(StartDetachSharedLayerFromLowLevelLayer);
        NN_RESULT_DO(m_ClientHolder->StartDetachSharedClientLayerFromLowLevelLayer(hSharedLayer));
        NN_RESULT_SUCCESS;
    }

    nn::Result ManagerDisplayServiceImpl::FinishDetachSharedLayerFromLowLevelLayer(nn::vi::fbshare::SharedLayerHandle hSharedLayer) NN_NOEXCEPT
    {
        NN_VISRV_TRACE_IPC_M(FinishDetachSharedLayerFromLowLevelLayer);
        NN_RESULT_DO(m_ClientHolder->FinishDetachSharedClientLayerFromLowLevelLayer(hSharedLayer));
        NN_RESULT_SUCCESS;
    }

    nn::Result ManagerDisplayServiceImpl::GetSharedLayerDetachReadyEvent(nn::sf::Out<nn::sf::NativeHandle> outHandle, nn::vi::fbshare::SharedLayerHandle hSharedLayer) NN_NOEXCEPT
    {
        NN_VISRV_TRACE_IPC_M(GetSharedLowLevelLayerSynchronizedEvent);
        nn::sf::NativeHandle h = {};
        NN_RESULT_DO(m_ClientHolder->GetSharedClientLayerDetachReadyEvent(h, hSharedLayer));
        outHandle.Set(std::move(h));
        NN_RESULT_SUCCESS;
    }

    nn::Result ManagerDisplayServiceImpl::ForceDetachSharedLayerFromLowLevelLayer(nn::vi::fbshare::SharedLayerHandle hSharedLayer) NN_NOEXCEPT
    {
        NN_VISRV_TRACE_IPC_M(ForceDetachSharedLayerFromLowLevelLayer);
        NN_RESULT_DO(m_ClientHolder->ForceDetachSharedLayerFromLowLevelLayer(hSharedLayer));
        NN_RESULT_SUCCESS;
    }

    nn::Result ManagerDisplayServiceImpl::GetSharedLowLevelLayerSynchronizedEvent(nn::sf::Out<nn::sf::NativeHandle> outHandle, nn::vi::LayerId layerId) NN_NOEXCEPT
    {
        NN_VISRV_TRACE_IPC_M(GetSharedLowLevelLayerSynchronizedEvent);
        nn::sf::NativeHandle h = {};
        NN_RESULT_DO(m_ClientHolder->GetSharedLowLevelLayerSynchronizedEvent(h, layerId));
        outHandle.Set(std::move(h));
        NN_RESULT_SUCCESS;
    }

    nn::Result ManagerDisplayServiceImpl::CheckSharedLowLevelLayerSynchronized(nn::sf::Out<std::int64_t> outDisplayedIndex, nn::vi::LayerId layerId) NN_NOEXCEPT
    {
        NN_VISRV_TRACE_IPC_M(CheckSharedLowLevelLayerSynchronized);
        int index = -1;
        NN_RESULT_DO(m_ClientHolder->CheckSharedLayerSynchronized(&index, layerId));
        outDisplayedIndex.Set(static_cast<int64_t>(index));
        NN_RESULT_SUCCESS;
    }

    nn::Result ManagerDisplayServiceImpl::RegisterSharedBufferImporterAruid(nn::vi::fbshare::SharedBufferHandle hBuffer, nn::applet::AppletResourceUserId importerAruid) NN_NOEXCEPT
    {
        NN_VISRV_TRACE_IPC_M(RegisterSharedBufferImporterAruid);
        NN_RESULT_DO(m_ClientHolder->RegiterSharedBufferImporterAruid(hBuffer, importerAruid));
        NN_RESULT_SUCCESS;
    }

    nn::Result ManagerDisplayServiceImpl::UnregisterSharedBufferImporterAruid(nn::vi::fbshare::SharedBufferHandle hBuffer, nn::applet::AppletResourceUserId importerAruid) NN_NOEXCEPT
    {
        NN_VISRV_TRACE_IPC_M(UnregisterSharedBufferImporterAruid);
        NN_RESULT_DO(m_ClientHolder->UnregiterSharedBufferImporterAruid(hBuffer, importerAruid));
        NN_RESULT_SUCCESS;
    }

    nn::Result ManagerDisplayServiceImpl::GetSharedLayerLayerStacks(nn::sf::Out<nn::vi::LayerStackFlagType> outStacks, nn::vi::fbshare::SharedLayerHandle hSharedLayer) NN_NOEXCEPT
    {
        NN_VISRV_TRACE_IPC_M(GetSharedLayerLayerStacks);
        NN_RESULT_DO(m_ClientHolder->GetSharedClientLayerLayerStacks(outStacks.GetPointer(), hSharedLayer));
        NN_RESULT_SUCCESS;
    }

    nn::Result ManagerDisplayServiceImpl::SetSharedLayerLayerStacks(nn::vi::fbshare::SharedLayerHandle hSharedLayer, nn::vi::LayerStackFlagType stacks) NN_NOEXCEPT
    {
        NN_VISRV_TRACE_IPC_M(SetSharedLayerLayerStacks(hSharedLayer, stacks));
        NN_RESULT_DO(m_ClientHolder->SetSharedClientLayerLayerStacks(hSharedLayer, stacks));
        NN_RESULT_SUCCESS;
    }

    nn::Result ManagerDisplayServiceImpl::PresentDetachedSharedFrameBufferToLowLevelLayer(nn::vi::fbshare::SharedBufferHandle hSharedBuffer, nn::vi::LayerId layerId, std::int64_t index) NN_NOEXCEPT
    {
        NN_VISRV_TRACE_IPC_M(PresentDetachedSharedFrameBufferToLowLevelLayer);
        NN_RESULT_DO(m_ClientHolder->PresentDetachedSharedFrameBufferToLowLevelLayer(hSharedBuffer, layerId, static_cast<int>(index)));
        NN_RESULT_SUCCESS;
    }

    nn::Result ManagerDisplayServiceImpl::FillDetachedSharedFrameBufferColor(nn::vi::fbshare::SharedBufferHandle hSharedBuffer, std::int64_t index, std::uint32_t color, const nn::vi::fbshare::SharedTextureOption& option) NN_NOEXCEPT
    {
        NN_VISRV_TRACE_IPC_M(FillDetachedSharedFrameBufferColor);
        NN_RESULT_DO(m_ClientHolder->FillDetachedSharedFrameBufferColor(hSharedBuffer, static_cast<int>(index), color, option));
        NN_RESULT_SUCCESS;
    }

    nn::Result ManagerDisplayServiceImpl::GetDetachedSharedFrameBufferImage(nn::sf::Out<std::uint64_t> outDataSize, const nn::sf::OutBuffer& data, nn::vi::fbshare::SharedBufferHandle hSharedBuffer, std::int64_t index) NN_NOEXCEPT
    {
        NN_VISRV_TRACE_IPC_M(GetDetachedSharedFrameBufferImage);
        size_t outSize = 0;
        NN_RESULT_DO(m_ClientHolder->GetDetachedSharedFrameBufferImage(&outSize, data.GetPointerUnsafe(), data.GetSize(), hSharedBuffer, static_cast<int>(index)));
        outDataSize.Set(outSize);
        NN_RESULT_SUCCESS;
    }

    nn::Result ManagerDisplayServiceImpl::SetDetachedSharedFrameBufferImage(nn::vi::fbshare::SharedBufferHandle hSharedBuffer, std::int64_t index, const nn::sf::InBuffer& data, const nn::vi::fbshare::SharedTextureOption& dstOption, nn::vi::ImageTransformType srcTransform) NN_NOEXCEPT
    {
        NN_VISRV_TRACE_IPC_M(SetDetachedSharedFrameBufferImage);
        NN_RESULT_DO(m_ClientHolder->SetDetachedSharedFrameBufferImage(hSharedBuffer, static_cast<int>(index), data.GetPointerUnsafe(), data.GetSize(), dstOption, srcTransform));
        NN_RESULT_SUCCESS;
    }

    nn::Result ManagerDisplayServiceImpl::SetDetachedSharedFrameBufferSubImage(nn::vi::fbshare::SharedBufferHandle hSharedBuffer, std::int64_t index, std::int32_t x, std::int32_t y, std::int32_t w, std::int32_t h, std::uint32_t bgColor, const nn::sf::InBuffer& data, const nn::vi::fbshare::SharedTextureOption& dstOption, nn::vi::ImageTransformType srcTransform) NN_NOEXCEPT
    {
        NN_VISRV_TRACE_IPC_M(SetDetachedSharedFrameBufferSubImage);
        NN_RESULT_DO(m_ClientHolder->SetDetachedSharedFrameBufferSubImage(
            hSharedBuffer,
            static_cast<int>(index),
            static_cast<int>(x),
            static_cast<int>(y),
            static_cast<int>(w),
            static_cast<int>(h),
            bgColor,
            data.GetPointerUnsafe(),
            data.GetSize(),
            dstOption,
            srcTransform
        ));
        NN_RESULT_SUCCESS;
    }

    nn::Result ManagerDisplayServiceImpl::ExpandStartupLogoOnSharedFrameBuffer(nn::vi::fbshare::SharedBufferHandle hSharedBuffer, std::int64_t dstIndex, const nn::vi::fbshare::SharedTextureOption& option) NN_NOEXCEPT
    {
        NN_VISRV_TRACE_IPC_M(ExpandStartupLogoOnSharedFrameBuffer);
        NN_RESULT_DO(m_ClientHolder->ExpandStartupLogoOnSharedFrameBuffer(
            hSharedBuffer,
            static_cast<int>(dstIndex),
            option
        ));
        NN_RESULT_SUCCESS;
    }

    nn::Result ManagerDisplayServiceImpl::CopyDetachedSharedFrameBufferImage(nn::vi::fbshare::SharedBufferHandle hDstSharedBuffer, std::int64_t dstIndex, nn::vi::fbshare::SharedBufferHandle hSrcSharedBuffer, std::int64_t srcIndex, const nn::vi::fbshare::SharedTextureOption& option, nn::vi::LayerStackFlagType stacksFilter, nn::vi::LayerStackFlagType nullStacks) NN_NOEXCEPT
    {
        NN_VISRV_TRACE_IPC_M(CopyDetachedSharedFrameBufferImage);
        NN_RESULT_DO(m_ClientHolder->CopyDetachedSharedFrameBufferImage(hDstSharedBuffer, static_cast<int>(dstIndex), hSrcSharedBuffer, static_cast<int>(srcIndex), option, stacksFilter, nullStacks));
        NN_RESULT_SUCCESS;
    }

    nn::Result ManagerDisplayServiceImpl::GetSharedFrameBufferContentParameter(nn::sf::Out<nn::vi::LayerStackFlagType> outLayerStacks, nn::sf::Out<nn::vi::CropRegion> outCropRegion, nn::sf::Out<std::int32_t> outScalingMode, nn::sf::Out<std::uint32_t> outTransform, nn::sf::Out<std::int32_t> outPresentInterval, nn::vi::fbshare::SharedBufferHandle hSharedBuffer, std::int64_t index) NN_NOEXCEPT
    {
        NN_VISRV_TRACE_IPC_M(GetSharedFrameBufferContentParameter);
        NN_RESULT_DO(m_ClientHolder->GetSharedFrameBufferContentParameter(
            outLayerStacks.GetPointer(),
            outCropRegion.GetPointer(),
            outScalingMode.GetPointer(),
            outTransform.GetPointer(),
            outPresentInterval.GetPointer(),
            hSharedBuffer,
            static_cast<int>(index)
        ));
        NN_RESULT_SUCCESS;
    }

    nn::Result ManagerDisplayServiceImpl::AcquireLayerTexturePresentingEvent(nn::sf::Out<nn::sf::NativeHandle> outHandle, nn::vi::LayerId layerId) NN_NOEXCEPT
    {
        NN_VISRV_TRACE_IPC_M(AcquireLayerTexturePresentingEvent);
        nn::os::NativeHandle h = {};
        bool isManaged = false;
        NN_RESULT_DO(m_ClientHolder->AcquireLayerTexturePresentingEvent(&h, &isManaged, layerId));
        outHandle.Set(nn::sf::NativeHandle(h, isManaged));
        NN_RESULT_SUCCESS;
    }

    nn::Result ManagerDisplayServiceImpl::ReleaseLayerTexturePresentingEvent(nn::vi::LayerId layerId) NN_NOEXCEPT
    {
        NN_VISRV_TRACE_IPC_M(ReleaseLayerTexturePresentingEvent);
        return m_ClientHolder->ReleaseLayerTexturePresentingEvent(layerId);
    }

    nn::Result ManagerDisplayServiceImpl::RemoveFromLayerStack(nn::vi::LayerId layerId, nn::vi::LayerStackType stackId) NN_NOEXCEPT
    {
        NN_VISRV_TRACE_IPC_M(RemoveFromLayerStack);
        return m_ClientHolder->RemoveFromLayerStack(layerId, stackId);
    }

    nn::Result ManagerDisplayServiceImpl::SetLayerConfig(nn::vi::LayerId layerId, nn::vi::LayerConfig config) NN_NOEXCEPT
    {
        NN_VISRV_TRACE_IPC_M(SetLayerConfig);
        return m_ClientHolder->SetLayerConfig(layerId, config);
    }

    nn::Result ManagerDisplayServiceImpl::AttachLayerPresentationTracer(nn::vi::LayerId layerId) NN_NOEXCEPT
    {
        NN_VISRV_TRACE_IPC_M(AttachLayerPresentationTracer);
        return m_ClientHolder->AttachLayerPresentationTracer(layerId);
    }

    nn::Result ManagerDisplayServiceImpl::DetachLayerPresentationTracer(nn::vi::LayerId layerId) NN_NOEXCEPT
    {
        NN_VISRV_TRACE_IPC_M(DetachLayerPresentationTracer);
        return m_ClientHolder->DetachLayerPresentationTracer(layerId);
    }

    nn::Result ManagerDisplayServiceImpl::StartLayerPresentationRecording(nn::vi::LayerId layerId) NN_NOEXCEPT
    {
        NN_VISRV_TRACE_IPC_M(StartLayerPresentationRecording);
        return m_ClientHolder->StartLayerPresentationRecording(layerId);
    }

    nn::Result ManagerDisplayServiceImpl::StopLayerPresentationRecording(nn::vi::LayerId layerId) NN_NOEXCEPT
    {
        NN_VISRV_TRACE_IPC_M(StopLayerPresentationRecording);
        return m_ClientHolder->StopLayerPresentationRecording(layerId);
    }

    nn::Result ManagerDisplayServiceImpl::StartLayerPresentationFenceWait(nn::vi::LayerId layerId) NN_NOEXCEPT
    {
        NN_VISRV_TRACE_IPC_M(StartLayerPresentationFenceWait);
        return m_ClientHolder->StartLayerPresentationFenceWait(layerId);
    }

    nn::Result ManagerDisplayServiceImpl::StopLayerPresentationFenceWait(nn::vi::LayerId layerId) NN_NOEXCEPT
    {
        NN_VISRV_TRACE_IPC_M(StopLayerPresentationFenceWait);
        return m_ClientHolder->StopLayerPresentationFenceWait(layerId);
    }

    nn::Result ManagerDisplayServiceImpl::GetLayerPresentationAllFencesExpiredEvent(nn::sf::Out<nn::sf::NativeHandle> outHandle, nn::vi::LayerId layerId) NN_NOEXCEPT
    {
        NN_VISRV_TRACE_IPC_M(GetLayerPresentationAllFencesExpiredEvent);
        return m_ClientHolder->GetLayerPresentationAllFencesExpiredEvent(*outHandle, layerId);
    }

    void ManagerDisplayServiceImpl::SetContentVisibility(bool isVisible) NN_NOEXCEPT
    {
        NN_VISRV_TRACE_IPC_M(SetContentVisibility);
        nn::visrv::master::SetContentVisibility(isVisible);
    }

    nn::Result ManagerDisplayServiceImpl::GetDisplayHotplugState(nn::sf::Out<nn::vi::HotplugStateType> outState, nn::vi::DisplayId displayId) NN_NOEXCEPT
    {
        NN_VISRV_TRACE_IPC_M(GetDisplayHotplugState);
        return m_ClientHolder->GetDisplayHotplugState(outState.GetPointer(), displayId);
    }

    nn::Result ManagerDisplayServiceImpl::SetDisplayAlpha(nn::vi::DisplayId displayId, float alpha) NN_NOEXCEPT
    {
        NN_VISRV_TRACE_IPC_M(SetDisplayAlpha);
        return m_ClientHolder->SetDisplayAlpha(displayId, alpha);
    }

    nn::Result ManagerDisplayServiceImpl::SetDisplayLayerStack(nn::vi::DisplayId displayId, nn::vi::LayerStackType id) NN_NOEXCEPT
    {
        NN_VISRV_TRACE_IPC_M(SetDisplayLayerStack);
        return m_ClientHolder->SetDisplayLayerStack(displayId, id);
    }

    nn::Result ManagerDisplayServiceImpl::SetDisplayPowerState(nn::vi::DisplayId displayId, nn::vi::PowerStateType state) NN_NOEXCEPT
    {
        NN_VISRV_TRACE_IPC_M(SetDisplayPowerState);
        return m_ClientHolder->SetDisplayPowerState(displayId, state);
    }

    nn::Result ManagerDisplayServiceImpl::GetDisplayHotplugEvent(nn::sf::Out<nn::sf::NativeHandle> outHandle, nn::vi::DisplayId displayId) NN_NOEXCEPT
    {
        NN_VISRV_TRACE_IPC_M(GetDisplayHotplugEvent);
        nn::os::NativeHandle handle;
        bool isManaged;

        nn::Result result = m_ClientHolder->GetDisplayHotplugEvent(&handle, &isManaged, displayId);

        if (result.IsSuccess())
        {
            *outHandle = nn::sf::NativeHandle(handle, isManaged);
        }

        return result;
    }

    nn::Result ManagerDisplayServiceImpl::GetDisplayModeChangedEvent(nn::sf::Out<nn::sf::NativeHandle> outHandle, nn::vi::DisplayId displayId) NN_NOEXCEPT
    {
        NN_VISRV_TRACE_IPC_M(GetDisplayModeChangedEvent);
        nn::os::NativeHandle handle;

        NN_RESULT_DO(m_ClientHolder->GetDisplayModeChangedEvent(&handle, displayId));
        *outHandle = nn::sf::NativeHandle(handle, true);

        NN_RESULT_SUCCESS;
    }

    nn::Result ManagerDisplayServiceImpl::GetDisplayErrorEvent(nn::sf::Out<nn::sf::NativeHandle> outHandle, nn::vi::DisplayId displayId) NN_NOEXCEPT
    {
        NN_VISRV_TRACE_IPC_M(GetDisplayErrorEvent);
        nn::os::NativeHandle handle;
        bool isManaged;

        nn::Result result = m_ClientHolder->GetDisplayErrorEvent(&handle, &isManaged, displayId);

        if (result.IsSuccess())
        {
            *outHandle = nn::sf::NativeHandle(handle, isManaged);
        }

        return result;
    }

    nn::Result ManagerDisplayServiceImpl::GetCompositorErrorInfo(nn::sf::Out<nn::vi::CompositorError> outErrorInfo, nn::sf::Out<std::int32_t> outLength, std::int64_t errorId, nn::vi::DisplayId displayId) NN_NOEXCEPT
    {
        NN_VISRV_TRACE_IPC_M(GetCompositorErrorInfo);

        nn::Result result = m_ClientHolder->GetCompositorErrorInfo(outErrorInfo.GetPointer(), outLength.GetPointer(), errorId, displayId);

        return result;
    }

    Result ManagerDisplayServiceImpl::SetConductorLayer(nn::vi::LayerId layerId, bool isConductor) NN_NOEXCEPT
    {
        NN_VISRV_TRACE_IPC_M(SetConductorLayer);
        return m_ClientHolder->SetConductorLayer(layerId, isConductor);
    }

    nn::Result ManagerDisplayServiceImpl::SetIndirectProducerFlipOffset(nn::vi::IndirectLayerHandleType layerHandle, nn::vi::IndirectProducerHandleType producerHandle, nn::TimeSpan offset) NN_NOEXCEPT
    {
        NN_VISRV_TRACE_IPC_M(SetIndirectProducerFlipOffset);
        return m_ClientHolder->SetIndirectProducerFlipOffset(layerHandle, producerHandle, offset);
    }

    nn::Result ManagerDisplayServiceImpl::SetLayerVisibility(nn::vi::LayerId layerId, bool isVisible) NN_NOEXCEPT
    {
        NN_VISRV_TRACE_IPC_M(SetLayerVisibility);
        return m_ClientHolder->SetLayerVisibility(layerId, isVisible);
    }

    nn::Result ManagerDisplayServiceImpl::SetDefaultDisplay(nn::vi::DisplayId id) NN_NOEXCEPT
    {
        NN_VISRV_TRACE_IPC_M(SetDisplayConfig);
        return m_ClientHolder->SetDefaultDisplay(id);
    }

}}}

