﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "visrv_ReadWriteToSurface.h"

#include <algorithm>
#include <nn/result/result_HandlingUtility.h>
#include <nn/util/util_ScopeExit.h>
#include <nn/util/util_Color.h>
#include <nn/vi/vi_Result.h>

//#include <nn/nn_SdkLog.h>

namespace nn{ namespace visrv{ namespace native{

    namespace
    {
        static uint32_t CalculateBlockLinearOffset(
            uint32_t* pOutContigBytes,
            uint32_t x,
            uint32_t y,
            uint32_t pitch,
            uint32_t blockHeightLog2
        ) NN_NOEXCEPT
        {
            uint32_t blockHeight = (1 << blockHeightLog2);
            uint32_t gobOffset =
                (y / (8 * blockHeight)) * pitch * 8 * blockHeight +
                (x / 64) * 512 * blockHeight +
                ((y % (8 * blockHeight)) / 8) * 512;

            *pOutContigBytes = 16 - (x % 16);
            return gobOffset +
                ((x % 64) / 32) * 256 +
                ((y % 8) / 2) * 64 +
                ((x % 32) / 16) * 32 +
                (y % 2) * 16 +
                (x % 16);
        }

        uint32_t GetWriteColorRgba8(uint32_t color, WriteSurfaceDestinationOption option) NN_NOEXCEPT
        {
            nn::util::Color4u8 src;
            std::memcpy(&src, &color, sizeof(uint32_t));

            nn::util::Color4u8 dst = src;
            if(option & WriteSurfaceDestinationOption_PreMultipleAlpha)
            {
                uint32_t alpha1 = src.GetA() + 1;
                dst.SetR(static_cast<uint8_t>((alpha1 * src.GetR()) >> 8));
                dst.SetG(static_cast<uint8_t>((alpha1 * src.GetG()) >> 8));
                dst.SetB(static_cast<uint8_t>((alpha1 * src.GetB()) >> 8));
            }
            if(option & WriteSurfaceDestinationOption_Opaque)
            {
                dst.SetA(255);
            }

            uint32_t v;
            std::memcpy(&v, &dst, sizeof(uint32_t));
            return v;
        }

    }

    nn::Result ReadFromSurfaceBlockLinearRgba8(
        void* pDestinationBuffer,
        size_t destinationSize,
        const void* pSourceBuffer,
        int targetPosX,
        int targetPosY,
        int targetWidth,
        int targetHeight,
        uint32_t pitch,
        uint32_t blockHeightLog2
    ) NN_NOEXCEPT
    {
        const int PixelSize = 4;
        NN_RESULT_THROW_UNLESS(destinationSize >= PixelSize * targetWidth * targetHeight, nn::vi::ResultInvalidRange());

        char* pDst = reinterpret_cast<char*>(pDestinationBuffer);
        const char* pSrc = reinterpret_cast<const char*>(pSourceBuffer);

        for (uint32_t dy = 0; dy < targetHeight; dy++)
        {
            uint32_t y = targetPosY + dy;
            auto pDstLine = pDst + PixelSize * targetWidth * dy;

            uint32_t dx = 0;
            while(dx < targetWidth)
            {
                uint32_t x = targetPosX + dx;
                uint32_t contigBytes = 1;
                uint32_t offset = CalculateBlockLinearOffset(&contigBytes, PixelSize * x, y, pitch, blockHeightLog2);
                uint32_t sizeToWrite = std::min<uint32_t>(PixelSize * (targetWidth - dx), contigBytes);

                uint32_t pixelToWrite = sizeToWrite / PixelSize;

                if(pixelToWrite == 0)
                {
                    break;
                }
                std::memcpy(pDstLine + PixelSize * dx, pSrc + offset, PixelSize * pixelToWrite);

                dx += pixelToWrite;
            }
        }

        NN_RESULT_SUCCESS;
    }

    nn::Result WriteToSurfaceBlockLinearRgba8(
        void* pDestinationBuffer,
        size_t destinationSize,
        uint32_t destWidth,
        uint32_t destHeight,
        const void* pSourceBuffer,
        size_t sourceBufferSize,
        int targetPosX,
        int targetPosY,
        int targetWidth,
        int targetHeight,
        uint32_t pitch,
        uint32_t blockHeightLog2,
        WriteSurfaceDestinationOption dstOption,
        WriteSurfaceSourceOption srcOption
    ) NN_NOEXCEPT
    {
        const int PixelSize = 4;
        NN_RESULT_THROW_UNLESS(destinationSize >= PixelSize * destWidth * destHeight, nn::vi::ResultInvalidRange());
        NN_RESULT_THROW_UNLESS(sourceBufferSize >= PixelSize * targetWidth * targetHeight, nn::vi::ResultInvalidRange());

        if(targetWidth == 0 || targetHeight == 0)
        {
            NN_RESULT_SUCCESS;
        }

        NN_RESULT_THROW_UNLESS(targetPosX >= 0, nn::vi::ResultInvalidRange());
        NN_RESULT_THROW_UNLESS(targetPosY >= 0, nn::vi::ResultInvalidRange());
        NN_RESULT_THROW_UNLESS(targetWidth >= 0, nn::vi::ResultInvalidRange());
        NN_RESULT_THROW_UNLESS(targetHeight >= 0, nn::vi::ResultInvalidRange());
        NN_RESULT_THROW_UNLESS(targetPosX + targetWidth <= destWidth, nn::vi::ResultInvalidRange());
        NN_RESULT_THROW_UNLESS(targetPosY + targetHeight <= destHeight, nn::vi::ResultInvalidRange());


        char* pDst = reinterpret_cast<char*>(pDestinationBuffer);
        const char* pSrc = reinterpret_cast<const char*>(pSourceBuffer);

        const char* pSrcLine = pSrc;
        ptrdiff_t srcPitch = PixelSize * targetWidth;
        if(srcOption & WriteSurfaceSourceOption_FlipY)
        {
            pSrcLine += (targetHeight - 1) * srcPitch;
            srcPitch *= -1;
        }

        uint32_t dstY = targetPosY;
        int32_t dstYDelta = 1;
        if(dstOption & WriteSurfaceDestinationOption_FlipY)
        {
            dstY = destHeight - 1 - targetPosY;
            dstYDelta = -1;
        }

        for(uint32_t dy = 0; dy < targetHeight; dy++)
        {
            //NN_SDK_LOG("copy dstY=%d srcY=%d\n", dstY, static_cast<int>((pSrcLine - pSrc) / (PixelSize * targetWidth)));
            uint32_t dx = 0;
            while(dx < targetWidth)
            {
                uint32_t x = targetPosX + dx;
                uint32_t contigBytes = 1;
                uint32_t offset = CalculateBlockLinearOffset(&contigBytes, PixelSize * x, dstY, pitch, blockHeightLog2);
                uint32_t sizeToWrite = std::min<uint32_t>(PixelSize * (targetWidth - dx), contigBytes);

                uint32_t pixelToWrite = sizeToWrite / PixelSize;
                if(pixelToWrite == 0)
                {
                    break;
                }

                for(uint32_t k = 0; k < pixelToWrite; k++)
                {
                    char* pDstPixel = pDst + offset + PixelSize * k;
                    const char* pSrcPixel = pSrcLine + PixelSize * (dx + k);
                    uint32_t srcColor;
                    std::memcpy(&srcColor, pSrcPixel, sizeof(uint32_t));
                    uint32_t dstColor = GetWriteColorRgba8(srcColor, dstOption);
                    std::memcpy(pDstPixel, &dstColor, sizeof(uint32_t));
                }

                dx += pixelToWrite;
            }

            dstY += dstYDelta;
            pSrcLine += srcPitch;
        }

        NN_RESULT_SUCCESS;
    }

    nn::Result FillSurfaceRgba8(
        void* pDestinationBuffer,
        size_t destinationSize,
        uint32_t color,
        WriteSurfaceDestinationOption dstOption
    ) NN_NOEXCEPT
    {
        const int PixelSize = 4;
        NN_RESULT_THROW_UNLESS(destinationSize % PixelSize == 0, nn::vi::ResultInvalidRange());

        uint32_t v = GetWriteColorRgba8(color, dstOption);

        char* pDstBeg = reinterpret_cast<char*>(pDestinationBuffer);
        char* pDstEnd = pDstBeg + destinationSize;
        for(char* p = pDstBeg; p < pDstEnd; p += sizeof(v))
        {
            std::memcpy(p, &v, sizeof(v));
        }

        NN_RESULT_SUCCESS;
    }

}}}
