﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "visrv_NativeType.h"

#include "../visrv_Log.h"

#include <hos/nvhost_devctl.h>
#include <nvrm_channel.h>
#include <nvnflinger_service.h>

namespace nn{ namespace visrv{ namespace native{

    NativeRmFence::NativeRmFence() NN_NOEXCEPT
    {
        this->syncpointId = NVRM_INVALID_SYNCPOINT_ID;
        this->value = 0;
    }

    NativeRmFence::NativeRmFence(const NvRmFence& src) NN_NOEXCEPT
    {
        this->syncpointId = src.SyncPointID;
        this->value = src.Value;
    }

    //--------------------------------

    NN_STATIC_ASSERT(nn::visrv::native::NativeRmSync::FenceCountMax == LIBSYNC_FENCES_PER_FD);
    NN_STATIC_ASSERT(sizeof(nn::vi::native::NativeSync) == sizeof(nn::visrv::native::NativeRmSync));
    NN_STATIC_ASSERT(NN_ALIGNOF(nn::vi::native::NativeSync) >= NN_ALIGNOF(nn::visrv::native::NativeRmSync));

    const int NativeRmSync::FenceCountMax;

    NativeRmSync::NativeRmSync() NN_NOEXCEPT
        : count(0)
        , fenceList()
    {
    }

    void NativeRmSync::Clear() NN_NOEXCEPT
    {
        this->count = 0;
        for(int i = 0; i < FenceCountMax; i++)
        {
            this->fenceList[i] = NativeRmFence();
        }
    }

    void NativeRmSync::WriteTo(nn::vi::native::NativeSync* pOutValue) const NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_NOT_NULL(pOutValue);
        std::memcpy(pOutValue, this, sizeof(*this));
    }

    bool NativeRmSync::WriteTo(android::sp<android::Fence>* pOutValue) const NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_NOT_NULL(pOutValue);
        if(this->count == 0)
        {
            *pOutValue = android::Fence::NO_FENCE;
            return true;
        }

        NvRmFence fence[FenceCountMax] = {};
        for(int i = 0; i < this->count; i++)
        {
            fence[i].SyncPointID = this->fenceList[i].syncpointId;
            fence[i].Value = this->fenceList[i].value;
        }

        int fd = sync_create_nvrm_mapping(fence, this->count);
        if(fd < 0)
        {
            NN_VISRV_LOG_ERR("[sync]couldn't create sync mapping\n");
            *pOutValue = android::Fence::NO_FENCE;
            return false;
        }

        // fd's ownership is moved into android::Fence
        *pOutValue = new android::Fence(fd);
        return true;
    }

    void NativeRmSync::ReadFrom(const nn::vi::native::NativeSync& value) NN_NOEXCEPT
    {
        std::memcpy(this, &value, sizeof(*this));
    }

    void NativeRmSync::ReadFrom(const syncfd_nvrm_mapping& value) NN_NOEXCEPT
    {
        int n = value.num_fences;
        if(n > FenceCountMax)
        {
            NN_VISRV_LOG_WARN("[sync]too many fence (n=%d)\n", value.num_fences);
            n = FenceCountMax;
        }
        for(int i = 0; i < n; i++)
        {
            this->fenceList[i] = NativeRmFence(value.fences[i]);
        }
        for(int i = n; i < FenceCountMax; i++)
        {
            this->fenceList[i] = NativeRmFence();
        }
        this->count = value.num_fences;
    }

    void NativeRmSync::ReadFrom(const android::sp<android::Fence>& value) NN_NOEXCEPT
    {
        if(value == nullptr)
        {
            Clear();
        }
        else
        {
            syncfd_nvrm_mapping mapping = {};
            auto fd = value->dup();
            NN_UTIL_SCOPE_EXIT{ sync_close(fd); };
            sync_get_nvrm_mapping(fd, &mapping);
            ReadFrom(mapping);
        }
    }

}}}
