﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "visrv_Layer.h"
#include "visrv_Layer-os.horizon.h"
#include "../../vi/vi_Config.h"

#include <new>
#include <nn/vi/vi_Result.h>
#include "detail/visrv_Android.h"
#include "detail/visrv_Common.h"
#include "visrv_Display.h"
#include "../visrv_Log.h"

namespace nn{ namespace visrv{ namespace native{

    // defined in native/visrv_SurfaceComposerClientHolder.cpp
    android::SurfaceComposerClient& GetSurfaceComposerClient() NN_NOEXCEPT;
}}}

namespace nn{ namespace visrv{ namespace master{

    Layer::Layer(Display* pDisplay, nn::vi::PolicyLevelType policyLevel) NN_NOEXCEPT
        : m_Client(native::GetSurfaceComposerClient())
        , m_Control(nullptr)
        , m_pDisplay(pDisplay)
        , m_Z(pDisplay->GetInfo(policyLevel).GetPolicy().GetZOrderCountMin())
        , m_CurrentWidth(0)
        , m_CurrentHeight(0)
        , m_X(0.f)
        , m_Y(0.f)
        , m_PolicyLevel(policyLevel)
    {
        NN_SDK_ASSERT_NOT_NULL(m_pDisplay);
        m_Settings.Reset();
        m_Stack.Reset();

        m_Stack.Set<nn::vi::LayerStackFlags::Default>();
    }

    Layer::~Layer() NN_NOEXCEPT
    {
        m_GraphicBufferProducer = nullptr;
        m_Handle = nullptr;
        m_Control = nullptr;
    }

    nn::Result Layer::Initialize(nn::vi::LayerSettings settings, nn::applet::AppletResourceUserId aruid) NN_NOEXCEPT
    {
        m_Settings = settings;
        uint32_t flags = detail::ConvertLayerSettings(settings);

        if( settings.Test<nn::vi::LayerFlags::Fullscreen>() )
        {
            NN_ABORT_UNLESS_RESULT_SUCCESS(m_pDisplay->GetLogicalResolution(&m_CurrentWidth, &m_CurrentHeight));
        }

        m_Control = m_Client.createSurface(
            android::String8(),
            nn::vi::LayerDefaultWidth,  // Using same default as EGL workaround
            nn::vi::LayerDefaultHeight, // Using same default as EGL workaround
            android::PIXEL_FORMAT_RGBA_8888, // doesn't matter in our implementation
            flags,
            m_Handle,
            m_GraphicBufferProducer
            );

        if( m_Control == nullptr )
        {
            VI_LOG("Failed to create surface.");
            return nn::vi::ResultOperationFailed();
        }

        m_Client.setSurfaceARUID(m_Handle, aruid);

        android::SurfaceComposerClient::openGlobalTransaction();
        m_Control->setSize(m_CurrentWidth, m_CurrentHeight);
        android::SurfaceComposerClient::closeGlobalTransaction();

        return SetZ(m_Z);
    }

    nn::Result Layer::SetPosition(float x, float y) NN_NOEXCEPT
    {
        int displayWidth;
        int displayHeight;

        NN_ABORT_UNLESS_RESULT_SUCCESS(m_pDisplay->GetLogicalResolution(&displayWidth, &displayHeight));

        if( !(0.f <= x &&
              0.f <= y &&
              x <= displayWidth - GetCurrentWidth() &&
              y <= displayHeight - GetCurrentHeight()) )
        {
            return nn::vi::ResultInvalidRange();
        }

        android::SurfaceComposerClient::openGlobalTransaction();
        android::status_t status = m_Control->setPosition(x, y);
        android::SurfaceComposerClient::closeGlobalTransaction();

        if( status != android::OK )
        {
            return nn::vi::ResultOperationFailed();
        }

        m_X = x;
        m_Y = y;

        return nn::ResultSuccess();
    }

    nn::Result Layer::SetZ(int z) NN_NOEXCEPT
    {
        if( !IsValidZOrder(z) )
        {
            return nn::vi::ResultInvalidRange();
        }

        android::SurfaceComposerClient::openGlobalTransaction();
        auto result = SetZUnsafe(z);
        android::SurfaceComposerClient::closeGlobalTransaction();

        return result;
    }

    nn::Result Layer::SetZUnsafe(int z) NN_NOEXCEPT
    {
        if( !m_pDisplay->GetInfo(m_PolicyLevel).ValidateZOrder(z) )
        {
            return nn::vi::ResultInvalidRange();
        }

        android::status_t status = m_Control->setLayer(z);

        if( status != android::OK )
        {
            return nn::vi::ResultOperationFailed();
        }

        m_Z = z;

        return nn::ResultSuccess();
    }

    nn::Result Layer::SetSize(int width, int height) NN_NOEXCEPT
    {
        int displayWidth;
        int displayHeight;

        NN_ABORT_UNLESS_RESULT_SUCCESS(m_pDisplay->GetLogicalResolution(&displayWidth, &displayHeight));
        NN_SDK_ASSERT(width >= 0);
        NN_SDK_ASSERT(height >= 0);

        if( m_Settings.Test<nn::vi::LayerFlags::Fullscreen>() )
        {
            if( width != displayWidth || height != displayHeight )
            {
                return nn::vi::ResultInvalidRange();
            }
            else
            {
                // fullscreen and dimensions are already equal to logical resolution
                // nothing left to do
                return nn::ResultSuccess();
            }
        }
        else
        {
            if( !(width + GetX() <= displayWidth &&
                   height + GetY() <= displayHeight) )
            {
                return nn::vi::ResultInvalidRange();
            }
        }

        android::SurfaceComposerClient::openGlobalTransaction();
        android::status_t status = m_Control->setSize(width, height);
        android::SurfaceComposerClient::closeGlobalTransaction();

        if( status != android::OK )
        {
            return nn::vi::ResultOperationFailed();
        }

        m_CurrentWidth = width;
        m_CurrentHeight = height;

        return nn::ResultSuccess();
    }

    nn::Result Layer::SetVisibility(bool isVisible) NN_NOEXCEPT
    {
        android::SurfaceComposerClient::openGlobalTransaction();
        android::status_t status = SetVisibilityUnsafe(isVisible);
        android::SurfaceComposerClient::closeGlobalTransaction();

        if( status != android::OK )
        {
            return nn::vi::ResultOperationFailed();
        }

        return nn::ResultSuccess();
    }

    nn::Result Layer::SetAlpha(float alpha) NN_NOEXCEPT
    {
        NN_SDK_ASSERT(alpha >= 0.f && alpha <= 1.f);

        android::SurfaceComposerClient::openGlobalTransaction();
        android::status_t status = m_Control->setAlpha(alpha);
        android::SurfaceComposerClient::closeGlobalTransaction();

        if( status != android::OK )
        {
            return nn::vi::ResultOperationFailed();
        }

        return nn::ResultSuccess();
    }

    nn::Result Layer::AddToStack(nn::vi::LayerStack id) NN_NOEXCEPT
    {
        if ( !IsValidStack(id) )
        {
            return nn::vi::ResultInvalidRange();
        }

        m_Stack.Set(id);

        nn::Result result = UpdateLayerStack();

        if ( result.IsFailure() )
        {
            m_Stack.Reset(id);
        }

        return result;
    }

    nn::Result Layer::RemoveFromStack(nn::vi::LayerStack id) NN_NOEXCEPT
    {
        if (!IsValidStack(id))
        {
            return nn::vi::ResultInvalidRange();
        }

        m_Stack.Reset(id);

        nn::Result result = UpdateLayerStack();

        if ( result.IsFailure() )
        {
            m_Stack.Set(id);
        }

        return result;
    }

    nn::Result Layer::SetStacks(std::uint32_t stacks) NN_NOEXCEPT
    {
        if (!IsValidStackMask(stacks))
        {
            return nn::vi::ResultInvalidRange();
        }

        auto prev = m_Stack;
        m_Stack._storage[0] = stacks;

        nn::Result result = UpdateLayerStack();

        if ( result.IsFailure() )
        {
            m_Stack = prev;
        }

        return result;
    }

    std::uint32_t Layer::GetStacks() const NN_NOEXCEPT
    {
        return m_Stack._storage[0];
    }

    float Layer::GetX() const NN_NOEXCEPT
    {
        return m_X;
    }

    float Layer::GetY() const NN_NOEXCEPT
    {
        return m_Y;
    }

    int Layer::GetZ() const NN_NOEXCEPT
    {
        return m_Z;
    }

    int Layer::GetCurrentWidth() const NN_NOEXCEPT
    {
        return m_CurrentWidth;
    }

    int Layer::GetCurrentHeight() const NN_NOEXCEPT
    {
        return m_CurrentHeight;
    }

    nn::vi::NativeWindowHandle Layer::GetNativeWindow() const NN_NOEXCEPT
    {
        return GetWindow();
    }

    Display* Layer::GetDisplay() const NN_NOEXCEPT
    {
        return m_pDisplay;
    }

    nn::vi::LayerSettings Layer::GetSettings() const NN_NOEXCEPT
    {
        return m_Settings;
    }

    ANativeWindow* Layer::GetWindow() const NN_NOEXCEPT
    {
        return static_cast<ANativeWindow*>(m_Control->getSurface().get());
    }

    bool Layer::IsValidStack(nn::vi::LayerStack id) NN_NOEXCEPT
    {
        return ((1u << id) & nn::vi::ValidLayerStackFlags) != 0;
    }

    bool Layer::IsValidStackMask(std::uint32_t stacks) NN_NOEXCEPT
    {
        return (stacks & nn::vi::ValidLayerStackFlags) == stacks;
    }

    nn::Result Layer::UpdateLayerStack() NN_NOEXCEPT
    {
        android::SurfaceComposerClient::openGlobalTransaction();
        android::status_t status = m_Control->setLayerStackMask(m_Stack._storage[0]);
        android::SurfaceComposerClient::closeGlobalTransaction();

        if ( status != android::OK )
        {
            return nn::vi::ResultOperationFailed();
        }

        return nn::ResultSuccess();
    }

    nn::Result Layer::SetStackUnsafe(nn::vi::LayerStackSet stacks, nn::vi::LayerStackSet mask) NN_NOEXCEPT
    {
        nn::vi::LayerStackSet newStacks = (m_Stack & ~mask) | (stacks & mask);
        android::status_t status = m_Control->setLayerStackMask(newStacks._storage[0]);

        if( status != android::OK )
        {
            return nn::vi::ResultOperationFailed();
        }

        m_Stack = newStacks;

        return nn::ResultSuccess();
    }

    android::status_t Layer::SetVisibilityUnsafe(bool isVisible) NN_NOEXCEPT
    {
        android::status_t status;

        if (isVisible)
        {
            status = m_Control->show();
        }
        else
        {
            status = m_Control->hide();
        }

        return status;
    }

    bool Layer::IsValidZOrder(int z) NN_NOEXCEPT
    {
        return m_pDisplay->GetInfo(m_PolicyLevel).ValidateZOrder(z);
    }

    bool Layer::IsValidStackSet(nn::vi::LayerStackSet stacks) NN_NOEXCEPT
    {
        NN_STATIC_ASSERT(sizeof(nn::vi::LayerStackSet) == sizeof(nn::vi::ValidLayerStackFlags));

        return (stacks._storage[0] & ~nn::vi::ValidLayerStackFlags) == 0;
    }
}}}
