﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "visrv_Display.h"
#include <new>
#include <nn/nn_SdkAssert.h>
#include <nn/vi/vi_Result.h>
#include "detail/visrv_IModeFilter.h"
#include "detail/visrv_Common.h"
#include "detail/visrv_IDisplayStrategy.h"
#include "detail/visrv_IPhysicalDisplay.h"

namespace nn{ namespace visrv{ namespace master{

    Display::Display(
        const detail::PlatformDisplayInfoSet& infoSet,
        Allocator* pAllocator,
        detail::IDisplayStrategy* pStrategy
        ) NN_NOEXCEPT

        : m_pStrategy(pStrategy)
        , m_InfoSet(infoSet)
        , m_pAllocator(pAllocator)
        , m_Lock(false)
        , m_OpenCount(0)
    {
        NN_SDK_ASSERT_NOT_NULL(pAllocator);
        NN_SDK_ASSERT_NOT_NULL(pStrategy);
    }

    Display::~Display() NN_NOEXCEPT
    {
        if( IsOpen() )
        {
            ForceClose();
        }
    }

    bool Display::IsOpen() const NN_NOEXCEPT
    {
        return m_OpenCount != 0;
    }

    nn::Result Display::Open() NN_NOEXCEPT
    {
        ++m_OpenCount;

        return nn::ResultSuccess();
    }

    void Display::Close() NN_NOEXCEPT
    {
        NN_SDK_ASSERT(m_OpenCount > 0);

        --m_OpenCount;

        if( !IsOpen() )
        {
            ForceClose();
        }
    }

    void Display::ForceClose() NN_NOEXCEPT
    {
        LayerList::iterator i = m_Layers.begin();

        while( i != m_Layers.end() )
        {
            LayerList::iterator temp = i++;
            temp->~Layer();
            m_pAllocator->Free(&*temp);
        }
    }

    nn::Result Display::CreateLayer(Layer** pOutLayer, nn::vi::LayerSettings settings, nn::vi::PolicyLevelType policyLevel, nn::applet::AppletResourceUserId aruid) NN_NOEXCEPT
    {
        *pOutLayer = nullptr;

        detail::Autolock lock(&m_Lock);

        auto info = GetInfo(policyLevel);
        if(info.GetName() == nullptr || info.GetName()[0] == '\0')
        {
            return nn::vi::ResultDenied();
        }

        // NOTE: Don't check layer count here. All layers are created by vi process.
        //
        //if( m_Info.GetPolicy().HasLayerLimit() &&
        //    m_Layers.size() >= m_Info.GetPolicy().GetLayerCountMax() )
        //{
        //    return ResultDenied();
        //}

        void* block = reinterpret_cast<Layer*>(m_pAllocator->Allocate());
        NN_SDK_ASSERT_NOT_NULL(block);

        Layer* pLayer = new (block) Layer(this, policyLevel);

        nn::Result result = pLayer->Initialize(settings, aruid);
        if( result.IsSuccess() )
        {
            m_Layers.push_back(*pLayer);
            *pOutLayer = pLayer;
        }
        else
        {
            pLayer->~Layer();
            m_pAllocator->Free(pLayer);
        }

        return result;
    }

    const detail::PlatformDisplayInfo& Display::GetInfo(nn::vi::PolicyLevelType policyLevel) const NN_NOEXCEPT
    {
        NN_SDK_ASSERT_RANGE(policyLevel, 0, nn::vi::PolicyLevelCount);
        return m_InfoSet.GetInfo(policyLevel);
    }

    nn::Result Display::SetEnabled(bool isEnabled) NN_NOEXCEPT
    {
        nn::Result result = nn::ResultSuccess();

        detail::Autolock lock(&m_Lock);

        for( LayerList::iterator i = m_Layers.begin(); i != m_Layers.end(); ++i )
        {
            // TODO: investigate performance of opening transaction outside this loop
            result = i->SetVisibility(isEnabled);

            if( result.IsFailure() )
            {
                break;
            }
        }

        return result;
    }

    void Display::DestroyLayer(Layer* pLayer) NN_NOEXCEPT
    {
        detail::Autolock lock(&m_Lock);

        m_Layers.erase(m_Layers.iterator_to(*pLayer));
        pLayer->~Layer();
        m_pAllocator->Free(pLayer);
    }

    int Display::ListModes(nn::vi::DisplayModeInfo* pOutModes, int modeCountMax, nn::vi::PolicyLevelType policyLevel) const NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_RANGE(policyLevel, 0, nn::vi::PolicyLevelCount);
        NN_SDK_ASSERT_NOT_NULL(pOutModes);
        NN_SDK_ASSERT_GREATER(modeCountMax, 0);

        NN_UNUSED(policyLevel);

        return m_pStrategy->GetDisplay()->ListModes(pOutModes, modeCountMax);
    }

    nn::Result Display::SetMode(const nn::vi::DisplayModeInfo& mode, nn::vi::PolicyLevelType policyLevel) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_RANGE(policyLevel, 0, nn::vi::PolicyLevelCount);

        NN_UNUSED(policyLevel);

        return m_pStrategy->GetDisplay()->SetMode(mode);
    }

    detail::Autolock Display::GetLayerListLock() const NN_NOEXCEPT
    {
        return detail::Autolock(&m_Lock);
    }

    const Display::LayerList& Display::GetLayerList() const NN_NOEXCEPT
    {
        return m_Layers;
    }

    void Display::SetContentVisibilityUnsafe(bool isVisible) NN_NOEXCEPT
    {
        detail::Autolock lock(GetLayerListLock());

        for( Layer& layer : m_Layers )
        {
            layer.SetVisibilityUnsafe(isVisible);
        }
    }

    nn::Result Display::GetHotplugState(nn::vi::HotplugStateType* pOutState) const NN_NOEXCEPT
    {
        NN_SDK_ASSERT_NOT_NULL(pOutState);
        return m_pStrategy->GetDisplay()->GetHotplugState(pOutState);
    }

    nn::Result Display::GetMode(nn::vi::DisplayModeInfo* pOutMode) const NN_NOEXCEPT
    {
        NN_SDK_ASSERT_NOT_NULL(pOutMode);
        return m_pStrategy->GetDisplay()->GetMode(pOutMode);
    }

    int Display::ListRgbRanges(nn::vi::RgbRangeType* pOutRanges, int rgbRangeCountMax) const NN_NOEXCEPT
    {
        NN_SDK_ASSERT_NOT_NULL(pOutRanges);
        NN_SDK_ASSERT_GREATER(rgbRangeCountMax, 0);
        return m_pStrategy->GetDisplay()->ListRgbRanges(pOutRanges, rgbRangeCountMax);
    }

    nn::Result Display::GetRgbRange(nn::vi::RgbRangeType* pOutRange) const NN_NOEXCEPT
    {
        NN_SDK_ASSERT_NOT_NULL(pOutRange);
        return m_pStrategy->GetDisplay()->GetRgbRange(pOutRange);
    }

    nn::Result Display::SetRgbRange(nn::vi::RgbRange range) NN_NOEXCEPT
    {
        return m_pStrategy->GetDisplay()->SetRgbRange(range);
    }

    nn::Result Display::SetUnderscan(int underscan) NN_NOEXCEPT
    {
        return m_pStrategy->GetDisplay()->SetUnderscan(underscan);
    }

    nn::Result Display::GetUnderscan(int* pOutUnderscan) const NN_NOEXCEPT
    {
        NN_SDK_ASSERT_NOT_NULL(pOutUnderscan);
        return m_pStrategy->GetDisplay()->GetUnderscan(pOutUnderscan);
    }

    nn::Result Display::SetAlpha(float alpha) NN_NOEXCEPT
    {
        return m_pStrategy->GetDisplay()->SetAlpha(alpha);
    }

    nn::Result Display::SetPowerState(nn::vi::PowerState state) NN_NOEXCEPT
    {
        return m_pStrategy->GetDisplay()->SetPowerState(state);
    }

    nn::Result Display::SetLayerStack(nn::vi::LayerStackType id) NN_NOEXCEPT
    {
        return m_pStrategy->GetDisplay()->SetLayerStack(id);
    }

    nn::Result Display::GetLayerStack(nn::vi::LayerStackType* pOutId) const NN_NOEXCEPT
    {
        return m_pStrategy->GetDisplay()->GetLayerStack(pOutId);
    }

    nn::Result Display::SetCmuLuma(float luma) NN_NOEXCEPT
    {
        return m_pStrategy->GetDisplay()->SetCmuLuma(luma);
    }

    nn::Result Display::GetCmuLuma(float* pOutLuma) const NN_NOEXCEPT
    {
        NN_SDK_ASSERT_NOT_NULL(pOutLuma);
        return m_pStrategy->GetDisplay()->GetCmuLuma(pOutLuma);
    }

    nn::Result Display::GetLogicalResolution(int* pOutWidth, int* pOutHeight) const NN_NOEXCEPT
    {
        NN_SDK_ASSERT_NOT_NULL(pOutWidth);
        NN_SDK_ASSERT_NOT_NULL(pOutHeight);
        return m_pStrategy->GetDisplay()->GetLogicalResolution(pOutWidth, pOutHeight);
    }

    bool Display::IsHotplugEventSupported() const NN_NOEXCEPT
    {
        return m_pStrategy->GetDisplay()->IsHotplugEventSupported();
    }

    bool Display::IsVsyncEventSupported() const NN_NOEXCEPT
    {
        return m_pStrategy->GetDisplay()->IsVsyncEventSupported();
    }

    bool Display::IsModeChangedEventSupported() const NN_NOEXCEPT
    {
        return m_pStrategy->GetDisplay()->IsModeChangedEventSupported();
    }

    nn::Result Display::GetCmuMode(nn::vi::CmuModeType* pOutMode) const NN_NOEXCEPT
    {
        NN_SDK_ASSERT_NOT_NULL(pOutMode);
        return m_pStrategy->GetDisplay()->GetCmuMode(pOutMode);
    }

    nn::Result Display::SetCmuMode(nn::vi::CmuMode mode) NN_NOEXCEPT
    {
        return m_pStrategy->GetDisplay()->SetCmuMode(mode);
    }

    nn::Result Display::GetContrastRatio(float* pOutRatio) const NN_NOEXCEPT
    {
        NN_SDK_ASSERT_NOT_NULL(pOutRatio);
        return m_pStrategy->GetDisplay()->GetContrastRatio(pOutRatio);
    }

    nn::Result Display::SetContrastRatio(float ratio) NN_NOEXCEPT
    {
        NN_SDK_ASSERT_GREATER_EQUAL(ratio, 0.f);
        NN_SDK_ASSERT_LESS_EQUAL(ratio, 1.f);
        return m_pStrategy->GetDisplay()->SetContrastRatio(ratio);
    }

    nn::Result Display::GetGamma(float* pOutGamma) const NN_NOEXCEPT
    {
        NN_SDK_ASSERT_NOT_NULL(pOutGamma);
        return m_pStrategy->GetDisplay()->GetGamma(pOutGamma);
    }

    nn::Result Display::SetGamma(float gamma) NN_NOEXCEPT
    {
        NN_SDK_ASSERT_GREATER_EQUAL(gamma, 1.f);
        NN_SDK_ASSERT_LESS_EQUAL(gamma, 3.f);
        return m_pStrategy->GetDisplay()->SetGamma(gamma);
    }

    nn::Result Display::SetViewport(int x, int y, int width, int height) NN_NOEXCEPT
    {
        NN_SDK_ASSERT_GREATER_EQUAL(x, 0);
        NN_SDK_ASSERT_GREATER_EQUAL(y, 0);
        NN_SDK_ASSERT_GREATER(width, 0);
        NN_SDK_ASSERT_GREATER(height, 0);

        return m_pStrategy->GetDisplay()->SetViewport(x, y, width, height);
    }
    nn::Result Display::GetCompositorErrorInfo(nn::vi::CompositorError* pOutErrorInfo, int* pOutLength, int errorID) NN_NOEXCEPT
    {
        NN_SDK_ASSERT_NOT_NULL(pOutErrorInfo);
        return m_pStrategy->GetDisplay()->GetCompositorErrorInfo(pOutErrorInfo, pOutLength, errorID);
    }

}}}
