﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "visrv_EdidDisplay.h"

#include <nn/vi/vi_Result.h>
#include <nn/vi/vi_DisplayModeInfoInternal.h>
#include "visrv_Edid.h"
#include "visrv_IModeFilter.h"

namespace nn{ namespace visrv{ namespace master{ namespace detail{

    EdidDisplay::EdidDisplay(const IModeFilter* pFilter) NN_NOEXCEPT
        : IPhysicalDisplay(pFilter)
        , m_Handle{ nullptr, 0 }
    {
        memset(&m_Data, 0, sizeof(m_Data));
    }

    nn::Result EdidDisplay::Open() NN_NOEXCEPT
    {
        // make sure there's no padding, otherwise this trick won't work
        NN_STATIC_ASSERT(sizeof(nn::settings::system::Edid) == 256);

        nn::settings::system::GetEdid(&m_Data);

        if( nn::edid::OpenEdid(&m_Handle, &m_Data, sizeof(nn::settings::system::Edid)) == nn::edid::Error_None )
        {
            return nn::ResultSuccess();
        }

        return nn::vi::ResultOperationFailed();
    }

    void EdidDisplay::Close() NN_NOEXCEPT
    {
        nn::edid::CloseEdid(&m_Handle);
    }

    nn::Result EdidDisplay::GetHotplugState(nn::vi::HotplugStateType* pOutState) const NN_NOEXCEPT
    {
        // this isn't a real display
        return nn::vi::ResultNotSupported();
    }

    nn::Result EdidDisplay::GetMode(nn::vi::DisplayModeInfo* pOutMode) const NN_NOEXCEPT
    {
        // this isn't a real display
        return nn::vi::ResultOperationFailed();
    }

    int EdidDisplay::ListModes(nn::vi::DisplayModeInfo* pOutModes, int modeCountMax) const NN_NOEXCEPT
    {
        DisplayModeVisitorData data = { 0, pFilter, pOutModes, modeCountMax };

        nn::edid::VisitDisplayModes(&m_Handle, DisplayModeVisitor, &data);

        return data.writeCount;
    }

    nn::Result EdidDisplay::SetMode(const nn::vi::DisplayModeInfo& mode) NN_NOEXCEPT
    {
        NN_UNUSED(mode);

        return nn::vi::ResultNotSupported();
    }

    bool EdidDisplay::DisplayModeVisitor(const nn::edid::DisplayModeInfo* pMode,
                                         const nn::edid::DisplayTimingInfo* pTimingInfo,
                                         const nn::edid::ImageSizeInfo* pSizeInfo,
                                         void* pUserData) NN_NOEXCEPT
    {
        DisplayModeVisitorData* pData = static_cast<DisplayModeVisitorData*>(pUserData);

        if( pData->writeCount >= pData->modeCountMax )
        {
            return false;
        }

        // SIGLONTD-4999:  Interlaced isn't supported on NX.  It would be better to add this to the filter,
        //                 but it's a riskier change since nn::vi::DisplayModeInfo doesn't currently have a
        //                 flag to indicate interlaced.
        if( pMode->isInterlaced )
        {
            return false;
        }

        nn::vi::DisplayModeInfoInternal mode;
        if( detail::ConvertDisplayMode(&mode, pMode) && pData->pFilter->IsValid(mode) )
        {
            bool isDuplicate = false;

            for( int i = 0; i < pData->writeCount; ++i )
            {
                if( pData->pOutModes[i].width == mode.clientInfo.width &&
                    pData->pOutModes[i].height == mode.clientInfo.height &&
                    pData->pOutModes[i].refreshRate == mode.clientInfo.refreshRate &&
                    pData->pOutModes[i].mode == mode.clientInfo.mode )
                {
                    isDuplicate = true;
                    break;
                }
            }

            if( !isDuplicate )
            {
                pData->pOutModes[pData->writeCount] = mode.clientInfo;
                ++pData->writeCount;
            }
        }

        return true;
    }

    int EdidDisplay::ListRgbRanges(nn::vi::RgbRangeType* pOutRanges, int rgbCountMax) const NN_NOEXCEPT
    {
        nn::edid::DisplayInfo info;
        nn::edid::GetDisplayInfo(&info, &m_Handle);

        return WriteCeFormatRgbRange(pOutRanges, rgbCountMax, info.isRgbQuantizationSelectable);
    }

    nn::Result EdidDisplay::GetRgbRange(nn::vi::RgbRangeType* pOutRange) const NN_NOEXCEPT
    {
        NN_UNUSED(pOutRange);

        return nn::vi::ResultOperationFailed();
    }

    nn::Result EdidDisplay::SetRgbRange(nn::vi::RgbRange range) NN_NOEXCEPT
    {
        NN_UNUSED(range);

        return nn::vi::ResultNotSupported();
    }

    nn::Result EdidDisplay::SetUnderscan(int underscan) NN_NOEXCEPT
    {
        NN_UNUSED(underscan);

        return nn::vi::ResultNotSupported();
    }

    nn::Result EdidDisplay::GetUnderscan(int* pOutUnderscan) const NN_NOEXCEPT
    {
        NN_UNUSED(pOutUnderscan);

        return nn::vi::ResultNotSupported();
    }

    nn::Result EdidDisplay::SetAlpha(float alpha) NN_NOEXCEPT
    {
        NN_UNUSED(alpha);

        return nn::vi::ResultNotSupported();
    }

    nn::Result EdidDisplay::GetPowerState(nn::vi::PowerStateType* pOutState) NN_NOEXCEPT
    {
        NN_UNUSED(pOutState);

        return nn::vi::ResultNotSupported();
    }

    nn::Result EdidDisplay::SetPowerState(nn::vi::PowerState state) NN_NOEXCEPT
    {
        NN_UNUSED(state);

        return nn::vi::ResultNotSupported();
    }

    nn::Result EdidDisplay::SetLayerStack(nn::vi::LayerStackType id) NN_NOEXCEPT
    {
        NN_UNUSED(id);

        return nn::vi::ResultNotSupported();
    }

    nn::Result EdidDisplay::GetLayerStack(nn::vi::LayerStackType* pOutId) const NN_NOEXCEPT
    {
        NN_UNUSED(pOutId);

        return nn::vi::ResultNotSupported();
    }

    nn::Result EdidDisplay::SetCmuLuma(float luma) NN_NOEXCEPT
    {
        NN_UNUSED(luma);

        return nn::vi::ResultNotSupported();
    }

    nn::Result EdidDisplay::GetCmuLuma(float* pOutLuma) const NN_NOEXCEPT
    {
        NN_UNUSED(pOutLuma);

        return nn::vi::ResultNotSupported();
    }

    bool EdidDisplay::IsHotplugEventSupported() const NN_NOEXCEPT
    {
        return false;
    }

    bool EdidDisplay::IsVsyncEventSupported() const NN_NOEXCEPT
    {
        return false;
    }

    bool EdidDisplay::IsModeChangedEventSupported() const NN_NOEXCEPT
    {
        return false;
    }

    nn::Result EdidDisplay::GetLogicalResolution(int* pOutWidth, int* pOutHeight) const NN_NOEXCEPT
    {
        NN_UNUSED(pOutWidth);
        NN_UNUSED(pOutHeight);

        return nn::vi::ResultNotSupported();
    }

    nn::Result EdidDisplay::GetCmuMode(nn::vi::CmuModeType* pOutMode) const NN_NOEXCEPT
    {
        NN_UNUSED(pOutMode);

        return nn::vi::ResultNotSupported();
    }

    nn::Result EdidDisplay::SetCmuMode(nn::vi::CmuMode mode) NN_NOEXCEPT
    {
        NN_UNUSED(mode);

        return nn::vi::ResultNotSupported();
    }

    nn::Result EdidDisplay::GetContrastRatio(float* pOutRatio) const NN_NOEXCEPT
    {
        NN_UNUSED(pOutRatio);

        return nn::vi::ResultNotSupported();
    }

    nn::Result EdidDisplay::SetContrastRatio(float ratio) NN_NOEXCEPT
    {
        NN_UNUSED(ratio);

        return nn::vi::ResultNotSupported();
    }

    nn::Result EdidDisplay::GetGamma(float* pOutGamma) const NN_NOEXCEPT
    {
        NN_UNUSED(pOutGamma);

        return nn::vi::ResultNotSupported();
    }

    nn::Result EdidDisplay::SetGamma(float gamma) NN_NOEXCEPT
    {
        NN_UNUSED(gamma);

        return nn::vi::ResultNotSupported();
    }

    nn::Result EdidDisplay::SetViewport(int x, int y, int width, int height) NN_NOEXCEPT
    {
        NN_UNUSED(x);
        NN_UNUSED(y);
        NN_UNUSED(width);
        NN_UNUSED(height);

        return nn::vi::ResultNotSupported();
    }

    nn::Result EdidDisplay::GetCompositorErrorInfo(nn::vi::CompositorError* pOutErrorInfo, int* pOutLength, int errorID) NN_NOEXCEPT
    {
        NN_UNUSED(pOutErrorInfo);
        NN_UNUSED(pOutLength);
        NN_UNUSED(errorID);

        return nn::vi::ResultNotSupported();
    }
}}}}
