﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cmath>
#include <nn/vi/vi_DisplayModeInfoInternal.h>
#include "visrv_Edid.h"

namespace
{
    bool AreAboutEqual(float f, float g) NN_NOEXCEPT
    {
        return std::abs(f - g) <= 0.001f;
    }
}

namespace nn{ namespace visrv{ namespace master{ namespace detail{

    bool ConvertAspectRatio(nn::vi::AspectRatio* pOutRatio, float ratio) NN_NOEXCEPT
    {
        NN_SDK_ASSERT_NOT_NULL(pOutRatio);

        if( AreAboutEqual(ratio, 4.f / 3.f) )
        {
            *pOutRatio = nn::vi::AspectRatio_4_3;
            return true;
        }

        if( AreAboutEqual(ratio, 16.f / 9.f) )
        {
            *pOutRatio = nn::vi::AspectRatio_16_9;
            return true;
        }

        if( AreAboutEqual(ratio, 64.f / 27.f) )
        {
            *pOutRatio = nn::vi::AspectRatio_64_27;
            return true;
        }

        if( AreAboutEqual(ratio, 256.f / 135.f) )
        {
            *pOutRatio = nn::vi::AspectRatio_256_135;
            return true;
        }

        return false;
    }

    bool ConvertStereoMode(nn::vi::StereoModeType* pOutMode, nn::edid::StereoMode mode) NN_NOEXCEPT
    {
        NN_SDK_ASSERT_NOT_NULL(pOutMode);

        switch( mode )
        {
        case nn::edid::StereoMode_None:
            *pOutMode = nn::vi::StereoMode_None;
            return true;
        case nn::edid::StereoMode_FramePacking:
            *pOutMode = nn::vi::StereoMode_FramePacking;
            return true;
        case nn::edid::StereoMode_TopAndBottom:
            *pOutMode = nn::vi::StereoMode_TopAndBottom;
            return true;
        case nn::edid::StereoMode_SideBySide:
            *pOutMode = nn::vi::StereoMode_SideBySide;
            return true;
        default:
            // not everything has a mapping in nn::vi
            return false;
        }
    }

    bool ConvertDisplayMode(nn::vi::DisplayModeInfoInternal* pOutMode, const nn::edid::DisplayModeInfo* pMode) NN_NOEXCEPT
    {
        NN_SDK_ASSERT_NOT_NULL(pOutMode);
        NN_SDK_ASSERT_NOT_NULL(pMode);

        pOutMode->clientInfo.width = pMode->width;
        pOutMode->clientInfo.height = pMode->height;
        pOutMode->clientInfo.refreshRate = pMode->refreshRate;

        return ConvertAspectRatio(&pOutMode->ratio, pMode->imageAspectRatio) &&
               ConvertStereoMode(&pOutMode->clientInfo.mode, pMode->stereoMode);
    }

}}}}
