﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "visrv_DisplayManager.h"

#include <cstring>
#include <nn/vi/vi_Result.h>
#include "../../visrv_Log.h"
#include "../../settings/visrv_Settings.h"

namespace nn{ namespace visrv{ namespace master{ namespace detail{

    DisplayManager::DisplayManager(
        void* displaySegmentStartAddress, size_t displaySegmentSize,
        void* layerSegmentStartAddress, size_t layerSegmentSize,
        PlatformDisplayInfo** platformDisplayListList,
        int* platformDisplayCountList,
        int platformDisplayCount
        ) NN_NOEXCEPT
        : m_LayerAllocator(layerSegmentStartAddress, layerSegmentSize, NN_ALIGNOF(Layer))
        , m_DisplayFactory(displaySegmentStartAddress, displaySegmentSize, settings::GetPlatformConfig())
        , m_Lock(false)
    {
        NN_SDK_ASSERT_EQUAL(platformDisplayCount, nn::vi::PolicyLevelCount);
        for(int p = 0; p < nn::vi::PolicyLevelCount; p++)
        {
            m_PlatformDisplayCount[p] = platformDisplayCountList[p];
            m_PlatformDisplays[p] = platformDisplayListList[p];
        }
    }

    DisplayManager::~DisplayManager() NN_NOEXCEPT
    {
        DisplayList::iterator i = m_OpenDisplays.begin();

        while( i != m_OpenDisplays.end() )
        {
            DisplayList::iterator temp = i++;
            // not pretty, but need the pointer...
            m_DisplayFactory.Destroy(&*temp);
        }
    }

    Display* DisplayManager::Open(const char* name, nn::vi::PolicyLevelType policyLevel) NN_NOEXCEPT
    {
        Autolock lock(&m_Lock);

        if(name == nullptr || name[0] == '\0')
        {
            return nullptr;
        }

        // check to see if the display is already open by the process
        for( DisplayList::iterator i = m_OpenDisplays.begin(); i != m_OpenDisplays.end(); ++i )
        {
            if( strncmp(name, i->GetInfo(policyLevel).GetName(), nn::vi::DisplayInfo::NameLengthMax) == 0 )
            {
                i->Open();
                return &*i;
            }
        }

        PlatformDisplayInfoSet displaySet = {};

        // check in named platform displays first
        for(nn::vi::PolicyLevelType p = 0; p < nn::vi::PolicyLevelCount; p++)
        {
            for(int i = 0; i < m_PlatformDisplayCount[p]; i++)
            {
                if( strncmp(name, m_PlatformDisplays[p][i].GetName(), nn::vi::DisplayInfo::NameLengthMax) == 0 )
                {
                    displaySet.SetInfo(p, m_PlatformDisplays[p][i]);
                    break;
                }
            }
        }

        // create if found
        if(displaySet.GetInfo(policyLevel).HasValidName())
        {
            Display* pDisplay = m_DisplayFactory.Create(displaySet, &m_LayerAllocator, name);

            if( pDisplay != nullptr )
            {
                if( pDisplay->Open().IsFailure() )
                {
                    m_DisplayFactory.Destroy(pDisplay);
                    return nullptr;
                }

                m_OpenDisplays.push_back(*pDisplay);
            }

            return pDisplay;
        }

        // TODO: enumerate other displays (useful for PC)

        return nullptr;
    }

    void DisplayManager::Close(Display* pDisplay) NN_NOEXCEPT
    {
        Autolock lock(&m_Lock);

        pDisplay->Close();

        if( !pDisplay->IsOpen() )
        {
            m_OpenDisplays.erase(m_OpenDisplays.iterator_to(*pDisplay));
            m_DisplayFactory.Destroy(pDisplay);
        }
    }

    Autolock DisplayManager::GetDisplayListLock() const NN_NOEXCEPT
    {
        return Autolock(&m_Lock);
    }

    const DisplayManager::DisplayList& DisplayManager::GetDisplayList() const NN_NOEXCEPT
    {
        return m_OpenDisplays;
    }

    void DisplayManager::SetContentVisibility(bool isVisible) NN_NOEXCEPT
    {
        Autolock lock(GetDisplayListLock());

        android::SurfaceComposerClient::openGlobalTransaction();

        for( Display& display : m_OpenDisplays )
        {
            display.SetContentVisibilityUnsafe(isVisible);
        }

        android::SurfaceComposerClient::closeGlobalTransaction();
    }

}}}}
