﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "visrv_DisplayFactory.h"

#include <cstring>
#include <nn/vi/vi_Types.h>
#include "visrv_PlatformDisplayDefines.h"
#include "visrv_DefaultDisplay-spec.NX.h"
#include "../../settings/visrv_Settings.h"
#include "../../settings/visrv_PlatformConfig.h"
#include "../../settings/visrv_Strategy.h"
#include "../../visrv_Log.h"

namespace nn{ namespace visrv{ namespace master{ namespace detail{

    DisplayFactory::DisplayFactory(void* startAddress, size_t size, const settings::PlatformConfig* m_pConfig) NN_NOEXCEPT
        : m_Allocator(startAddress, size)
        , m_Controllers{ Controller(&m_Allocator), Controller(&m_Allocator) }
        , m_NullDisplayStrategy(&m_NullDisplay)
        , m_pConfig(m_pConfig)
    {
        NN_ABORT_UNLESS_RESULT_SUCCESS(m_NullDisplay.Open());
        NN_ABORT_UNLESS_NOT_NULL(m_pConfig);

        NN_STATIC_ASSERT(sizeof(m_Controllers) / sizeof(m_Controllers[0]) == sizeof(m_pConfig->dcList) / sizeof(m_pConfig->dcList[0]));
        for( int i = 0; i < sizeof(m_Controllers) / sizeof(m_Controllers[0]); ++i )
        {
            const settings::DisplayController& dc = m_pConfig->dcList[i];
            m_Controllers[i].Initialize(dc.interface, dc.modes, dc.modeCount);
        }
    }

    DisplayFactory::~DisplayFactory() NN_NOEXCEPT
    {
        m_NullDisplay.Close();
    }

    Display* DisplayFactory::Create(const PlatformDisplayInfoSet& infoSet, Display::Allocator* pAllocator, const char* name) NN_NOEXCEPT
    {
        if( std::strncmp(DefaultDisplayName, name, nn::vi::DisplayInfo::NameLengthMax) == 0 )
        {
            IDisplayStrategy* pStrategy = nullptr;

            switch( m_pConfig->defaultDisplayStrategy )
            {
            case settings::Strategy::HighestPriority:
            {
                const settings::DisplayController* pMax = m_pConfig->dcList;
                for( int i = 1; i < sizeof(m_pConfig->dcList) / sizeof(m_pConfig->dcList[0]); ++i )
                {
                    if( m_pConfig->dcList[i].priority > pMax->priority )
                    {
                        pMax = m_pConfig->dcList + i;
                    }
                }

                NN_STATIC_ASSERT(sizeof(m_Controllers) / sizeof(m_Controllers[0]) == sizeof(m_pConfig->dcList) / sizeof(m_pConfig->dcList[0]));

                if( pMax->priority >= 0 )
                {
                    m_DefaultHighestPriority.Initialize(m_Controllers[pMax - m_pConfig->dcList].GetDisplay());
                }
                else
                {
                    NN_VISRV_LOG_WARN("Using null display as default.\n");
                    m_DefaultHighestPriority.Initialize(&m_NullDisplay);
                }

                pStrategy = &m_DefaultHighestPriority;
            }
            break;

            case settings::Strategy::PriorityFallback:
            {
                IPhysicalDisplay* displays[sizeof(m_pConfig->dcList) / sizeof(m_pConfig->dcList[0])];
                std::memset(displays, 0, sizeof(displays));

                struct DcInfo
                {
                    int id;
                    const settings::DisplayController* dc;
                };

                DcInfo sorted[sizeof(m_pConfig->dcList) / sizeof(m_pConfig->dcList[0])];
                for( int i = 0; i < sizeof(sorted) / sizeof(sorted[0]); ++i )
                {
                    sorted[i].id = i;
                    sorted[i].dc = m_pConfig->dcList + i;
                }

                std::sort(sorted,
                          sorted + sizeof(sorted) / sizeof(sorted[0]),
                          [](const DcInfo& lhs, const DcInfo& rhs)
                          {
                              return rhs.dc->priority < lhs.dc->priority;
                          });

                int displayCount = 0;
                for( int i = 0; i < sizeof(sorted) / sizeof(sorted[0]); ++i )
                {
                    if( sorted[i].dc->priority >= 0 )
                    {
                        NN_SDK_ASSERT_GREATER_EQUAL(sorted[i].id, 0);
                        NN_SDK_ASSERT_LESS(sorted[i].id, sizeof(displays) / sizeof(displays[0]));

                        displays[i] = m_Controllers[sorted[i].id].GetDisplay();
                        ++displayCount;
                    }
                }

                if( displayCount != 0 )
                {
                    m_DefaultPriorityFallback.Initialize(displays, displayCount);
                    pStrategy = &m_DefaultPriorityFallback;
                }
                else
                {
                    NN_VISRV_LOG_WARN("Using null display as default.\n");
                    m_DefaultHighestPriority.Initialize(&m_NullDisplay);
                    pStrategy = &m_DefaultHighestPriority;
                }
            }
            break;
            default:
                NN_UNEXPECTED_DEFAULT;
            }

            return new (AllocateDisplay<DefaultDisplay>()) DefaultDisplay(infoSet, pAllocator, pStrategy);
        }

        for( int i = 0; i < sizeof(m_pConfig->displays) / sizeof(m_pConfig->displays[0]); ++i )
        {
            const settings::PhysicalDisplay* pSettings = m_pConfig->displays + i;
            if( std::strncmp(pSettings->name, name, nn::vi::DisplayInfo::NameLengthMax) == 0 )
            {
                return new (AllocateDisplay<Display>()) Display(infoSet, pAllocator, m_Controllers[pSettings->dc].GetStrategy());
            }
            else
            {
                if( std::strlen(pSettings->edidDisplayName) != 0 && std::strncmp(pSettings->edidDisplayName, name, nn::vi::DisplayInfo::NameLengthMax) == 0 )
                {
                    // TODO: EDID display should be created on a per-process basis
                    // Displays are no longer responsible for opening the IPhysicalDisplay, so it must be opened here
                    if( m_Controllers[pSettings->dc].GetEdid()->Open().IsFailure() )
                    {
                        return nullptr;
                    }

                    return new (AllocateDisplay<Display>()) Display(infoSet, pAllocator, m_Controllers[pSettings->dc].GetEdidStrategy());
                }
            }
        }

        if( std::strncmp(NullDisplayName, name, nn::vi::DisplayInfo::NameLengthMax) == 0 )
        {
            return new (AllocateDisplay<Display>()) Display(infoSet, pAllocator, &m_NullDisplayStrategy);
        }

        return nullptr;
    } // NOLINT(impl/function_size)

    void DisplayFactory::Destroy(Display* pDisplay) NN_NOEXCEPT
    {
        NN_SDK_ASSERT_NOT_NULL(pDisplay);
        pDisplay->~Display();
        m_Allocator.Free(pDisplay);
    }

}}}}
