﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "visrv_AndroidDisplay.h"

#include <nn/vi/vi_Result.h>
#include <nn/vi/vi_ResultPrivate.h>
#include <nn/vi/vi_DisplayModeInfo.h>
#include <nn/vi/vi_DisplayModeInfoInternal.h>
#include <nn/vi/vi_CmuMode.h>
#include <nn/vi/vi_LayerStack.h>
#include <nn/settings/system/settings_Tv.h>
#include "visrv_IModeFilter.h"
#include "../../native/visrv_GetSurfaceComposerClient.h"
#include "../../visrv_Log.h"

namespace nn{ namespace visrv{ namespace master{ namespace detail{

    AndroidDisplay::AndroidDisplay(const IModeFilter* pFilter, int id, nn::vi::LayerStack stack) NN_NOEXCEPT
        : IPhysicalDisplay(pFilter)
        , m_Id(id)
        , m_Client(native::GetSurfaceComposerClient())
        , m_BrightnessMax(0.f)
        , m_PowerState(nn::vi::PowerState_Off)
        , m_Stack(stack)
    {
        switch( id )
        {
        case android::ISurfaceComposer::eDisplayIdHdmi:
        case android::ISurfaceComposer::eDisplayIdMain:
        case android::ISurfaceComposer::eDisplayIdNull:
            break;
        default:
            NN_UNEXPECTED_DEFAULT;
        };
    }

    AndroidDisplay::~AndroidDisplay() NN_NOEXCEPT
    {
        Close();
    }

    nn::Result AndroidDisplay::Open() NN_NOEXCEPT
    {
        m_Display = android::SurfaceComposerClient::getBuiltInDisplay(m_Id);

        if( m_Display == nullptr )
        {
            return nn::vi::ResultOperationFailed();
        }

        return nn::ResultSuccess();
    }

    void AndroidDisplay::Close() NN_NOEXCEPT
    {
        // necessary for reference counting
        m_Display = nullptr;
    }

    nn::Result AndroidDisplay::GetMode(nn::vi::DisplayModeInfo* pOutMode) const NN_NOEXCEPT
    {
        NN_SDK_ASSERT_NOT_NULL(m_Display);

        // only return modes if a display is connected
        // DC might be in prepare state and have a fake mode
        nn::vi::HotplugStateType hotplug;
        if( m_PowerState == nn::vi::PowerState_On &&
            GetHotplugState(&hotplug).IsSuccess() && hotplug == nn::vi::HotplugState_Connected )
        {
            android::DisplayInfo info;
            if( android::SurfaceComposerClient::getDisplayInfo(m_Display, &info) == android::OK )
            {
                ConvertDisplayMode(pOutMode, info);
                return nn::ResultSuccess();
            }
        }

        return nn::vi::ResultDisplayDisconnected();
    }

    int AndroidDisplay::ListModes(nn::vi::DisplayModeInfo* pOutModes, int modeCountMax) const NN_NOEXCEPT
    {
        NN_SDK_ASSERT_NOT_NULL(m_Display);
        NN_SDK_ASSERT_NOT_NULL(pOutModes);
        NN_SDK_ASSERT_GREATER(modeCountMax, 0);
        NN_SDK_ASSERT_NOT_NULL(pFilter);

        // only return modes if a display is connected
        // DC might be in prepare state and have a fake mode
        nn::vi::HotplugStateType hotplug;
        if( m_PowerState == nn::vi::PowerState_On &&
            GetHotplugState(&hotplug).IsSuccess() && hotplug == nn::vi::HotplugState_Connected )
        {
            android::Vector<android::DisplayInfo> configs;

            if( android::SurfaceComposerClient::getDisplayConfigs(m_Display, &configs) != 0 )
            {
                return 0;
            }

            int writeIndex = 0;
            for( android::DisplayInfo info : configs )
            {
                if( writeIndex >= modeCountMax )
                {
                    break;
                }

                nn::vi::DisplayModeInfoInternal mode;
                ConvertDisplayMode(&mode, info);

                if( pFilter->IsValid(mode) )
                {
                    bool isDuplicate = false;

                    for( int i = 0; i < writeIndex; ++i )
                    {
                        if( pOutModes[i].width == mode.clientInfo.width &&
                            pOutModes[i].height == mode.clientInfo.height &&
                            pOutModes[i].refreshRate == mode.clientInfo.refreshRate &&
                            pOutModes[i].mode == mode.clientInfo.mode )
                        {
                            isDuplicate = true;
                            break;
                        }
                    }

                    if( !isDuplicate )
                    {
                        pOutModes[writeIndex] = mode.clientInfo;
                        ++writeIndex;
                    }
                }
            }

            return writeIndex;
        }

        return 0;
    }

    nn::Result AndroidDisplay::SetMode(const nn::vi::DisplayModeInfo& mode) NN_NOEXCEPT
    {
        nn::vi::HotplugStateType hotplug;
        if( m_PowerState == nn::vi::PowerState_On &&
            GetHotplugState(&hotplug).IsSuccess() && hotplug == nn::vi::HotplugState_Connected )
        {
            android::Vector<android::DisplayInfo> configs;

            if( android::SurfaceComposerClient::getDisplayConfigs(m_Display, &configs) != 0 ||
                configs.size() == 0 )
            {
                return nn::vi::ResultDisplayDisconnected();
            }

            int index = -1;

            for( int i = 0; i < configs.size(); ++i )
            {
                nn::vi::DisplayModeInfoInternal available;
                if( ConvertDisplayMode(&available, configs[i]) && pFilter->IsValid(available) )
                {
                    if( mode == configs[i] )
                    {
                        index = i;
                        break;
                    }
                }
            }

            if( index == -1 )
            {
                return nn::vi::ResultNotSupported();
            }

            // TODO: double check error codes
            if( android::SurfaceComposerClient::setActiveConfig(m_Display, index) != 0 )
            {
                return nn::vi::ResultOperationFailed();
            }

            return nn::ResultSuccess();
        }

        return nn::vi::ResultDisplayDisconnected();
    }

    nn::Result AndroidDisplay::GetRgbRange(nn::vi::RgbRangeType* pOutRange) const NN_NOEXCEPT
    {
        int range;
        if( android::SurfaceComposerClient::getRGBRange(m_Display, range) == 0 )
        {
            nn::vi::RgbRange temp;
            if( ConvertRgbRangeToVi(&temp, range) )
            {
                *pOutRange = temp;
                return nn::ResultSuccess();
            }
        }

        return nn::vi::ResultOperationFailed();
    }

    nn::Result AndroidDisplay::SetUnderscan(int underscan) NN_NOEXCEPT
    {
        if( underscan < 0 || underscan > 20 )
        {
            return nn::vi::ResultInvalidRange();
        }

        if( android::SurfaceComposerClient::setUnderscan(m_Display, underscan) == 0 )
        {
            return nn::ResultSuccess();
        }

        return nn::vi::ResultOperationFailed();
    }

    nn::Result AndroidDisplay::GetUnderscan(int* pOutUnderscan) const NN_NOEXCEPT
    {
        NN_SDK_ASSERT_NOT_NULL(pOutUnderscan);

        if( android::SurfaceComposerClient::getUnderscan(m_Display, *pOutUnderscan) == 0 )
        {
            return nn::ResultSuccess();
        }

        return nn::vi::ResultOperationFailed();
    }

    // controls fade for displays from OMM
    nn::Result AndroidDisplay::SetAlpha(float alpha) NN_NOEXCEPT
    {
        NN_SDK_ASSERT_GREATER_EQUAL(alpha, 0.f);
        NN_SDK_ASSERT_LESS_EQUAL(alpha, 1.f);

        if( m_BrightnessMax > 0 )
        {
            // cmu luma range is [-1, 1]
            // next calculations will only work for <= 0
            m_BrightnessMax = 0.f;
        }

        alpha *= 1.f + m_BrightnessMax;

        // needs to be in [-1.f,0.f]
        alpha -= 1.f;

        if( android::SurfaceComposerClient::setBrightness(m_Display, alpha) == android::OK )
        {
            return nn::ResultSuccess();
        }

        return nn::vi::ResultOperationFailed();
    }

    nn::Result AndroidDisplay::GetPowerState(nn::vi::PowerStateType* pOutState) NN_NOEXCEPT
    {
        *pOutState = m_PowerState;

        return nn::ResultSuccess();
    }

    nn::Result AndroidDisplay::SetPowerState(nn::vi::PowerState state) NN_NOEXCEPT
    {
        NN_VISRV_LOG("Setting display %d power to %d\n", m_Id, state);

        android::status_t status = android::SurfaceComposerClient::setDisplayPowerMode(m_Display, ConvertPowerState(state));

        if( status == android::OK )
        {
            // caching actual power state
            m_PowerState = state;

            return nn::ResultSuccess();
        }

        switch( m_PowerState )
        {
        case nn::vi::PowerState_Off:
            switch( state )
            {
            case nn::vi::PowerState_Off:
                return nn::vi::ResultPowerStateAlreadyRequested();
            case nn::vi::PowerState_Blank:
                return nn::vi::ResultDisplayControllerFailure();
            default:
                // Not possible to differentiate what happened with current
                // error codes from display stack.
                // This covers Off -> On transitions.
                return nn::vi::ResultOperationFailed();
            }
            break;
        case nn::vi::PowerState_Blank:
            switch( state )
            {
            case nn::vi::PowerState_Off:
                return nn::vi::ResultDisplayControllerFailure();
            case nn::vi::PowerState_On:
                return nn::vi::ResultDisplayInterfaceFailure();
            default:
                return nn::vi::ResultOperationFailed();
            }
            break;
        case nn::vi::PowerState_On:
            switch( state )
            {
            case nn::vi::PowerState_Off:
                // Failed to disable the entire pipeline.
                // The exact state of the DC is uncertain and this error
                // could imply that both DC0 and DC1 are running
                // simultaneously.
                return nn::vi::ResultDisplayControllerFailure();
            case nn::vi::PowerState_Blank:
                // Interface couldn't be disabled, but DC should still be
                // running.  There's a chance to recover on next HPD or
                // USB-C event.
                return nn::vi::ResultDisplayInterfaceFailure();
            case nn::vi::PowerState_On:
                // Connection state likely changed during On -> On sequence
                // Normally, this should be a NOP and not a fatal error
                return nn::vi::ResultPowerStateAlreadyRequested();
            default:
                return nn::vi::ResultOperationFailed();
            }
            break;
        default:
            return nn::vi::ResultOperationFailed();
        }
    }

    nn::Result AndroidDisplay::SetLayerStack(nn::vi::LayerStackType id) NN_NOEXCEPT
    {
        android::SurfaceComposerClient::openGlobalTransaction();
        android::SurfaceComposerClient::setDisplayLayerStack(m_Display, id);
        android::SurfaceComposerClient::closeGlobalTransaction();

        m_Stack = static_cast<nn::vi::LayerStack>(id);

        return nn::ResultSuccess();
    }

    nn::Result AndroidDisplay::GetLayerStack(nn::vi::LayerStackType* pOutId) const NN_NOEXCEPT
    {
        NN_SDK_ASSERT_NOT_NULL(pOutId);

        *pOutId = m_Stack;

        return nn::ResultSuccess();
    }

    // controls brightness setting from idle library
    nn::Result AndroidDisplay::SetCmuLuma(float luma) NN_NOEXCEPT
    {
        if( luma < -1.f || luma > 1.f )
        {
            return nn::vi::ResultInvalidRange();
        }

        if( android::SurfaceComposerClient::setBrightness(m_Display, luma) == android::OK )
        {
            m_BrightnessMax = luma;
            return nn::ResultSuccess();
        }

        return nn::vi::ResultOperationFailed();
    }

    nn::Result AndroidDisplay::GetCmuLuma(float* pOutLuma) const NN_NOEXCEPT
    {
        NN_SDK_ASSERT_NOT_NULL(pOutLuma);

        if( android::SurfaceComposerClient::getBrightness(m_Display, *pOutLuma) == android::OK )
        {
            return nn::ResultSuccess();
        }

        return nn::vi::ResultOperationFailed();
    }

    nn::Result AndroidDisplay::GetLogicalResolution(int* pOutWidth, int* pOutHeight) const NN_NOEXCEPT
    {
        NN_SDK_ASSERT_NOT_NULL(pOutWidth);
        NN_SDK_ASSERT_NOT_NULL(pOutHeight);

        *pOutWidth = nn::vi::LogicalResolutionWidth;
        *pOutHeight = nn::vi::LogicalResolutionHeight;

        return nn::ResultSuccess();
    }

    nn::Result AndroidDisplay::GetCmuMode(nn::vi::CmuModeType* pOutMode) const NN_NOEXCEPT
    {
        NN_SDK_ASSERT_NOT_NULL(pOutMode);

        DisplayCMUMode mode;
        float unused;
        if( GetCmuMode(m_Display, mode, unused) == android::OK )
        {
            nn::vi::CmuMode temp;
            if( ConvertCmuMode(&temp, mode) )
            {
                *pOutMode = temp;
                return nn::ResultSuccess();
            }
        }

        return nn::vi::ResultOperationFailed();
    }

    nn::Result AndroidDisplay::SetCmuMode(nn::vi::CmuMode mode) NN_NOEXCEPT
    {
        DisplayCMUMode androidMode = ConvertCmuMode(mode);
        float initialValue;

        switch( mode )
        {
        case nn::vi::CmuMode_Disabled:
            return nn::vi::ResultNotSupported();
        case nn::vi::CmuMode_Default:
        {
            nn::settings::system::TvSettings settings;
            nn::settings::system::GetTvSettings(&settings);
            initialValue = settings.tvGamma;
        }
            break;
        case nn::vi::CmuMode_InvertColor:
            initialValue = 0.f;
            break;
        case nn::vi::CmuMode_HighContrast:
        {
            nn::settings::system::TvSettings settings;
            nn::settings::system::GetTvSettings(&settings);
            initialValue = settings.contrastRatio;
        }
            break;
        case nn::vi::CmuMode_Grayscale:
            initialValue = 0.f;
            break;
        default:
            NN_UNEXPECTED_DEFAULT;
        }

        if( android::SurfaceComposerClient::setCMU(m_Display, androidMode, initialValue) == android::OK )
        {
            return nn::ResultSuccess();
        }

        return nn::vi::ResultOperationFailed();
    }

    nn::Result AndroidDisplay::GetContrastRatio(float* pOutRatio) const NN_NOEXCEPT
    {
        NN_SDK_ASSERT_NOT_NULL(pOutRatio);

        DisplayCMUMode mode;
        float ratio;
        if( GetCmuMode(m_Display, mode, ratio) == android::OK )
        {
            if( mode == DISPLAY_CMU_HighContrast )
            {
                *pOutRatio = ratio;
                return nn::ResultSuccess();
            }
            else
            {
                return nn::vi::ResultNotSupported();
            }
        }

        return nn::vi::ResultOperationFailed();
    }

    nn::Result AndroidDisplay::SetContrastRatio(float ratio) NN_NOEXCEPT
    {
        NN_SDK_ASSERT_GREATER_EQUAL(ratio, 0.f);
        NN_SDK_ASSERT_LESS_EQUAL(ratio, 1.f);

        DisplayCMUMode mode;
        float unused;
        if( GetCmuMode(m_Display, mode, unused) == android::OK )
        {
            if( mode == DISPLAY_CMU_HighContrast )
            {
                if( android::SurfaceComposerClient::setCMU(m_Display, mode, ratio) == android::OK )
                {
                    return nn::ResultSuccess();
                }
            }
            else
            {
                return nn::vi::ResultNotSupported();
            }
        }

        return nn::vi::ResultOperationFailed();
    }

    nn::Result AndroidDisplay::GetGamma(float* pOutGamma) const NN_NOEXCEPT
    {
        NN_SDK_ASSERT_NOT_NULL(pOutGamma);

        DisplayCMUMode mode;
        float gamma;
        if( GetCmuMode(m_Display, mode, gamma) == android::OK )
        {
            if( mode == DISPLAY_CMU_Normal )
            {
                *pOutGamma = gamma;
                return nn::ResultSuccess();
            }
            else
            {
                return nn::vi::ResultNotSupported();
            }
        }

        return nn::vi::ResultOperationFailed();
    }

    nn::Result AndroidDisplay::SetGamma(float gamma) NN_NOEXCEPT
    {
        NN_SDK_ASSERT_GREATER_EQUAL(gamma, 1.f);
        NN_SDK_ASSERT_LESS_EQUAL(gamma, 3.f);

        DisplayCMUMode mode;
        float unused;
        if( GetCmuMode(m_Display, mode, unused) == android::OK )
        {
            if( mode == DISPLAY_CMU_Normal )
            {
                if( android::SurfaceComposerClient::setCMU(m_Display, mode, gamma) == android::OK )
                {
                    return nn::ResultSuccess();
                }
            }
            else
            {
                return nn::vi::ResultNotSupported();
            }
        }

        return nn::vi::ResultOperationFailed();
    }

    android::status_t AndroidDisplay::GetCmuMode(const android::sp<android::IBinder>& display, DisplayCMUMode& mode, float& adjustment) NN_NOEXCEPT
    {
        int id;
        android::status_t ret = android::SurfaceComposerClient::getCMU(display, id, adjustment);
        mode = static_cast<DisplayCMUMode>(id);

        return ret;
    }

    nn::Result AndroidDisplay::SetViewport(int x, int y, int width, int height) NN_NOEXCEPT
    {
        NN_SDK_ASSERT_GREATER_EQUAL(x, 0);
        NN_SDK_ASSERT_GREATER_EQUAL(y, 0);
        NN_SDK_ASSERT_GREATER(width, 0);
        NN_SDK_ASSERT_GREATER(height, 0);

        NN_RESULT_THROW_UNLESS(x + width <= nn::vi::LogicalResolutionWidth, nn::vi::ResultInvalidDimensions());
        NN_RESULT_THROW_UNLESS(y + height <= nn::vi::LogicalResolutionHeight, nn::vi::ResultInvalidDimensions());

        nn::vi::DisplayModeInfo mode;
        std::memset(&mode, 0, sizeof(mode));

        // Note: Don't care about return value.
        //       If mode is available, display is enabled already and
        //       must preserve the display space.
        //       Otherwise, display space will be set when the display is
        //       enabled and a fake value is ok.
        GetMode(&mode);

        android::SurfaceComposerClient::openGlobalTransaction();
        android::SurfaceComposerClient::setDisplayProjection(m_Display,
                                                             android::DISPLAY_ORIENTATION_0,
                                                             android::Rect(x, y, x + width, y + height),
                                                             android::Rect(mode.width, mode.height));
        android::SurfaceComposerClient::closeGlobalTransaction();

        NN_RESULT_SUCCESS;
    }

    nn::Result AndroidDisplay::GetCompositorErrorInfo(nn::vi::CompositorError* pOutErrorInfo, int* pOutLength, int errorID) NN_NOEXCEPT
    {
        NN_SDK_ASSERT_NOT_NULL(pOutErrorInfo);
        NN_SDK_ASSERT_NOT_NULL(pOutLength);

        android::String8 str;
        m_Client.getErrorText(errorID, str); //This API does not return error...
        memcpy(pOutErrorInfo->buffer, str.string(), str.length());
        *pOutLength = str.length();

        return nn::ResultSuccess();

    }
}}}}
